/*
 * Copyright (C) 2002,2003,2004 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: world.cc,v 1.90 2004/05/22 13:04:24 dheck Exp $
 */
#include "world.hh"
#include "objects.hh"
#include "laser.hh"
#include "display.hh"
#include "player.hh"
#include "sound.hh"
#include "options.hh"
#include "game.hh"
#include "server.hh"
#include "lua.hh"
#include "client.hh"
#include "main.hh"
#include "stones_internal.hh"

#include "px/px.hh"
#include "px/dict.hh"

#include <iostream>
#include <algorithm>
#include <functional>
#include <numeric>
#include <cassert>
#include <cmath>

// remove comment from define below to switch on verbose messaging
// note: VERBOSE_MESSAGES is defined in multiple source files!
// #define VERBOSE_MESSAGES

using namespace std;
using namespace enigma::world;
using namespace px;

#include "world_internal.hh"


/* -------------------- Auxiliary functions -------------------- */

namespace
{

    /*! Enigma's stones have rounded corners; this leads to realistic
      behaviour when a marble rebounds from the corner of a single
      stone. But it's annoying when there are two adjacent stones and
      a marble rebounds from any one (or even both) corners, because
      it changes the direction of the marble quite unpredictably.
       
      This function modifies the contact information for two adjacent
      stones in such a way that the two collisions are treated as a
      single collision with a flat surface. */
    void maybe_join_contacts (StoneContact & a, StoneContact &b)
    {
//    double maxd = 4.0/32;   // Max distance between joinable collisions

        if ((a.is_contact || b.is_contact)  
            && a.is_collision && b.is_collision 
            && a.response==STONE_REBOUND && b.response==STONE_REBOUND)
            // && length(a.contact_point - b.contact_point) <= maxd)
        {
            if (a.is_contact && b.is_contact)
                b.ignore = true; // Don't rebound from `b'

            DirectionBits fa = contact_faces(a);
            DirectionBits fb = contact_faces(b);

            StoneContact &c = (a.is_contact) ? a : b;
            switch (fa & fb) {
            case NORTHBIT: c.normal = V2(0,-1); break;
            case EASTBIT: c.normal = V2(1,0); break;
            case SOUTHBIT: c.normal = V2(0,1); break;
            case WESTBIT: c.normal = V2(-1,0); break;
            case NODIRBIT:
                //fprintf(stderr, "Strange: contacts have no direction in common\n");
                break;
            default:
                //fprintf(stderr, "Strange: contacts have multiple directions in common\n");
                break;
            }
        }
    }

    /*! Find an already existing contact point in the ContactList that
      is similar to the second argument. */
    bool has_nearby_contact (const ContactList &cl, const Contact &c)
    {
        double posdelta = 0.2;
        double normaldelta = 0.1;
        for (size_t i=0; i<cl.size(); ++i) {
            if (length (cl[i].pos - c.pos) < posdelta && length (cl[i].normal - c.normal) < normaldelta)
                return true;
        }
        return false;
    }
}


/* -------------------- ActorInfo -------------------- */

ActorInfo::ActorInfo()
: pos(), vel(), forceacc(),
  charge(0), mass(1), radius(1),
  grabbed(false), ignore_contacts (false),
  last_pos(), oldpos(), force(),
  contacts(), new_contacts()
{}



/* -------------------- Messages -------------------- */

Message::Message ()
{
}
 
Message::Message (const std::string &message_,
                  const enigma::Value &value_,
                  GridPos from_)
: message (message_),
  value (value_),
  gridpos (from_)
{
}


/* -------------------- Signals -------------------- */

namespace
{

    void emit_signal (const Signal *s, int value)
    {
        Object *src = s->source;
        Object *dst = GetObject(s->destloc);
        
#if defined(VERBOSE_MESSAGES)
        src->warning("emit_from: msg='%s'", // dest=%i/%i obj=%p",
                     s->message.c_str()
//                      destloc.pos.x, destloc.pos.y,
//                      dst);
                     );
#endif
        if (GridObject *go = dynamic_cast<GridObject*>(src))
            SendMessage (dst, Message (s->message, value, go->get_pos()));
        else
            SendMessage (dst, Message (s->message, value));
    }

    void emit_from (const SignalList &sl, Object *source, int value)
    {
        size_t size = sl.size();
        for (unsigned i=0; i<size; ++i) {
            if (sl[i].source == source)
                emit_signal (&sl[i], value);
        }
//         // signals may have side effects. To minimize them
//         //   1. collect all targets and then
//         //   2. emit signals to targets

//         vector<Object*> targets;

//         for (unsigned i=0; i<size; ++i) {
//             if (m_signals[i].get_source() == source)
//                 target.push_back (m_signals[i]);

//             for (unsigned i=0; i<size; ++i)
//                 if (GetObject(m_signals[i].get_target_loc()) == targets[i])
//                     m_signals[i].emit_from (source, value);
//         }
    }

    bool emit_by_index (const SignalList &sl, Object *source, int signalidx, int value) 
    {
        size_t size      = sl.size();
        int    signalcnt = 0;
        for (unsigned i=0; i<size; ++i) {
            if (sl[i].source == source) {
                if (signalcnt == signalidx) {
                    emit_signal (&sl[i], value);
                    return true;
                }
                signalcnt += 1;
            }
        }
        return false;
    }

    Object *find_single_destination (const SignalList &sl, Object *src)
    {
        Object *found = 0;
        size_t  size  = sl.size();

        for (unsigned i = 0; i<size; ++i) {
            if (sl[i].source == src) {
                if (Object *candidate = GetObject(sl[i].destloc)) {
                    if (!found)
                        found = candidate;
                    else if (candidate != found)
                        return 0;   // multiple targets
                }
            }
        }
        return found;
    }
}


/* -------------------- RubberBand -------------------- */

RubberBand::RubberBand (Actor *a1, Actor *a2, double strength_, double length_)
: actor(a1), actor2(a2), stone(0),
  model(display::AddRubber(get_p1(),get_p2())),
  strength(strength_), length(length_)
{
    assert(actor);
    assert(length >= 0);
}

RubberBand::RubberBand (Actor *a1, Stone *st, double strength_, double length_)
: actor(a1), actor2(0), stone(st), model(0),
  strength(strength_), length(length_)
{
    assert(actor);
    assert(length >= 0);
    model = display::AddRubber(get_p1(), get_p2());
}

RubberBand::~RubberBand() {
    model.kill();
}

void RubberBand::tick(double)
{
    V2 v = get_p2()-get_p1();
    double vv = px::length(v);

    if (vv > length) {
        V2 force = v * strength*(vv-length)/vv;
        V2 f(force[0],force[1]);
        actor->add_force(f);
        if (actor2)
            actor2->add_force(-f);
    }

    model.update_first (get_p1());
    model.update_second (get_p2());
}

V2 RubberBand::get_force(Actor */*a*/)
{
    return V2();
}

V2 RubberBand::get_p1() const
{
    return V2(actor->get_pos()[0], actor->get_pos()[1]);
}

V2 RubberBand::get_p2() const
{
    if (!stone)
        return V2(actor2->get_pos()[0], actor2->get_pos()[1]);
    else
        return stone->get_pos().center();
}


/* -------------------- Field -------------------- */

Field::Field()
{
    floor=0;
    item=0;
    stone=0;
}

Field::~Field()
{
    DisposeObject(floor);
    DisposeObject(item);
    DisposeObject(stone);
}


/* -------------------- StoneContact -------------------- */

StoneContact::StoneContact(Actor *a, GridPos p,
                           const V2 &cp, const V2 &n)
: actor(a), stonepos(p),
  response(STONE_PASS),
  contact_point(cp),
  normal(n),
  is_collision(false),
  ignore (false),
  new_collision(false),
  is_contact(true)
{}

StoneContact::StoneContact()
: is_collision(false),
  ignore (false),
  new_collision(false),
  is_contact(false)
{}

DirectionBits
world::contact_faces(const StoneContact &sc)
{
    using namespace enigma;

    int dirs = NODIRBIT;

    if (sc.normal[0] < 0)
        dirs |= WESTBIT;
    else if (sc.normal[0] > 0)
        dirs |= EASTBIT;
    if (sc.normal[1] < 0)
        dirs |= NORTHBIT;
    else if (sc.normal[1] > 0)
        dirs |= SOUTHBIT;

    return DirectionBits(dirs);
}

Direction
world::contact_face(const StoneContact &sc)
{
    using namespace enigma;
    if (sc.normal == V2(-1,0))
        return WEST;
    else if (sc.normal == V2(1,0))
        return EAST;
    else if (sc.normal == V2(0,-1))
        return NORTH;
    else if (sc.normal == V2(0,1))
        return SOUTH;
    else
        return NODIR;
}


/* -------------------- Global variables -------------------- */

namespace
{
    auto_ptr<World> level;
}

util::Timer    world::GameTimer;
bool           world::TrackMessages;
Actor         *world::CurrentCollisionActor = 0;



/* -------------------- World -------------------- */

World::World(int ww, int hh) 
: fields(ww,hh),
  preparing_level(true)
{
    w = ww;
    h = hh;

    scrambleIntensity = server::GetDifficulty() == DIFFICULTY_EASY ? 3 : 10;
}

World::~World()
{
    fields = FieldArray(0,0);
    for_each(actorlist.begin(), actorlist.end(), mem_fun(&Actor::dispose));
    delete_sequence (m_rubberbands.begin(), m_rubberbands.end());
}

bool World::contains (GridPos p) {
    return (p.x>=0 && p.y>=0 && p.x<w && p.y<h);
}

bool World::is_border (GridPos p) {
    return(p.x==0 || p.y==0 || p.x==w-1 || p.y==h-1);
}

Field *World::get_field (GridPos p) {
    if (this->contains(p))
        return &fields(p.x,p.y);
    else
        return 0;
}


void World::remove (ForceField *ff)
{
    ForceList::iterator i=find (forces.begin(), forces.end(), ff);
    if (i != forces.end())
        forces.erase(i);
}

Object *World::get_named (const string &name)
{
    px::Dict<Object*>::iterator found = m_objnames.find(name);
    return (found != m_objnames.end()) ? found->second : 0;
}

void World::name_object (Object *obj, const std::string &name)
{
    m_objnames.insert(name, obj); // [name] = obj;
    obj->set_attrib("name", name);
}

void World::unname (Object *obj)
{
    assert(obj);
    string name;
    if (obj->string_attrib("name", &name)) {
        m_objnames.remove(name);
        obj->set_attrib("name", "");
    }
}

void World::add_actor (Actor *a)
{
    add_actor (a, a->get_pos());
}

void World::add_actor (Actor *a, const V2 &pos)
{
    actorlist.push_back(a);
    a->get_actorinfo()->pos = pos;
    if (!preparing_level) {
        // if game is already running, call on_creation() from here
        a->on_creation(pos);
    }
}

void World::tick (double dtime)
{
    const double timestep = 15.0/1000;
    static double timeaccu = 0.0;

    timeaccu += dtime;

    while (timeaccu >= timestep)
    {
	// Move actors
        for (unsigned i=0; i<actorlist.size(); ++i) {
            Actor     *a  = actorlist[i];
            ActorInfo *ai = a->get_actorinfo();
            ai->force = ai->forceacc;
            ai->forceacc = V2();
        }

        const int AC_STEPS = 1;
        double ac_timestep = timestep / AC_STEPS;
        for (int j=0; j<AC_STEPS; ++j) {
            for (unsigned i=0; i<actorlist.size(); ++i) {
                Actor     *a  = actorlist[i];
                tick_actor(a, ac_timestep);
            }
        }


        handle_delayed_impulses (timestep);

        // Tell floors and items about new stones.
        for (unsigned i=0; i<changed_stones.size(); ++i)
            stone_change(changed_stones[i]);
        changed_stones.clear();


        m_mouseforce.tick (timestep);
        for_each (forces.begin(), forces.end(),
                  bind2nd(mem_fun(&ForceField::tick), timestep));

        for_each (m_rubberbands.begin(), m_rubberbands.end(),
                  bind2nd(mem_fun(&RubberBand::tick), timestep));

        GameTimer.tick(timestep);

        lasers::RecalcLightNow();   // recalculate laser beams if necessary
        timeaccu -= timestep;
        timeaccu = 0;
    }
}

/* ---------- Puzzle scrambling -------------------- */

void World::add_scramble(GridPos p, Direction dir)
{
    scrambles.push_back(Scramble(p, dir, scrambleIntensity));
}

void World::scramble_puzzles()
{
    while (!scrambles.empty()) {
        list<Scramble>::iterator i = scrambles.begin();
        list<Scramble>::iterator e = scrambles.end();

        for (; i != e; ++i) {
            Stone *puzz = GetStone(i->pos);
            if (puzz && i->intensity) {
                SendMessage(puzz, "scramble", Value(double(i->dir)));
                --i->intensity;
            }
            else {
                fprintf(stderr, "no stone found for scramble at %i/%i\n", i->pos.x, i->pos.y);
                i->intensity = 0;
            }
        }

        scrambles.remove_if(mem_fun_ref(&Scramble::expired));
    }
}

/* ---------- Physics simulation ---------- */

V2 World::get_mouseforce (Actor *a)
{
    GridPos p(a->get_pos());
    if (Floor *floor = GetFloor(p))
        return floor->process_mouseforce(a, m_mouseforce.get_force(a));
    return V2();
}

/* Calculate the total acceleration on an actor A at time TIME.  The
   actor's current position X and velocity V are also passed.  [Note
   that the position and velocity entries in ActorInfo will be updated
   only after a *successful* time step, so they cannot be used
   here.] */
V2 World::get_accel (Actor *a, const V2 & x, const V2 & v, double time)
{
    V2 f;
    GridPos p((int)x[0], (int)x[1]);

    if (a->is_on_floor()) {
        if (Floor *floor = GetFloor(p)) {
            // Constant force
            f += m_flatforce.get_force(a, x, v, time);

            // Mouse force
            V2 mousef  = get_mouseforce(a);
            if (a->is_drunken()) {
                // rotate mouse force by random angle
                double maxangle = M_PI * 0.7;
                double angle = DoubleRand (-maxangle, maxangle);
                mousef = V2(mousef[0]*cos(angle) - mousef[1]*sin(angle),
                            mousef[0]*sin(angle) + mousef[1]*cos(angle));
            }
            f += mousef;

            // Friction
            double friction = floor->friction();
            if (a->has_spikes())
                friction += 7.0;

            double vv=length(v);
            if (vv > 0) {
                V2 frictionf = v * (server::FrictionFactor*friction);
                frictionf /= vv;
                frictionf *= pow(vv, 0.8);
                f -= frictionf;
            }

            f += floor->get_force(a);
        }

        if (Item *item = GetItem(p)) {
            f += item->get_force(a);
        }
    }

    // Electrostatic forces between actors.
    if (double q = get_charge(a)) {
        for (ActorList::iterator i=actorlist.begin();
             i != actorlist.end(); ++i)
        {
            Actor *a2 = *i;
            if (a2 == a) continue;
            if (double q2 = get_charge(a2)) {
                V2 distv = a->get_pos() - a2->get_pos();
                if (double dist = distv.normalize())
//                    f += server::ElectricForce * q * q2 / (dist*dist) * distv;
                    f += server::ElectricForce * q * q2 / (dist) * distv;
            }
        }
    }

    // All other force fields.
    for (ForceList::iterator i=forces.begin();
         i != forces.end(); ++i)
    {
        f += (*i)->get_force(a, x, v, time);
    }

    ActorInfo *ai = a->get_actorinfo();
    f += ai->force;

    return f / ai->mass;
}

/* This function performs one step in the numerical integration of an
   actor's equation of motion.  TIME ist the current absolute time and
   H the size of the integration step. */
void World::advance_actor (Actor *a, double time, double dtime)
{
    const double MAXVEL = 70;

    ActorInfo &ai = *a->get_actorinfo();
    V2 accel = get_accel(a, ai.pos, ai.vel, time);

    // If the actor is currently in contact with other objects, remove
    // the force components in the direction of these objects.
    for (unsigned i=0; i<ai.contacts.size(); ++i)
    {
        const V2 &normal = ai.contacts[i].normal;
        double normal_component = normal * accel;
        if (normal_component < 0) {
            accel -= normal_component * normal;
        }
    }

    ai.vel += dtime * accel;
    ai.pos += dtime * ai.vel;

    // Limit maximum velocity
    double q = length(ai.vel) / MAXVEL;
    if (q > 1)
        ai.vel /= q;
}

/* If the actor and the stone are not in contact, `contact' is filled
   with information about the closest feature on the stone and
   `is_contact' is set to false.
*/

void World::find_contact_with_stone (Actor *a, StoneContact &contact, int x, int y) {
    contact.actor      = a;
    contact.stonepos   = GridPos(x, y);
    contact.is_contact = false;

    Stone *stone = world::GetStone(contact.stonepos);
    if (!stone)
        return;

    const ActorInfo &ai = *a->get_actorinfo();
    double r = ai.radius;
    double ax = ai.pos[0];
    double ay = ai.pos[1];
    const double contact_e = 0.02;
    const double erad = 2.0/32; // edge radius

    // Closest feature == north or south face of the stone?
    if (ax>x+erad && ax<x+1-erad) {
        double dist = r+5;

        // south
        if (ay>y+1) {
            contact.contact_point = V2(ax, y+1);
            contact.normal        = V2(0,+1);
            dist                  = ay-(y+1);
        }
        // north
        else if (ay<y) {
            contact.contact_point = V2(ax, y);
            contact.normal        = V2(0,-1);
            dist                  = y-ay;
        }
	contact.is_contact = (dist-r < contact_e);
    }
    // Closest feature == west or east face of the stone?
    else if (ay>y+erad && ay<y+1-erad) {
        double dist=r+5;
        if (ax>x+1) { // east
            contact.contact_point = V2(x+1,ay);
            contact.normal        = V2(+1,0);
            dist                  = ax-(x+1);
        }
        else if (ax<x) { // west
            contact.contact_point = V2(x,ay);
            contact.normal        = V2(-1,0);
            dist                  = x-ax;
        }
	contact.is_contact = (dist-r < contact_e);
    }
    // Closest feature == any of the four corners
    else {
        int xcorner=(ax >= x+1-erad);
        int ycorner=(ay >= y+1-erad);
        double cx[2] = {erad, -erad};

        V2 corner(x+xcorner+cx[xcorner], y+ycorner+cx[ycorner]);
        V2 b=V2(ax,ay) - corner;

        contact.is_contact    = (length(b)-r-erad < contact_e);
        contact.normal        = normalize(b);
        contact.contact_point = corner + contact.normal*erad;
    }

    // treat this as a collision only if actor not inside the stone
    // and velocity towards stone
    if (ax >= x && ax < x+1 && ay >= y && ay < y+1)
        contact.is_collision = false;
    else
        contact.is_collision  = contact.normal*ai.vel < 0;

    contact.response = stone->collision_response(contact);
    contact.sound = stone->collision_sound();
}

void World::find_stone_contacts (Actor *a, StoneContactList &cl)
{
    ActorInfo &ai = *a->get_actorinfo();
    px::Rect r(round_down<int>(ai.pos[0]-0.5), 
               round_down<int>(ai.pos[1]-0.5),
	       round_down<int>(ai.pos[0]+0.5), 
               round_down<int>(ai.pos[1]+0.5));
    r.w -= r.x;
    r.h -= r.y;

    StoneContact contacts[2][2];

    // Test for collisions with the nearby stones
    for (int i=0; i<=r.w; i++)
        for (int j=0; j<=r.h; j++)
            find_contact_with_stone(a, contacts[i][j], r.x+i, r.y+j);

    maybe_join_contacts (contacts[0][0], contacts[1][0]);
    maybe_join_contacts (contacts[0][0], contacts[0][1]);
    maybe_join_contacts (contacts[1][0], contacts[1][1]);
    maybe_join_contacts (contacts[0][1], contacts[1][1]);

    for (int i=0; i<=r.w; i++)
        for (int j=0; j<=r.h; j++)
            if (contacts[i][j].is_contact)
                cl.push_back(contacts[i][j]);
}

/*! This function is called for every contact between an actor and a
  stone.  It is here that the stones are informed about contacts.
  Note that a stone may kill itself during a call to actor_hit() or
  actor_contact() so we must not refer to it after having called these
  functions.  (Incidentally, this is why StoneContact only contains a
  'stonepos' and no 'stone' entry.) */
void World::handle_stone_contact (StoneContact &sc) 
{
    Actor     *a           = sc.actor;
    ActorInfo &ai          = *a->get_actorinfo();
    double     restitution = 1.0; //0.85;

    if (server::NoCollisions)
        return;

    Contact contact (sc.contact_point, sc.normal);
    
    if (sc.is_collision) {
        if (!sc.ignore && sc.response == STONE_REBOUND) {
            ai.new_contacts.push_back (contact);
            bool handle_collision = length (ai.vel) > 0.8;
            if (handle_collision && !has_nearby_contact (ai.contacts, contact)) {
		if (Stone *stone = world::GetStone(sc.stonepos))
                {
                    CurrentCollisionActor = a;
		    stone->actor_hit(sc);
                    CurrentCollisionActor = 0;

                    client::Msg_Sparkle (sc.contact_point);
                    double volume = std::max (1.0, length(ai.vel)/2);
                    sound::SoundEvent (sc.sound.c_str(), sc.contact_point, volume);
                }
            }
            ai.next_pos = ai.oldpos; // Reset actor position
            ai.vel -= (1 + restitution)*(ai.vel*sc.normal)*sc.normal;
        }
    }
    else if (sc.is_contact) {
//         if (sc.response == STONE_REBOUND)
//             ai.new_contacts.push_back (contact);
        if (Stone *stone = world::GetStone(sc.stonepos))
            stone->actor_contact(sc.actor);
    }
}

void World::handle_contacts (Actor *actor1) 
{
    ActorInfo &a1 = *actor1->get_actorinfo();

    a1.next_pos = a1.pos;

    if (a1.ignore_contacts)
        return;

    // List of current contacts is updated in this function
    a1.new_contacts.clear();
    a1.contact_normals.clear();

    // Handle contacts with stones
    StoneContactList cl;
    find_stone_contacts(actor1, cl);
    for (StoneContactList::iterator i=cl.begin(); i != cl.end(); ++i)
    {
        handle_stone_contact (*i);
    }

    // Handle contacts with other actors
    V2     p1 = a1.pos;
    double m1 = a1.mass;

    unsigned s=actorlist.size();
    for (unsigned j=0; j<s; ++j)
    {
	Actor     *actor2 = actorlist[j];
        ActorInfo &a2     = *actor2->get_actorinfo();

        if (actor2 == actor1 || a2.grabbed)
	    continue;

        V2     p2      = a2.pos;
        double overlap = a1.radius + a2.radius - length(p1-p2);
        if (overlap > 0)
        {
            V2 n  = normalize(p1-p2);	// normal to contact surface
            V2 v1 = n * (a1.vel*n);
            V2 v2 = n * (a2.vel*n);



            bool collisionp = (a1.vel-a2.vel)*n < 0;

            if (!collisionp)
                continue;

            Contact contact (p2+n*a2.radius, n);
            a1.new_contacts.push_back (contact);


            a1.pos = a1.oldpos;		// Reset actor position

            actor1->on_collision (actor2);
            actor2->on_collision (actor1);

            bool reboundp = (actor1->is_movable() && actor2->is_movable() &&
                             (actor1->is_on_floor() == actor2->is_on_floor()));
            if (reboundp) {
                    double m2 = a2.mass;
                    double M = m1+m2;

                    double restitution = 1.0; //0.95;

                    a1.vel += -v1 + (v2*(2*m2) + v1*(m1-m2)) / M;
                    a2.vel += -v2 + (v1*(2*m1) + v2*(m2-m1)) / M;

                    a1.vel *= restitution;
                    a2.vel *= restitution;

                if (!has_nearby_contact (a1.contacts, contact)) {
                    double volume = length (v1) +length(v2);
                    if (volume > 0.4)
                        sound::SoundEvent ("ballcollision", contact.pos, volume);
                }
            }
        }
    }

    a1.pos = a1.next_pos;
    swap (a1.contacts, a1.new_contacts);
}

void World::tick_actor(Actor *a, double dtime) 
{
    ActorInfo &ai = * a->get_actorinfo();

    ai.last_pos = ai.pos;

    double rest_time = dtime;
    double dt = 0.0025; // Size of one time step -- do not change!

    if (!a->can_move()) {
        if (length(get_mouseforce(a)) > 2.5)
            client::Msg_Sparkle (ai.pos);
        ai.vel = V2();
        a->think (dtime);
        a->move ();         // 'move' nevertheless, to pick up items etc
        return;
    }

    while (rest_time > 0 && !ai.grabbed && !a->is_dead())
    {
        advance_actor(a, dtime-rest_time, dt);
        handle_contacts (a);
        a->move ();
        rest_time -= dt;
        a->think(dt);
    }
}

void World::handle_delayed_impulses (double dtime)
{
    // Handle delayed impulses
    ImpulseList::iterator i = delayed_impulses.begin(),
        end = delayed_impulses.end();
    while (i != end) {
        // shall the impulse take effect now ?
        if (i->tick(dtime)) {
            if (Stone *st = GetStone(i->destination()))
                i->send_impulse(st);
            i = delayed_impulses.erase(i);
        }
        else
            ++i;
    }
}

void World::stone_change(GridPos p) 
{
    Stone *st = GetStone(p);
    if (st)
        st->on_floor_change();

    if (Item *it = GetItem(p))
        it->stone_change(st);

    if (Floor *fl = GetFloor(p))
        fl->stone_change(st);

    lasers::MaybeRecalcLight(p);
}




/* -------------------- Functions -------------------- */

void world::Resize (int w, int h)
{
    level.reset (new World(w,h));
    display::NewWorld(w, h);
}

void world::PrepareLevel (int w, int h)
{
    GameTimer.clear();
    CurrentCollisionActor = 0;
    server::GameReset();
    player::NewWorld();
    Resize (w, h);
}

bool world::InitWorld()
{
    level->scramble_puzzles();

    lasers::RecalcLight();
    lasers::RecalcLightNow();    // recalculate laser beams if necessary

    bool seen_player0 = false;

    for (ActorList::iterator i=level->actorlist.begin();
         i != level->actorlist.end(); ++i)
    {
        Actor *a = *i;
        a->on_creation(a->get_actorinfo()->pos);
        a->message ("init", Value());

        int iplayer;
        if (a->int_attrib("player", &iplayer)) {
            player::AddActor(iplayer,a);
            if (iplayer == 0) seen_player0 = true;
        }
    }

    level->changed_stones.clear();

    if (!seen_player0) {
        fprintf(stderr, "Error: No player 0 defined!\n");
        return false;
    }

    world::BroadcastMessage("init", Value(),
                            GridLayerBits(GRID_ITEMS_BIT | GRID_STONES_BIT));

    server::InitMoveCounter();
    STATUSBAR->show_move_counter (server::ShowMoves);

//    player::Tick(0.0);          // HACK: center main actor
    display::FocusReferencePoint();


    level->preparing_level = false;

    return true;
}

void world::SetMouseForce(V2 f)
{
    level->m_mouseforce.add_force(f);
}

void world::NameObject(Object *obj, const std::string &name)
{
    string old_name;
    if (obj->string_attrib("name", &old_name)) {
        obj->warning("name '%s' overwritten by '%s'",
                     old_name.c_str(), name.c_str());
        UnnameObject(obj);
    }
    level->name_object (obj, name);
}

void world::UnnameObject(Object *obj)
{
    level->unname(obj);
}

void world::TransferObjectName (Object *source, Object *target)
{
    string name;
    if (source->string_attrib("name", &name)) {
        UnnameObject(source);
        string targetName;
        if (target->string_attrib("name", &targetName)) {
            target->warning("name '%s' overwritten by '%s'",
                            targetName.c_str(), name.c_str());
            UnnameObject(target);
        }
        NameObject(target, name);
    }
}

Object * world::GetNamedObject (const std::string &name)
{
    return level->get_named (name);
}

bool world::IsLevelBorder(GridPos p)
{
    return level->is_border(p);
}

bool world::IsInsideLevel(GridPos p)
{
    return level->contains(p);
}


/* -------------------- Force fields -------------------- */

void world::AddForceField(ForceField *ff)
{
    level->forces.push_back(ff);
}

void world::RemoveForceField(ForceField *ff) {
    level->remove (ff);
}

void world::SetConstantForce (V2 force) {
    level->m_flatforce.set_force(force);
}



/* -------------------- Rubber bands -------------------- */

void world::AddRubberBand (Actor *a, Stone *st, double strength,double length)
{
    level->m_rubberbands.push_back(new RubberBand (a, st, strength, length));
}

void world::AddRubberBand (Actor *a, Actor *a2, double strength,double length)
{
    double l = px::Max (length, get_radius(a) + get_radius(a2));
    level->m_rubberbands.push_back(new RubberBand (a, a2, strength, l));
}

void world::KillRubberBand (Actor *a, Stone *st)
{
    assert(a);
    for (unsigned i=0; i<level->m_rubberbands.size(); ) {
        RubberBand &r = *level->m_rubberbands[i];
        if (r.get_actor() == a && r.get_stone() != 0)
            if (r.get_stone()==st || st==0) {
                delete &r;
                level->m_rubberbands.erase(level->m_rubberbands.begin()+i);
                continue;       // don't increment i
            }
        ++i;
    }
}

void world::KillRubberBand (Actor *a, Actor *a2)
{
    assert (a);
    for (unsigned i=0; i<level->m_rubberbands.size(); ) {
        RubberBand &r = *level->m_rubberbands[i];
        if (r.get_actor() == a && r.get_actor2() != 0)
            if (r.get_actor2()==a2 || a2==0) {
                delete &r;
                level->m_rubberbands.erase(level->m_rubberbands.begin()+i);
                continue;       // don't increment i
            }
        ++i;
    }
}

void world::KillRubberBands (Stone *st)
{
   for (unsigned i=0; i<level->m_rubberbands.size(); ) {
        RubberBand &r = *level->m_rubberbands[i];
        if (r.get_stone() != 0 && r.get_stone()==st) {
            delete &r;
            level->m_rubberbands.erase(level->m_rubberbands.begin()+i);
            continue;       // don't increment i
        }
        ++i;
    }
}

void world::GiveRubberBands (Stone *st, vector<Rubber_Band_Info> &rubs) {
   for (unsigned i=0; i<level->m_rubberbands.size(); ) {
        RubberBand &r = *level->m_rubberbands[i];
        if (r.get_stone() == st) {
	    Rubber_Band_Info rbi;
	    rbi.act = r.get_actor();
	    rbi.strength = r.get_strength();
	    rbi.length = r.get_length();
	    rubs.push_back(rbi);
	}
        ++i;
    }
}

bool world::HasRubberBand (Actor *a, Stone *st)
{
    for (unsigned i=0; i<level->m_rubberbands.size(); ++i) {
        RubberBand &r = *level->m_rubberbands[i];
        if (r.get_actor() == a && r.get_stone() == st)
            return true;
    }
    return false;
}

 
/* -------------------- Signals -------------------- */

void world::AddSignal (const GridLoc &srcloc, 
                       const GridLoc &dstloc, 
                       const string &msg)
{
#if defined(VERBOSE_MESSAGES)
    fprintf(stderr, "AddSignal src=%i/%i dest=%i/%i msg='%s'\n",
            srcloc.pos.x, srcloc.pos.y, dstloc.pos.x, dstloc.pos.y, msg.c_str());
#endif // VERBOSE_MESSAGES

    if (Object *src = GetObject(srcloc)) {
        src->set_attrib("action", "signal");
        level->m_signals.push_back (Signal (src, dstloc, msg));
    }
    else {
        Log << "AddSignal: Invalid signal source\n";
    }
}

bool world::HaveSignals (Object *src) 
{
    SignalList::const_iterator i=level->m_signals.begin(),
        end = level->m_signals.end();
    for (; i != end; ++i) 
        if (i->source == src) 
            return true;
    return false;
}


bool world::EmitSignalByIndex (Object *src, int signalidx, int value) 
{
    return emit_by_index (level->m_signals, src, signalidx, value);
}

bool world::GetSignalTargetPos (Object *src, GridPos &pos, int signalidx) 
{
    SignalList::const_iterator i = level->m_signals.begin(),
        end = level->m_signals.end();
    int idx = 0;
    for (; i != end; ++i) {
        if (i->source == src) {
            if (idx == signalidx) {
                pos = i->destloc.pos;
                return true;
            }
            idx += 1;
        }
    }
    return false;
}


void world::SendMessage(Object *o, const std::string &msg) 
{
    SendMessage (o, Message (msg, Value()));
}

void world::SendMessage(Object *o, const std::string &msg, const Value& value)
{
    SendMessage (o, Message (msg, value));
}

void world::SendMessage (Object *o, const Message &m)
{
    if (o) {
        if (TrackMessages)
            o->warning("will be sent message '%s' (with Value)", m.message.c_str());
        o->on_message (m);
    }
    else if (TrackMessages)
        fprintf(stderr, "Sending message '%s' to NULL-object\n", m.message.c_str());
}


void world::BroadcastMessage (const std::string& msg, 
                              const Value& value, 
                              GridLayerBits grids)
{
    int  width     = level->w;
    int  height    = level->h;
    bool to_floors = grids & GRID_FLOOR_BIT;
    bool to_items  = grids & GRID_ITEMS_BIT;
    bool to_stones = grids & GRID_STONES_BIT;

    for (int y = 0; y<height; ++y) {
        for (int x = 0; x<width; ++x) {
            GridPos p(x, y);
            Field *f = level->get_field(p);
            if (to_floors && f->floor) SendMessage (f->floor, msg, value);
            if (to_items && f->item)  SendMessage (f->item,  msg, value);
            if (to_stones && f->stone) SendMessage (f->stone, msg, value);
        }
    }
}


void world::PerformAction (Object *o, bool onoff) 
{
    string action = "idle";
    string target;

    o->string_attrib("action", &action);
    o->string_attrib("target", &target);

#if defined(VERBOSE_MESSAGES)
    o->warning("PerformAction action=%s target=%s", action.c_str(), target.c_str());
#endif // VERBOSE_MESSAGES

    if (action == "callback") {
        lua::CallFunc(lua::LevelState(), target.c_str(), Value(onoff));
    }
    else if (action == "signal") {
        emit_from (level->m_signals, o, onoff);
    }
    else if (Object *t = GetNamedObject(target)) {
        if (GridObject *go = dynamic_cast<GridObject*>(o))
            SendMessage (t, Message (action, Value(onoff), go->get_pos()));
        else
            SendMessage (t, Message (action, Value(onoff)));
    }
    else if (action != "idle") {
        fprintf (stderr, "Unknown target '%s' for action '%s'\n",
                 target.c_str(), action.c_str());
    }
}



// explosion messages:
// -------------------
//
// "ignite"    : (sent by dynamite)
// "expl"      : (sent by bomb)
// "bombstone" : (sent by bombstone)

namespace
{
    void explosion (GridPos p, ItemID expl)
    {
        Item  *item  = GetItem(p);
        Stone *stone = GetStone(p);
        SendMessage(stone, "expl");
        if (item) {
            if (has_flags(item, itf_indestructible))
                SendMessage(item, "expl");
            else
                SetItem(p, expl);
        }
        else
            SetItem(p, MakeItem (expl));
    }
}

void world::SendExplosionEffect(GridPos center, ExplosionType type) 
{
    const int AFFECTED_FIELDS       = 8;

    for (int a = 0; a<AFFECTED_FIELDS; ++a) {
	GridPos  dest = get_neighbour (center, a+1);
	Item    *item            = GetItem(dest);
	Stone   *stone           = GetStone(dest);
	bool     direct_neighbor = a<4;

	switch (type) {
	case EXPLOSION_DYNAMITE:
	    if (stone) SendMessage(stone, "ignite");
	    if (item) SendMessage(item, "ignite");
	    break;

	case EXPLOSION_BLACKBOMB:
	    if (direct_neighbor) {
                explosion (dest, it_explosion1);
            } 
            else {
                SendMessage(stone, "ignite");
                SendMessage(item, "ignite");
            }
            break;

	case EXPLOSION_WHITEBOMB:
            explosion (dest, it_explosion3);
	    break;

	case EXPLOSION_BOMBSTONE:
	    if (direct_neighbor) {
		if (stone) SendMessage(stone, "bombstone");
		if (item) SendMessage(item, "bombstone");
	    }
	    break;

        case EXPLOSION_SPITTER:

            break;
	}
    }
}

Object *world::GetObject (const GridLoc &l)
{
    switch (l.layer) {
    case GRID_FLOOR:  return GetFloor(l.pos);
    case GRID_ITEMS:  return GetItem(l.pos);
    case GRID_STONES: return GetStone(l.pos);
    default: return 0;
    }
}


/* -------------------- Floor manipulation -------------------- */

void world::KillFloor(GridPos p) {
    level->fl_layer.kill(p);
}

Floor *world::GetFloor(GridPos p) {
    return level->fl_layer.get(p);
}

void world::SetFloor(GridPos p, Floor* fl) {
    level->fl_layer.set(p,fl);
    if (!level->preparing_level)
        if (Stone *st = GetStone(p))
            st->on_floor_change();
}


/* -------------------- Stone manipulation -------------------- */

Stone * world::GetStone(GridPos p) {
    return level->st_layer.get(p);
}

void world::KillStone(GridPos p) {
    level->st_layer.kill(p);
    level->changed_stones.push_back(p);
}

Stone * world::YieldStone(GridPos p) {
    Stone *st = level->st_layer.yield(p);
    level->changed_stones.push_back(p);
    return st;
}

void world::SetStone(GridPos p, Stone* st) {
    level->st_layer.set(p,st);
    level->changed_stones.push_back(p);
}

void world::ReplaceStone (GridPos p, Stone* st) {
    Stone *old = level->st_layer.get(p);
    if (old) {
        TransferObjectName(old, st);
        level->st_layer.kill(p);
    }
    SetStone(p, st);
}

void world::MoveStone (GridPos oldPos, GridPos newPos) {
    SetStone(newPos, YieldStone(oldPos));
}

void world::SetScrambleIntensity (int intensity) {
    level->scrambleIntensity = intensity;
}

void world::AddScramble(GridPos p, Direction d) {
    level->add_scramble(p, d);
}


/* -------------------- Item manipulation -------------------- */

void world::KillItem(GridPos p) 
{
    lasers::MaybeRecalcLight(p);
    level->it_layer.kill(p);
}

Item *world::GetItem(GridPos p) {
    return level->it_layer.get(p);
}

Item *world::YieldItem(GridPos p) {
    lasers::MaybeRecalcLight(p);
    return level->it_layer.yield(p);
}

void world::SetItem (GridPos p, Item* it) 
{
    lasers::MaybeRecalcLight(p);
    level->it_layer.set(p,it);
}

void world::SetItem (GridPos p, ItemID id) 
{
    SetItem (p, MakeItem (id));
}


/* -------------------- Actor manipulation -------------------- */

void world::AddActor(double x, double y, Actor* a) 
{
    level->add_actor (a, V2(x, y));
}

void world::AddActor (Actor *a)
{
    level->add_actor (a);
}

Actor * world::YieldActor(Actor *a) 
{
    ActorList::iterator i=find(level->actorlist.begin(), level->actorlist.end(), a);
    if (i != level->actorlist.end()) {
        level->actorlist.erase(i);
        GrabActor(a);
        return a;
    }
    return 0;
}

void world::KillActor (Actor *a) {
    delete YieldActor (a);
}

void world::WarpActor(Actor *a, double newx, double newy, bool keep_velocity)
{
    V2 newpos = V2(newx, newy);
    if (!keep_velocity)
        a->get_actorinfo()->vel = V2();
    a->warp(newpos);
}

void world::FastRespawnActor(Actor *a, bool keep_velocity) {
    a->find_respawnpos();
    const V2& p = a->get_respawnpos();
    WarpActor(a, p[0], p[1], keep_velocity);
}


void world::RespawnActor(Actor *a) {
    a->find_respawnpos();
    a->respawn();
}

Actor *FindActorByID (ActorID id)
{
    ActorList::iterator i = level->actorlist.begin(),
        end = level->actorlist.end();
    for (; i != end; ++i) {
        Actor *a = *i;
        if (get_id(a) == id)
            return a;
    }
    return 0;
}

unsigned world::CountActorsOfKind (ActorID id) 
{
    unsigned count = 0;
    ActorList::iterator i = level->actorlist.begin(),
        end = level->actorlist.end();
    for (; i != end; ++i) {
        Actor *a = *i;
        if (get_id(a) == id)
            ++count;
    }
    return count;
}

Actor *world::FindOtherMarble(Actor *thisMarble) 
{
    if (!thisMarble) 
        return 0;

    switch (get_id(thisMarble)) {
    case ac_blackball: return FindActorByID (ac_whiteball);
    case ac_whiteball: return FindActorByID (ac_blackball);
    default:
        return 0;
    }
}

bool world::ExchangeMarbles(Actor *marble1) {
    Actor *marble2 = FindOtherMarble(marble1);
    if (marble2) {
        ActorInfo *info1 = marble1->get_actorinfo();
        ActorInfo *info2 = marble2->get_actorinfo();

        swap(info1->pos, info2->pos);
        swap(info1->oldpos, info2->oldpos);

        return true;
    }
    return false;
}


void world::GrabActor(Actor *a)
{
    a->get_actorinfo()->grabbed = true;
}

void world::ReleaseActor(Actor *a)
{
    a->get_actorinfo()->grabbed = false;
}

bool world::GetActorsInRange (px::V2 center, double range,
                              vector<Actor*>& actors)
{
    ActorList &al = level->actorlist;
    for (ActorList::iterator i=al.begin(); i!=al.end(); ++i) {
        Actor *a = *i;
        if (length(a->get_pos() - center) < range)
            actors.push_back(a);
    }
    return !actors.empty();
}

bool world::GetActorsInsideField (const GridPos& pos, vector<Actor*>& actors)
{
    ActorList &al = level->actorlist;
    for (ActorList::iterator i=al.begin(); i!=al.end(); ++i) {
        Actor *a = *i;
        if (GridPos(a->get_pos()) == pos)
            actors.push_back(a);
    }
    return !actors.empty();
}

void world::ShatterActorsInsideField (const GridPos &p)
{
    vector<Actor *> actors;
    GetActorsInsideField (p, actors);
    vector<Actor *>::iterator i=actors.begin(),
        end = actors.end();
    for (; i != end; ++i) 
        SendMessage(*i, "shatter");
}


/* -------------------- Functions -------------------- */

void world::addDelayedImpulse (const Impulse& impulse, double delay, 
                               const Stone *estimated_receiver) 
{
    // @@@ FIXME: is a special handling necessary if several impulses hit same destination ?

    level->delayed_impulses.push_back(DelayedImpulse(impulse, delay, estimated_receiver));
}

void world::Tick(double dtime) {
    level->tick (dtime);
}

void world::Init()
{
    InitActors();
    lasers::Init();
    InitItems();
    stones::Init();
    InitFloors();
}

void world::Shutdown()
{
    level.reset();
    Repos_Shutdown();
}


/* -------------------- Object repository -------------------- */
namespace
{
    class ObjectRepos : public px::Nocopy {
    public:
        ObjectRepos();
        ~ObjectRepos();
        void add_templ(Object *o);
        void add_templ (const string &name, Object *o);
        bool has_templ(const string &name);
        Object *make(const string &name);
        Object *get_template(const string &name);

        void dump_info();
    private:
        typedef px::Dict<Object*> ObjectMap;
        ObjectMap objmap;           // repository of object templates
        int stonecount, floorcount, itemcount;
    };
}

ObjectRepos::ObjectRepos() {
    stonecount = floorcount = itemcount = 0;
}

ObjectRepos::~ObjectRepos()
{
    px::delete_map(objmap.begin(), objmap.end());
}


void ObjectRepos::add_templ (const string &kind, Object *o)
{
    if (has_templ(kind))
        enigma::Log << "add_templ: redefinition of object `" <<kind<< "'.\n";
    else
        objmap.insert(kind, o);
}

void ObjectRepos::add_templ (Object *o) {
    string kind = o->get_kind();
    if (has_templ(kind))
        enigma::Log << "add_templ: redefinition of object `" <<kind<< "'.\n";
    else
        objmap.insert(kind, o);
}

bool ObjectRepos::has_templ(const string &name) {
    return objmap.find(name) != objmap.end();
}

Object * ObjectRepos::make(const string &name) {
    ObjectMap::iterator i=objmap.find(name);
    if (i==objmap.end())
        return 0;
    else
        return i->second->clone();
}

Object * ObjectRepos::get_template(const string &name) {
    if (objmap.has_key(name))
        return objmap[name];
    else
        return 0;
}

/* Generate a list of all available objects and their attributes. */
void ObjectRepos::dump_info() {
    ObjectMap::iterator iter = objmap.begin();
    for (; iter != objmap.end(); ++iter) {
        cout << iter->first << "( ";
        Object *obj = iter->second;
        const Object::AttribMap &a = obj->get_attribs();
        for (Object::AttribMap::const_iterator j=a.begin();
             j!=a.end(); ++j)
        {
            cout << j->first << " ";
        }
        cout << ")\n";
    }
}


namespace
{
    ObjectRepos *repos;
    vector<Actor *> actor_repos(ac_COUNT);
    vector<Stone *> stone_repos(st_COUNT);
}

void world::Register (const string &kind, Object *obj) {
    if (!repos)
        repos = new ObjectRepos;
    if (kind.empty())
        repos->add_templ(obj->get_kind(), obj);
    else
        repos->add_templ(kind, obj);
}


void world::Register (Object *obj) {
    Register (obj->get_kind(), obj);
}

void world::Register (const string &kind, Floor *obj)
{
    Object *o = obj;
    Register(kind, o);
}

void world::Register (const string &kind, Stone *obj) {
    Object *o = obj;
    Register(kind, o);
}

void world::RegisterStone (Stone *stone) 
{
    Register(static_cast<Object*>(stone));
    StoneID id = get_id(stone);
    assert (id != st_INVALID);
    stone_repos[id] = stone;
}

void world::RegisterActor (Actor *actor) 
{
    Register(static_cast<Object*>(actor));
    ActorID id = get_id(actor);
    assert (id != ac_INVALID);
    actor_repos[id] = actor;
}

void world::Repos_Shutdown() {
    delete repos;
}

Object * world::MakeObject(const char *kind) {
    static Object *last_templ = 0;
    static string last_kind;

    if (last_kind != kind) {
        last_kind = kind;
        last_templ = repos->get_template(kind);
    }

    Object *o = 0;
    if (last_templ)
        o=last_templ->clone();
    if (!o)
        fprintf(stderr, "MakeObject: unkown object name `%s'\n",kind);
    return o;
}

Object * world::GetObjectTemplate(const std::string &kind) {
    if (!repos->has_templ(kind)) {
        cerr << "GetObjectTemplate: unkown object name `" <<kind<< "'.\n";
        return 0;
    } else
        return repos->get_template(kind);
}

Floor* world::MakeFloor(const char *kind) {
    return dynamic_cast<Floor*>(MakeObject(kind));
}

Stone * world::MakeStone (const char *kind) {
    return dynamic_cast<Stone*>(MakeObject(kind));
}

Actor * world::MakeActor (const char *kind) {
    return dynamic_cast<Actor*>(MakeObject(kind));
}

Actor *world::MakeActor (ActorID id) 
{
    if (Actor *ac = actor_repos[id])
        return ac->clone();
    else
        assert (0 || "internal error: built-in actor not defined!?");
    return 0;
}

Stone *world::MakeStone (StoneID id) 
{
    if (Stone *st = stone_repos[id])
        return st->clone();
    else
        assert (0 || "internal error: built-in stone not defined!?");
    return 0;
}


void world::DisposeObject(Object *o) {
    if (o != 0) {
        UnnameObject(o);
        o->dispose();
    }
}

void world::DefineSimpleFloor(const std::string &kind,
                              double friction, double mousefactor)
{
    Register(new Floor(kind.c_str(), friction, mousefactor));
}

void world::DumpObjectInfo() {
    repos->dump_info();
}


namespace
{
    vector<Item *> item_repos(it_COUNT);
}

void world::Register (const string &kind, Item *obj) 
{
    Object *o = obj;
    world::Register(kind, o);
}

void world::RegisterItem (Item *item) 
{
    Register(static_cast<Object*>(item));
    ItemID id = get_id(item);
    assert (id != it_INVALID);
    item_repos[id] = item;
}

Item *world::MakeItem (ItemID id) 
{
    if (Item *it = item_repos[id])
        return it->clone();
    else
        assert (0 || "internal error: built-in item not defined!?");
    return 0;
}

Item * world::MakeItem(const char *kind) {
    return dynamic_cast<Item*>(MakeObject(kind));
}

