# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

"""funcall.py - Use this to implement array functionality with function calls

For instance, if we evaluate 'A+B' where A and B are arrays, the result should
be an array containing the pairwise sum of A and B.

"""

import objects, lazyarray

def needs_convert_list(func, arglist):
	"""Return (size, boolean list) indicating what to upconvert
	
	For instance, if the result is (3, (True, False)), that means that the first
	argument needs to be treated like an array of length 3, but the second
	doesn't.  If the result is (None, None), no need to array-ize anything.
	
	"""
	normal_call, size = True, None
	for isscalar, arg in zip(func.scalar_list, arglist):
		if isscalar and not arg.isscalar():
			normal_call = False
			if size is None: size = arg.len()
			else: assert size == arg.len(), arglist

	if normal_call: return None, None
	else: return size, func.scalar_list # all the scalars need to be converted

def call(func, arglist):
	"""Call function on arglist and return the result
	
	func must be a ExFunc object. Arglist should be a python list/tuple of
	ExObjects.
	
	"""
	def helper(f, args):
		"""Return a thunk which applies function f to the given args
		
		If the first element in each arg is True, that means the argument in the
		second spot is given as a thunk and needs to be executed.
		
		"""
		def subhelper():
			return f(*[arg() if eval else arg for eval, arg in args])
		return subhelper

	size, nc_list = needs_convert_list(func, arglist)
	if size is None: return func.obj(*arglist)

	result = lazyarray.LazyArray(size)
	for i in range(size):
		arglist_instance = []
		for nc, arg in zip(nc_list, arglist):
			if nc and not arg.isscalar():
				arglist_instance.append((True, arg.obj.get_thunk(i)))
			else: arglist_instance.append((False, arg))
		result.set_value(i, helper(func.obj, arglist_instance))
	return objects.ExArray(result)
