#!/usr/bin/env python

"""PyQt4 port of the layouts/basiclayout example from Qt v4.x"""

import sys, types
from PyQt4 import QtCore, QtGui
import celltree, stylesheets

class TableSection(QtGui.QWidget):
    """A one dimensional table of simple cells"""
    coord_dict = {0: 'name', 1: 'func', 2: 'value'}
    comp_dict = {'name': 0, 'func': 1, 'value': 2}
    def __init__(self, orientation, length, begin = True, end = True):
        """Initialize with 'wide' or 'tall' and the number of elements

        Optionally, specify begin and/or end as False to indicate that
        this table section is not the beginning/end of the overall
        table.
        """
        QtGui.QWidget.__init__(self)
        self.setSize(orientation, length)
        self.begin, self.end = begin, end

        self.set_widget_dict()
        self.add_widget_styles()
        self.add_widgets_to_layout()
        self.setLayout(self.layout)        

    def setSize(self, orientation, length):
        """Set the orientation and size attributes"""
        if orientation == 'wide': self.rows, self.cols = 3, length
        elif orientation == 'tall': self.rows, self.cols = length, 3
        else: assert False, ('Invalid orientation '+orientation)
        self.orientation, self.length = orientation, length

    def get_widget(self, comptype, pos):
        """Return the widget at position pos and func/name/value"""
        if self.orientation == 'tall':
            return widget_dict[(pos, comp_dict(comptype))]
        else: return widget_dict[(self.coord_dict(comptype), pos)]

    def get_pos(self, row, col):
        """Return (comptype, postinio) from row and col coordinates"""
        if self.orientation == 'tall': return (self.coord_dict[col], row)
        else: return (self.coord_dict[row], col)

    def set_widget_dict(self):
        """Sets initial dictionary mapping (row, col) coords to widgets"""
        self.widget_dict = {}
        for i in range(self.rows):
            for j in range(self.cols):
                comptype = self.get_pos(i,j)[0]
                if comptype == 'value':
                    w = QtGui.QLabel()
                    w.setMinimumWidth(40)
                else: w = QtGui.QLineEdit()
                self.widget_dict[(i,j)] = w

    def add_widget_styles(self):
        """Process widget dictionary and add style sheets"""
        desc_sets = {}
        def add_to_desc_sets(i, j, desc):
            """Add description to desc_set at position i, j in desc_sets"""
            try: desc_set = desc_sets[(i,j)]
            except KeyError: desc_sets[(i,j)] = set((desc,))
            else: desc_set.add(desc)

        # Add top/bottom/left/right descriptions along major axis
        if self.orientation == 'tall':
            for i in range(self.rows):
                add_to_desc_sets(i, 0, 'left')
                add_to_desc_sets(i, self.cols-1, 'right')
        else:
            for j in range(self.cols):
                add_to_desc_sets(0, j, 'top')
                add_to_desc_sets(self.rows-1, j, 'bottom')

        # Add descriptions if beginning/end of table
        if self.orientation == 'tall':
            for j in range(self.cols):
                if self.begin: add_to_desc_sets(0, j, 'top')
                if self.end: add_to_desc_sets(self.rows-1, j, 'bottom')
        else:
            for i in range(self.rows):
                if self.begin: add_to_desc_sets(i, 0, 'left')
                if self.end: add_to_desc_sets(i, self.cols-1, 'right')

        # Finally make the style sheets and apply them
        for i in range(self.rows):
            for j in range(self.cols):
                if (i,j) in desc_sets: desc_set = desc_sets[i,j]
                else: desc_set = set()
                sheet = stylesheets.get_style(self.get_pos(i,j)[0],
                                              self.orientation, desc_set)
                self.widget_dict[(i,j)].setStyleSheet(sheet)

    def add_widgets_to_layout(self):
        """Create layout and add widgets in widget dictionary to it"""
        self.layout = QtGui.QGridLayout()
        self.layout.setSpacing(0)

        for i in range(self.rows):
            for j in range(self.cols):
                self.layout.addWidget(self.widget_dict[(i,j)], i, j)

    def setText(self, triple_list):
        """Set the text in the widgets

        Format of argument is a list of (name, func, value) strings
        """
        assert False, "not implemented yet"


class NameFunc(QtGui.QWidget):
    """Hold an isolated name and function for when value is array"""
    def __init__(self, orientation):
        """Initialize with 'wide' or 'tall' as the orientation

        This widget is actually wider when you specify 'tall', but the
        idea is the wide widget will match 'wide' tables and
        vice-versa.
        """
        QtGui.QWidget.__init__(self)
        self.orientation = orientation
        self.makeLayout()
        self.set_namefuncwidgets()

    def makeLayout(self):
        """Return the appropriate layout for the orientation"""
        if self.orientation == 'wide': self.layout = QtGui.QVBoxLayout()
        elif self.orientation == 'tall': self.layout = QtGui.QHBoxLayout()
        else: assert False, "Bad orientation "+self.orientation
        self.layout.setSpacing(0)
        self.setLayout(self.layout)

    def set_namefuncwidgets(self):
        """Make the name and function widgets and set them"""
        self.name_widget = QtGui.QLineEdit()
        self.func_widget = QtGui.QLineEdit()

        name_style = stylesheets.get_style('name', self.orientation, set())
        func_style = stylesheets.get_style('func', self.orientation, set())
        self.name_widget.setStyleSheet(name_style)
        self.func_widget.setStyleSheet(func_style)

        self.layout.addWidget(self.name_widget)
        self.layout.addWidget(self.func_widget)

    def setText(self, name_text, func_text):
        """Set the name and function text"""
        assert False, "not implemented yet"

