# Copyright 2009 Ben Escoto
#
# This file is part of Explicansubstr.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

import types, csv, cStringIO
import objects
from parsing import parse


class ARError(Exception):
	"""An AbsoluteReference exception"""
	pass

class AbsoluteReference:
	"""This holds a cell reference in absolute terms

	An absolute reference is built around a tuple of tuples of integers. For
	instance, () is root, (0,) is the first element under root, etc. This is
	true of how the reference is stored internally (in self.t) as well as the
	string representation.
	
	But if the reference is to a location inside a table, the last element
	in self.t will be a tuple itself.  Then the possibilities are:
		('C',) for the column name/formula
		('R',) for the row name/formula
		(0, n) for individual column names/formulas
		(m, 0) for individual row names/formulas
		(m, n) with n, m > 0 for body names and formulas
	However, the string form of the last tuple will look like this:
	    T:C
		T:R
		T:m,n
    """
	def __init__(self, initial_val):
		"""Initialize with tuple t or string s"""
		if type(initial_val) is types.StringType:
			self.setFromString(initial_val)
		else:
			if type(initial_val) is not types.TupleType:
				raise ARError("Bad Initial AR Value: %s" % (initial_val,))
			self.t = initial_val

	def append(self, n):
		"""Return new Absolute Reference to nth obj in current array"""
		assert not self.t or type(self.t[-1]) is types.IntType, self.t
		return AbsoluteReference(self.t + (n,))
		
	def append_table(self, rownum, colnum = None):
		"""Make a table reference.  args can be a pair of ints or 'R' or 'C'"""
		assert not self.t or type(self.t[-1]) is types.IntType, self.t
		assert rownum in ('R', 'C') or colnum is not None, (rownum, colnum)
		if colnum is None: return AbsoluteReference(self.t + ((rownum,),))
		else: return AbsoluteReference(self.t + ((rownum, colnum),))
	
	def __str__(self):
		"""Return string form of self.  Just use tuple notation for now"""
		def comma_sep(t): return ','.join(str(n) for n in t)

		if self.istable():
			return '(%s, T:%s)' % (comma_sep(self.t[:-1]),
								   comma_sep(self.t[-1]))
		else: return '(%s)' % (comma_sep(self.t),)
	
	def setFromString(self, s):
		"""Inverse of __str__; set reference from string s"""
		def err():
			raise ARError("Invalid Absolute Reference string: '%s'" % (s,))
		def string_to_elem(substr):
			"""Convert s to tuple element"""
			if substr == 'C': return 'C'
			elif substr.startswith('T:'):
				if substr[2:] == 'C': return ('C',)
				elif substr[2:] == 'R': return ('R',)
				split_list = substr[2:].split(',')
				if len(split_list) != 2: err()
				try: return (int(split_list[0]), int(split_list[1]))
				except ValueError: err()
			try: return int(substr)
			except ValueError: err()

		if len(s) < 2 or s[0] != '(' or s[-1] != ')': err()
		if len(s) == 2:
			self.t = ()
			return
		split_list = [substr.strip() for substr in s[1:-1].split(',')]
		if split_list and not split_list[-1]:
			split_list = split_list[:-1] # allow trailing comma
		result_list = []
		for i in range(len(split_list)):
			substr = split_list[i]
			if substr[0] in ('T', 'C'): # make sure T and C are last elements
				final_substr = ','.join(split_list[i:])
				result_list.append(string_to_elem(final_substr))
				break
			else:
				result_list.append(string_to_elem(substr))
		self.t = tuple(result_list)
			
	def __eq__(self, ar):
		"""Compare Absolute References based on the tuple value"""
		return self.t == ar.t
	
	def car(self):
		"""Return the first element in the absolute reference"""
		return self.t[0]
	
	def cdr(self):
		"""Return an AbsoluteReference minus the first index"""
		return AbsoluteReference(self.t[1:])
	
	def __len__(self):
		"""Return length of the AbsoluteReference"""
		return len(self.t)
	
	def isroot(self):
		"""True if abs ref is the root of the hierarchy"""
		return self.t == ()
	
	def parent(self):
		"""Return an absref immediately above in the hierarchy"""
		return AbsoluteReference(self.t[:-1])
		
	def last(self): return self.t[-1]

	def next(self, n):
		"""Add n to the last element of the absolute reference"""
		l = len(self.t)
		assert l > 0, self
		return AbsoluteReference(self.t[:l-1]+(self.t[l-1]+n,))

	def iscol(self):
		"""True if self is the absref for a column formula/propset"""
		return self.t[-1] == 'C'

	def istable(self):
		"""True if self is a reference inside a table"""
		return self.t and isinstance(self.t[-1], types.TupleType)


class PropertySet:
	"""The properties of an object that a program may specify

	For now objects only have a name and a formula.  Name should be a
	string but formula is a formula object. This also parses the formula into an
	abstract syntax tree.
	"""
	def __init__(self, name = None, form_str = None):
		"""Initialize with both name and formula"""
		assert (name is None or
				type(name) in (types.StringType, types.FloatType)), name
		assert (form_str is None or
				isinstance(form_str, types.StringType)), form_str
		self.name = name
		self.set_formstr(form_str)

	def set_formstr(self, form_str):
		"""Update propertyset with given formula string."""
		self.form_str = form_str
		self.ast = parse.parse_string(form_str) if form_str else None
		assert not self.form_str or self.ast, ("Bad formula: "+self.form_str)

	def get_exobj_name(self):
		"""Return the ExObject (ExNum or ExString) form of the name"""
		if self.name is None: return objects.ExBlank()
		elif type(self.name) is types.StringType:
			return objects.ExString(self.name)
		return objects.ExNum(float(self.name)) 


class ProgramError(Exception): pass

class Program:
	"""Represent an entire Explicans program

	For now, a program is just a mapping from AbsoluteReferences to
	properties, which at the moment are just names and formulas.
	"""
	Program_Header = ("Reference", "Name", "Formula")
	def __init__(self, s = None):
		"""Initialize program from string s, or blank if None"""
		self.d = {}
		if s is not None: self.add_from_string(s)
	
	def addline(self, abref, propset):
		"""AbsoluteReference and PropertySet to program"""
		key = abref.t # use tuple as key
		if key in self.d:
			assert False, "Abs Reference %s already in program" % (abref,)
		self.d[key] = propset
	
	def __str__(self):
		"""Represent in csv format"""
		sf = cStringIO.StringIO()
		writer = csv.writer(sf)
		writer.writerow(self.Program_Header)
		for abref_tuple, propset in self.d.items():
			writer.writerow((str(AbsoluteReference(abref_tuple)),
							 propset.name or "", propset.form_str or ""))
		return sf.getvalue()
	
	def __getitem__(self, abref):
		"""Given an absolute reference, return corresponding PropertySet"""
		return self.d[abref.t]
	
	def __len__(self):
		"""Return number of lines in the program"""
		return len(self.d)
	
	def add_from_string(self, s):
		"""Parse the program in string s and add the lines from there"""
		reader = csv.reader(cStringIO.StringIO(s.strip()),
					skipinitialspace=True)
		first_line = reader.next()
		if tuple(first_line) != self.Program_Header:
			raise ProgramError("Invalid Header: %s" % (first_line,))
		for abref_str, name, fstr in reader:
			if not name: name = None
			else:
				try: name = float(name)
				except ValueError: pass
			self.addline(AbsoluteReference(abref_str),PropertySet(name, fstr))
	
	def __setitem__(self, abref, val):
		"""Set a new item at given absolute reference"""
		self.d[abref.t] = val

	def getchildren(self, abref):
		"""Return list pairs (index, propset) for children of abref
		
		Current search through whole program---consider replacing node
		dictionary with tree.
		"""
		child_tuples = [t for t in self.d.keys()
					    if len(t) == len(abref)+1
						    and t[:len(abref)] == abref.t
							and t[-1] != 'C']
		child_tuples.sort()
		return [(t[-1], self.d[t]) for t in child_tuples]

	def get_col_ps(self, abref):
		"""Return the column propset for given abs ref, or raise KeyError"""
		return self[abref.append('C',)]

	def get_table_axis_propsets(self, abref):
		"""Return (row axis propset, col axis propset) for table at abref"""
		try: row_ps = self[abref.append_table('R')]
		except KeyError: row_ps = None

		try: col_ps = self[abref.append_table('C')]
		except KeyError: col_ps = None
		return (row_ps, col_ps)

	def get_table_rowname_ps(self, abref, rownum):
		"""Return the table row propset at rownum, or None"""
		try: return self[abref.append_table(rownum, 0)]
		except KeyError: return None
						 
	def get_table_colname_ps(self, abref, colnum):
		"""Return the table col propset at rownum, or None"""
		try: return self[abref.append_table(0, colnum)]
		except KeyError: return None

	def get_table_value_ps(self, abref, rownum, colnum):
		"""Return the table col propset at rownum, or None"""
		try: return self[abref.append_table(rownum, colnum)]
		except KeyError: return None
