/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class rt02d : declaration for rt02d               *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.0.8	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2005 CREUSE Emmanuel
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class rt02d
    \brief rt02d library \n

    \htmlonly 
    <FONT color="#838383">

    rt02d belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2005 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.0.8
    \date 2005
    \bug none
    \warning none
*/

 
#ifndef __cplusplus
#error Must use C++ for the type p1hp_2d
#endif

#ifndef _rt02d_h
#define _rt02d_h

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../../MOL++/matrix.h"
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MESH_TRI_2D_H)
#include "../meshes/mesh_tri_2d.h"
#endif

#if !defined(__INTEGRATION_H)
#include "../../MOL++/integration.h"
#endif
/*
#if !defined(__MATRIX_CREUSE_H)
#include "../../MOL++/matrix_creuse.h"
#endif
*/


#if !defined(__SPMATRIX_H)
#include "../../MOL++/spmatrix.h"
#endif


using namespace std;

//=============================
template <class T> class rt02d:public mesh_tri_2d<T>
//=============================
{
  private :
  int DofNumber_;
  int GlobalDofNumber_;
  public :
    rt02d();
    rt02d(const char*,const char*);
    rt02d(const char*);
    rt02d(const rt02d<T>&);
    rt02d<T>& operator=(const rt02d<T>&);
    friend  int operator == <T> (const rt02d<T>&,const rt02d<T>&);
    friend  int operator != <T> (const rt02d<T>&,const rt02d<T>&);
    int operator()(int,int) const;
    int DofNumber()const{return (DofNumber_);};
    int GlobalDofNumber()const{return (GlobalDofNumber_);};
    matrix<T> RefBasisFunction(T,T)const;
    vector<T> Fk(T,T, int) const;
    matrix<T> Ak(int, integration<T>, const vector<T> &, T) const;
    matrix<T> Bk(int) const;
    matrix<T> Ck(int) const;
    T int_K_f(T (*f)(T,T),int,integration<T> itg) const;
    void Assemble(spmatrix<T>&, vector<T>&,T (*f)(T,T),const vector<T>&,T,int) ;
    void Dirichlet(spmatrix<T>&, vector<T>&,T (*u)(T,T),T) ;
    void Compute_u(vector<T>&, const vector<T>&, T (*f)(T,T),const vector<T> &, T, int) ;
    void Compute_p(vector<T>&, const vector<T>&, const vector<T>&,const vector<T>&, T,int);
    T error_p(const vector<T>&,T (*px)(T,T),T (*py)(T,T),T,int) ;
    T error_u(const vector<T>&,T,T (*u)(T,T),int) ;
    T error_loc_u(T,T (*u)(T,T),int,int) ;
    T eta_11_K_p(const vector<T>&,T,int,int);
    T eta_13_K_p(const vector<T>&,T,int);
    T eta_1(const vector<T>&,T,int); 
    T eta_2_K_q(T (*f)(T,T),T,int,int); 
    T eta_2(T (*f)(T,T),T,int);
    void refine_FE(vector<int>&); // refine the Finite Element
    void refine_FE_angle(vector<int>&, T angle); // refine the Finite Element
};
/*!
  \brief default constructor of the rt02d class
*/
//=============================
template <class T>
rt02d<T>::rt02d() :mesh_tri_2d<T>()
//=============================
{
DofNumber_=0;
GlobalDofNumber_=0;
}

/*!
  \brief constructor of the rt02d class. The arguments are 2 files
  file1 and file2. file1 must be documented by the number of elements, node 's numerotation
  of each element. file2 indicate the dimension (2 or 3), 
  number of nodes and each node coordinates with a boundary condition (1 if node is on the boundary, 0 else).
  \param file1 string
  \param file2 string
  */
//=============================
template <class T> 
rt02d<T>::rt02d(const char*file1,const char*file2) :mesh_tri_2d<T>(file1,file2)
//=============================
{
DofNumber_=3;
GlobalDofNumber_=nbsegments_;
}

/*!
  \brief constructor of the rt02d class. It makes the same actions as the previous constructor, the 
  argument file is the concatenation of file1 and file2.
  \param file string
*/

//=============================
template <class T> 
rt02d<T>::rt02d(const char*file1) :mesh_tri_2d<T>(file1)
//=============================
{
DofNumber_=3;
GlobalDofNumber_=nbsegments_;
}

/*!
  \brief copy constructor of the rt02d class
  \param EF  reference on rt02d object
  */
//=============================
template <class T> 
rt02d<T>::rt02d(const rt02d<T>& EF) :mesh_tri_2d<T>(EF)
//=============================
{
  DofNumber_=EF.DofNumber_;
  GlobalDofNumber_=EF.GlobalDofNumber_;
}

/*!
  \brief affectation operator for the rt02d class.
  \param EF reference on rt02d object 
  \return  reference on rt02d object
 */
//========================================================
template <class T>
rt02d<T>& rt02d<T>::operator =(const rt02d<T>& EF)
//========================================================
{
 (*this).mesh_tri_2d<T>::operator=(EF);
  DofNumber_=EF.DofNumber_;
  GlobalDofNumber_=EF.GlobalDofNumber_;
  return(*this);
}

/*!
   \param xref type T
   \param yref type T
   \return matrix \f$(\vec{\varphi_1}(xref,yref),\vec{\varphi_2}(xref,yref),\vec{\varphi_3}(xref,yref))\f$ 
   where \f$(\vec{\varphi_i)}_{1\leq i\leq 3}\f$ is the i-th basis function.
  */
//=============================
template <class T> 
matrix<T> rt02d<T>::RefBasisFunction(T xref, T yref)const
//=============================
{
matrix<T> P(2,DofNumber_);
P(1,1)= xref   ; P(1,2)= xref-1   ; P(1,3)= xref     ;
P(2,1)= yref   ; P(2,2)= yref     ; P(2,3)= yref-1   ;
return(P);
}


/*!
   \param num_element element's number
   \param j the j-th dof 
   \return global number of the j-th of the element num_element
  */
//====================================================
template <class T> 
int rt02d<T>::operator ()(int num_element,int j) const
//====================================================
{
return EdgesPerTriangle_(num_element,j);
}

/*!
   \param EF1 type const rt02d<T>&
   \param EF2 type const rt02d<T>&
   \return 1 if EF1 and  EF2 are equal, 0 else
  */
//==========================================================================
template <class T>
int operator ==(const rt02d<T>& EF1,const rt02d<T>& EF2)
//==========================================================================
{
  int boolean=1;
  boolean*=(mesh_tri_2d<T>(EF1)==mesh_tri_2d<T>(EF2));
  boolean*=(EF1.DofNumber_==EF2.DofNumber_);
  boolean*=(EF1.GlobalDofNumber_==EF2.GlobalDofNumber_);
  return(boolean);
}

/*!
   \param EF1 type const rt02d<T>&
   \param EF2 type const rt02d<T>&
   \return 1 if EF1 and  EF2 are different, 0 else
  */
//==========================================================================
template <class T>
int operator !=(const rt02d<T>& EF1,const rt02d<T>& EF2)
//==========================================================================
{
  return(!(EF1==EF2));
}

/*!
   \param num_element type int
   \param itg type integration<T>
   \param P type vector<T>& (contains the ddl associated to each segment of the mesh, representing the value of \f$ \vec{p}=|\nabla u|^{p-2} \vec{\nabla u} \f$).)
   \param q type T
   \return matrix \f$ \int_{K_{num_{element}}} |\vec{p}|^{q-2} \vec{\phi}_j^{K_{num_{element}}}.\vec{\phi}_i^{K_{num_{element}}} d \Omega , \quad 1 \leq i,j \leq 3\f$.
  */
//====================================================
template <class T>
matrix<T> rt02d<T>::Ak(int num_element,integration<T> itg, const vector<T>& P, T q) const
//====================================================
{
assert((num_element>=1)&&(num_element<=nbelement_));
T omegak, omegal, xk, xl;
T sumx,sumy;

matrix<T> A(3,3);
matrix<T> Mk(element[num_element-1].Mk());


matrix<T> NormalDir(3,3);
for (int i=1; i<=3; i++)
 NormalDir(i,i)=(T)NormalDirection_(num_element,i);

matrix<T> LocalBasisFunction(2,3);

   for (int k=1; k<=itg.NbIntPts(); k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  LocalBasisFunction=Mk*RefBasisFunction(xk,xl)*NormalDir;
	  sumx=0; sumy=0;
	  for (int i=1; i<=3; i++)
	    {
	    sumx+=P[(*this)(num_element,i)]*LocalBasisFunction(1,i);
	    sumy+=P[(*this)(num_element,i)]*LocalBasisFunction(2,i);
	    }
	  A+= omegak*omegal*(T)(1.-xk)*t(LocalBasisFunction)*LocalBasisFunction*(T)(powf(sumx*sumx+sumy*sumy,(q-2.0)/2.0));
	 }
    }
  return(A/(T)(powf(((element[num_element-1]).AbsDet()),q-1.0)));
}


//====================================================
template <class T>
//void rt02d<T>::Assemble(matrix_creuse<T>& A, vector<T>& B, T (*f)(T,T), const vector<T>& P, T q, int NbIntPts) 
void rt02d<T>::Assemble(spmatrix<T>& A, vector<T>& B, T (*f)(T,T), const vector<T>& P, T q, int NbIntPts) 
//====================================================
{
integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);
matrix<T> LocalA(3,3);
matrix<T> invAk(3,3), invAkmod(3,3), matloc(3,3);
matrix<T> FullOne(3,3);
for (int i=1; i<=3; i++)
 for (int j=1; j<=3; j++)
  FullOne(i,j)=1;
T int_f_K, sum;
vector<T> sum_line(3);
matrix<T> vecBk(3,1);
matrix<T> tvecBk(1,3);
matrix<T> coeff(1,1);
//

for (int num_element=1; num_element<=nbelement_; num_element++)
{
invAk=gauss(Ak(num_element,itg,P,q)); // inverse of Ak
vecBk=Bk(num_element);
tvecBk=t(vecBk);
coeff=tvecBk*invAk*vecBk;
int_f_K=int_K_f(f,num_element,itg);

for (int i=1; i<=3; i++)
 for (int j=1; j<=3; j++)
  invAkmod(i,j)=invAk(i,j)*(T)NormalDirection_(num_element,i)*(T)NormalDirection_(num_element,j);

matloc=invAk*vecBk*tvecBk*invAk;
for (int i=1; i<=3; i++)
 for (int j=1; j<=3; j++)
  LocalA(i,j)=invAkmod(i,j)-(T)NormalDirection_(num_element,i)*matloc(i,j)*(T)NormalDirection_(num_element,j)/coeff(1,1);

for (int i=1; i<=3; i++)
 {
  B[(*this)(num_element,i)]+=(T)NormalDirection_(num_element,i)*(invAk*vecBk)(i,1)*int_f_K/coeff(1,1);
 for (int j=1; j<=3; j++)
   A.add((*this)(num_element,i),(*this)(num_element,j),LocalA(i,j));;
 }
}
}

/*! 
   \param A type matrix_creuse<T>
   \param B type vector<T>
   \param dudx type (*f)(T,T)
   \param dudy type (*f)(T,T)
   \param p type T
   \return the matrix \f$ A \f$ and the vector \f$ B \f$ in which the degrees of freedom associated to boundary
   segments are set to zero. In this case, the parameters \f$(*dudx)\f$ , \f$(*dudy)\f$ and \f$p\f$ are not needed. 
   This function can be modified to specify non homogeneous boundary conditions (the lines of the code are put into //), and
   in this case we can choose between rectangle or simpson numerical integration.
   
  */
//====================================================
template <class T>
//void rt02d<T>::Dirichlet(matrix_creuse<T>& A, vector<T>& B, T(*u)(T,T), T p) 
void rt02d<T>::Dirichlet(spmatrix<T>& A, vector<T>& B, T(*u)(T,T), T p) 
//====================================================
{
vector<T> VECM(2),VECA(2),VECB(2);
T uM;
T valeur_M;
T valeur_ddl ;
for (int num_edge=1; num_edge<=GlobalDofNumber_; num_edge++)
 {
 if (Type_of_edge_[num_edge])
  {
     VECA=vertices[SetOfEdges_(num_edge,1)-1];
     VECB=vertices[SetOfEdges_(num_edge,2)-1];
     VECM=(VECA+VECB)/(T)2.0; 
     uM=u(VECM[1],VECM[2]);
     valeur_ddl=uM;  
   B[num_edge]=valeur_ddl; 
   for(int j=1;j<=GlobalDofNumber_;j++)
	A.set(num_edge,j,0.0);
   A.set(num_edge,num_edge,1);	
   for(int k=1;k<=GlobalDofNumber_;k++)
	if(k!=num_edge)
	{
	B[k]+=-A.output(k,num_edge)*B[num_edge];
	A.set(k,num_edge,0.0);
	}	
  }
 }
}

/*! 
   \param f type (*f)(T,T)
   \param num_element type int
   \param itg type integration<T>
   \return the scalar T \f$ \int_{K_{num_{element}}} f(x,y) \, d \Omega \f$.
   
  */

//====================================================
template <class T>
T rt02d<T>::int_K_f(T (*f)(T,T),int num_element,integration<T> itg) const
//====================================================
{
T res=0;
T omegak, omegal, xk, xl;
vector<T> Fxkl;
   for (int k=1; k<=itg.NbIntPts(); k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Fxkl=Fk(xk,xl,num_element);
	    res+= omegak*omegal*(1-xk)*f(Fxkl[1],Fxkl[2]);
	 }
    }
 res*=element[num_element-1].AbsDet();
return(res);
}  

/*! 
   \param num_element type int
   \return the matrix<T> (in fact the vector of size 3) \f$ Bk[i]=|K|_{num_{element}} div \; \vec{\Phi_i}^{K_{num_{element}}}, \quad 1 \leq i \leq 3 \f$.
  */
  
//====================================================
template <class T>
matrix<T> rt02d<T>::Bk(int num_element) const
//====================================================
{
assert((num_element>=1)&&(num_element<=nbelement_));
matrix<T> u(3,1);
for (int i=1; i<=3; i++)
 u(i,1)=(T)NormalDirection_(num_element,i);
return(u);
}

/*! 
   \param num_element type int
   \return the matrix<T> of size \f$3 \times 3\f$ \f$ Ck(i,j)=\int_{e_j}\vec{\Phi_i}^{K_{num_{element}}}.\, \vec{n_{e_j}}^{K_{num_{element}}} \, d \Omega,  \quad 1 \leq i,j \leq 3 \f$, where \f$e_j\f$ is the jth segment of the triangle \f$K_{num_{element}}\f$, opposed to the jth node.
  */

//====================================================
template <class T>
matrix<T> rt02d<T>::Ck(int num_element) const
//====================================================
{
assert((num_element>=1)&&(num_element<=nbelement_));
matrix<T> M(3,3);
for (int i=1; i<=3; i++)
 M(i,i)=NormalDirection(num_element,i);
return(M);
}

/*!
   \param x type T
   \param y type T
   \param i type integer
   \return vector \f$F_k(x,y,i)\f$ 
  */
//====================================================
template <class T>
vector<T> rt02d<T>::Fk(T x,T y, int i) const
//====================================================
{
  matrix<T> MAT(2,2);
  vector<T> VEC(2);
  VEC[1]=x ; VEC[2]=y;

  vector<T> A1A2=element[i-1][2]-element[i-1][1];
  vector<T> A1A3=element[i-1][3]-element[i-1][1];
  MAT(1,1)=A1A2[1];  MAT(1,2)=A1A3[1] ;
  MAT(2,1)=A1A2[2];  MAT(2,2)=A1A3[2];
  return MAT*VEC+element[i-1][1];
}




//====================================================
template <class T>
void rt02d<T>::Compute_u(vector<T>& U, const vector<T>& lambda,T (*f)(T,T), const vector<T>& P, T q, int NbIntPts) 
//====================================================
{
integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);
matrix<T> invAk(3,3);
vector<T> sum_col(3);
matrix<T> vecBk(3,1);
matrix<T> tvecBk(1,3);
matrix<T> coeff(1,1);
matrix<T> C(3,3),line(1,3);

//
T sum;

for (int i=1; i<=nbelement_; i++)
 U[i]=0;
for (int num_element=1; num_element<=nbelement_; num_element++)
{
invAk=gauss(Ak(num_element,itg,P,q)); // inverse of Ak
vecBk=Bk(num_element);
tvecBk=t(vecBk);
coeff=tvecBk*invAk*vecBk;
for (int i=1; i<=3; i++)
 C(i,i)=(T)NormalDirection_(num_element,i);
line=tvecBk*invAk*C;

for (int j=1; j<=3; j++)
  U[num_element]+=line(1,j)*lambda[(*this)(num_element,j)];
U[num_element]+=int_K_f(f,num_element,itg);
U[num_element]/=coeff(1,1);
}

}

//====================================================
template <class T> void rt02d<T>::Compute_p(vector<T>& P, const vector<T>& U, const vector<T>& lambda, const vector<T>& Pm1, T q, int NbIntPts) 
//====================================================
{
// Here for the internal segment, the value of P associate to the segment is computed twice :
// one time for each element containing the segment. It is nevertheless not necessary !
matrix<T> invAk(3,3);
vector<T> vec(3),res(3);

integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

for (int num_element=1; num_element<=nbelement_; num_element++)
{
invAk=gauss(Ak(num_element,itg,Pm1,q)); // inverse of Ak
for (int i=1; i<=3; i++)
 vec[i]=(T)NormalDirection_(num_element,i)*(lambda[(*this)(num_element,i)]-U[num_element]);
res=invAk*vec;
for (int i=1; i<=3; i++)
 P[(*this)(num_element,i)]=res[i]; 
}
}


/*!
   \param P type vector<T>& (contains the ddl associated to each segment of the mesh, representing the value of \f$ \vec{p}=|\nabla u|^{p-2} \vec{\nabla u} \f$).
   \param px type (*f)(T,T)
   \param py type (*f)(T,T)
   \param q type T
   \param NbIntPts type int
   \return \f$ |P-pexact|_{0,q,\Omega} \f$
  */
//====================================================
template <class T> T rt02d<T>::error_p(const vector<T>& P, T (*px)(T,T), T (*py)(T,T), T q, int NbIntPts) 
//====================================================
{

T omegak, omegal, xk, xl;
T valeur_px, valeur_py;
vector<T> Fxkl(2), vec(2);
integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);
matrix<T> Mk(2,2),BasisFunction(2,3);

T error=0;
T error_loc;

for (int num_element=1; num_element<=nbelement_; num_element++)
{

 error_loc=0;
 Mk=element[num_element-1].Mk();
  for (int k=1; k<=itg.NbIntPts(); k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Fxkl=Fk(xk,xl,num_element);
	  BasisFunction=Mk*RefBasisFunction(xk,xl);
	  vec[1]=0; vec[2]=0;
	   for (int j=1; j<=3; j++)
	    {
	    vec[1]+=P[(*this)(num_element,j)]*BasisFunction(1,j)*(T)NormalDirection_(num_element,j);
	    vec[2]+=P[(*this)(num_element,j)]*BasisFunction(2,j)*(T)NormalDirection_(num_element,j);
	    }
	  vec/=element[num_element-1].AbsDet();
//	  if (NormGradu(Fxkl[1],Fxkl[2])<epsilon)
//	   {
//	    valeur_px=0;
//	    valeur_py=0;
//	    vec[1]=0;
//	    vec[2]=0;
//	   }
//	  else
//	   {  
	    valeur_px=px(Fxkl[1],Fxkl[2]);
	    valeur_py=py(Fxkl[1],Fxkl[2]);
//	   } 
	  error_loc+=omegak*omegal*(1-xk)*powf(sqrt((powf((valeur_px-vec[1]),2)+powf((valeur_py-vec[2]),2))),q);
	  
	 }
    }
  
    
    
  error_loc*=element[num_element-1].AbsDet();
 // cout << "num_element= " << num_element << " : error_loc= " << error_loc << "\n";
 // cout << "---------------------------------------------------------" <<"\n";
 // cout << "---------------------------------------------------------" <<"\n";
 // cout << "\n";
  error+=error_loc;  
}
return (powf(error,1.0/q));
}

/*!
   \param US type vector<T>& (contains the value of \f$u\f$ on each triangle of the mesh)
   \param u type (*f)(T,T)
   \param NbIntPts type int
   \return \f$ |US-uexact|_{0,2,\Omega} \f$
  */
//====================================================
template <class T> T rt02d<T>::error_u(const vector<T>& US, T s,T (*u)(T,T),int NbIntPts) 
//====================================================
{

T omegak, omegal, xk, xl;
vector<T> Fxkl(2);
integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

T error=0;
T error_loc;

for (int num_element=1; num_element<=nbelement_; num_element++)
{

 error_loc=0;
  for (int k=1; k<=itg.NbIntPts(); k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Fxkl=Fk(xk,xl,num_element);
	  error_loc+=omegak*omegal*(1-xk)*powf(abs(u(Fxkl[1],Fxkl[2])-US[num_element]),s);
	 }
    }
  error_loc*=element[num_element-1].AbsDet();
  error+=error_loc;  
}
  return (powf(error,1.0/s));
}

/*!
   \param Umaille type vector<T>& (contains the value of \f$u\f$ on each triangle of the mesh)
   \param u type (*f)(T,T)
   \param NbIntPts type int
   \param num_element type int
   \return \f$ |Umaille-uexact|_{0,2,K_{num_{element}}} \f$
  */
//====================================================
template <class T> T rt02d<T>::error_loc_u(T Umaille, T (*u)(T,T),int NbIntPts,int num_element) 
//====================================================
{

T omegak, omegal, xk, xl;
vector<T> Fxkl(2);
integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

T error_loc=0;

  for (int k=1; k<=itg.NbIntPts(); k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Fxkl=Fk(xk,xl,num_element);
	  error_loc+=omegak*omegal*(1-xk)*powf(u(Fxkl[1],Fxkl[2])-Umaille,2);
	 }
    }
  error_loc*=element[num_element-1].AbsDet();
return (sqrt(error_loc));
}

/*!
   \param TABTRI type vector<int>& (the size of TABTRI is the number of triangle of the mesh. TABTRI[i]=-1 if triangle i must be refined, else TABTRI[i]=0)
   \return the finite element, with the new mesh
  */
//=============================
template <class T>
void rt02d<T>::refine_FE(vector<int> & TABTRI)
//=============================
{
  refine_mesh(TABTRI);
  GlobalDofNumber_=nbsegments_;
}

/*!
   \param TABTRI type vector<int>& (the size of TABTRI is the number of triangle of the mesh. TABTRI[i]=-1 if triangle i must be refined, else TABTRI[i]=0)
   \return the finite element, with the new mesh (and angle restriction)
  */
//=============================
template <class T>
void rt02d<T>::refine_FE_angle(vector<int> & TABTRI, T angle)
//=============================
{
  refine_mesh_angle(TABTRI,angle);
  GlobalDofNumber_=nbsegments_;
}

//====================================================
template <class T> T rt02d<T>::eta_11_K_p(const vector<T>& P,T s,int NbPts,int num_element) 
//====================================================
{
T hK=hminT_[num_element];
T omegak, omegal, xk, xl;
vector<T> Fxkl(2), vec(2);
integration<T> itg;
itg.Gauss(NbPts,0.0,1.0);
matrix<T> Mk(2,2),BasisFunction(2,3);
T value_loc;

 value_loc=0;
 Mk=element[num_element-1].Mk();
  
  for (int k=1; k<=itg.NbIntPts(); k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Fxkl=Fk(xk,xl,num_element);
	  BasisFunction=Mk*RefBasisFunction(xk,xl);
	  vec[1]=0; vec[2]=0;
	   for (int j=1; j<=3; j++)
	    {
	    vec[1]+=P[(*this)(num_element,j)]*BasisFunction(1,j)*(T)NormalDirection_(num_element,j);
	    vec[2]+=P[(*this)(num_element,j)]*BasisFunction(2,j)*(T)NormalDirection_(num_element,j);
	    }
	  vec/=element[num_element-1].AbsDet();
	  value_loc+=omegak*omegal*(1-xk)*powf(sqrt((powf(vec[1],2)+powf(vec[2],2))),s/(s-1.0));
	 }
    }
  value_loc*=element[num_element-1].AbsDet();

return (powf(hK,s)*value_loc);
}

//====================================================
template <class T> T rt02d<T>::eta_13_K_p(const vector<T>& P,T s,int num_element) 
//====================================================
{
T value=0;
int num_seg,NumMaille1,NumMaille2;
int NumNoeudA,NumNoeudB;
T value_i;

for (int i=1; i<=3; i++)
    {
     num_seg=EdgesPerTriangle_(num_element,i);
     NumNoeudA=SetOfEdges_(num_seg,1);
     NumNoeudB=SetOfEdges_(num_seg,2);
     NumMaille1=TrianglesPerEdge_(num_seg,1); // c'est K+
     NumMaille2=TrianglesPerEdge_(num_seg,2); // c'est K-
     vector<T> AB(vertices[NumNoeudB-1]-vertices[NumNoeudA-1]);
     vector<T> M((vertices[NumNoeudB-1]+vertices[NumNoeudA-1])/((T)2.0));
     // K+ est  droite du segment quand on le parcourt dans le sens A->B
     T E=AB.norm();
     vector<T> n(AB.ortho2D()/E); // c'est donc la normale unitaire sortante  K+
     vector<T> p_M_Maille1(2),p_M_Maille2(2);
     
     
     if (NumMaille1 && NumMaille2) // segment interne
      {
      p_M_Maille1*=(T)0.0; p_M_Maille2*=(T)0.0;
      for (int j=1; j<=3; j++)
       {
       p_M_Maille1+=(P[EdgesPerTriangle_(NumMaille1,j)])*(M-element[NumMaille1-1][j])*(T)NormalDirection_(NumMaille1,j);
       p_M_Maille2+=(P[EdgesPerTriangle_(NumMaille2,j)])*(M-element[NumMaille2-1][j])*(T)NormalDirection_(NumMaille2,j);
       }
       p_M_Maille1/=element[NumMaille1-1].AbsDet();
       p_M_Maille2/=element[NumMaille2-1].AbsDet();
       value_i=powf(abs(
               powf(sqrt((pow(p_M_Maille1[1],2)+pow(p_M_Maille1[2],2))),(2-s)/(s-1))*(p_M_Maille1*AB/E)
              -powf(sqrt((pow(p_M_Maille2[1],2)+pow(p_M_Maille2[2],2))),(2-s)/(s-1))*(p_M_Maille2*AB/E)
	               ),s
		   )*E;
      
     value+=value_i*E;
     } 
     }
     
return(value);
}

//======================================================================
template <class T> T rt02d<T>::eta_1(const vector<T>& P,T s,int NbPts) 
//======================================================================
{
T value=(T)0.0;
for (int num_element=1; num_element<=nbelement_; num_element++)
 //value+=eta_11_K_p(P,s,NbPts,num_element);
value+=eta_11_K_p(P,s,NbPts,num_element)+eta_13_K_p(P,s,num_element);
return (powf(value,1.0/s)); 
}

//=================================================================================
template <class T> T rt02d<T>::eta_2_K_q(T (*f)(T,T),T s,int NbPts,int num_element) 
//=================================================================================
{
T omegak, omegal, xk, xl;
vector<T> Fxkl(2);
integration<T> itg;
itg.Gauss(NbPts,0.0,1.0);

T error_loc=0;
vector<T> M(Fk(1.0/3.0,1.0/3.0,num_element)); // Barycentre de la maille

  for (int k=1; k<=itg.NbIntPts(); k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Fxkl=Fk(xk,xl,num_element);
	  error_loc+=omegak*omegal*(1-xk)*powf(f(Fxkl[1],Fxkl[2])-f(M[1],M[2]),2);
	 }
    }
  error_loc*=element[num_element-1].AbsDet();
return (powf(error_loc,s/2.0));
}
//=================================================================
template <class T> T rt02d<T>::eta_2(T (*f)(T,T),T s,int NbPts) 
//=================================================================
{
T value=0.0;
for (int num_element=1; num_element<=nbelement_; num_element++)
value+=eta_2_K_q(f,s,NbPts,num_element);
return (powf(value,1.0/s)); 
}


	 
#endif


