/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class tetrap0 : declaration for tetrap0               *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.0.8	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2004 CREUSE Emmanuel
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class tetrap0
    \brief tetrap0 library \n

    \htmlonly
    <FONT color="#838383">

    tetrap0 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2004 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.2.0
    \date 2003-2006
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type tetrap0
#endif

#ifndef _tetrap0_h
#define _tetrap0_h

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../../MOL++/matrix.h"
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MESH_H)
#include "../meshes/mesh.h"
#endif

using namespace std;


//=============================
template <class T> class tetrap0:public mesh<tetrahedre,T>
//=============================
{
  private :
  int DofNumber_;
  int GlobalDofNumber_;

  public :
  tetrap0();
  tetrap0(const char*,const char*);
  tetrap0(const char*);
  tetrap0(const tetrap0<T>&);
  tetrap0<T>& operator=(const tetrap0<T>&);//tetrap0 affectation
  int DofNumber()const{return (DofNumber_);};
  int GlobalDofNumber()const{return (GlobalDofNumber_);};
  vector<T> BasisFunction(T,T,T)const;
  matrix<T> DiffBasisFunction(T xref=0,T yref=0, T zref=0) const;
  virtual int operator ()(int,int) const;
  friend  int operator == <T> (const tetrap0<T>&,const tetrap0<T>&);
  friend  int operator != <T> (const tetrap0<T>&,const tetrap0<T>&);
  vector<T> Fk(T,T,T, int) const;
  T Integrate_f(T (*f)(T,T,T),int numele,integration<T>) const; //compute int_numele(f)dx
  T error_L2 (T (*u)(T,T,T),const vector<T>& uh,int NbIntPts) const ;
  T error_L2_loc_2 (T (*u)(T,T,T),const vector<T>& uh,int NbIntPts, int num_element) const ;
};           
//=============================
template <class T> 
tetrap0<T>::tetrap0() :mesh<tetrahedre,T>()
//=============================
{
DofNumber_=0;
GlobalDofNumber_=0;
}



//=============================
template <class T> 
tetrap0<T>::tetrap0(const char*file1,const char*file2) :mesh<tetrahedre,T>(file1,file2)
//=============================
{
DofNumber_=1;
GlobalDofNumber_=nbelement_;
}

//=============================
template <class T> 
tetrap0<T>::tetrap0(const char*file) :mesh<tetrahedre,T>(file)
//=============================
{
DofNumber_=1;
GlobalDofNumber_=nbelement_;
}

//=============================
template <class T> 
tetrap0<T>::tetrap0(const tetrap0<T>& EF) :mesh<tetrahedre,T>(EF)
//=============================
{
  DofNumber_=EF.DofNumber_;
  GlobalDofNumber_=EF.GlobalDofNumber_;
}
//========================================================
template <class T>
tetrap0<T>& tetrap0<T>::operator =(const tetrap0<T>& EF)
//========================================================
{
 (*this).mesh<tetrahedre,T>::operator=(EF);
  DofNumber_=EF.DofNumber_;
  GlobalDofNumber_=EF.GlobalDofNumber_;
  return(*this);
}

//=============================
template <class T> 
vector<T> tetrap0<T>::BasisFunction(T xref, T yref, T zref)const
//=============================
{
vector<T> P(DofNumber_);
P[1]=1;
return(P);
}

//=============================
template <class T> 
matrix<T> tetrap0<T>::DiffBasisFunction(T xref,T yref, T zref)const
//=============================

{
matrix<T> DP(3,DofNumber_);
DP(1,1)=0;
DP(2,1)=0;
DP(3,1)=0;
return(DP);
}
//====================================================
template <class T> 
int tetrap0<T>::operator ()(int num_element,int j) const
//====================================================
{
assert((j>=1)&&(j<=DofNumber_)&&(num_element>=1)&&(num_element<=nbelement_));
return num_element;
}

//==========================================================================
template <class T>
int operator ==(const tetrap0<T>& EF1,const tetrap0<T>& EF2)
//==========================================================================
{
  int boolean=1;
  boolean*=(mesh<tetrahedre,T>(EF1)==mesh<tetrahedre,T>(EF2));
  boolean*=(EF1.DofNumber_==EF2.DofNumber_);
  boolean*=(EF1.GlobalDofNumber_==EF2.GlobalDofNumber_);
  return(boolean);
}

//==========================================================================
template <class T>
int operator !=(const tetrap0<T>& EF1,const tetrap0<T>& EF2)
//==========================================================================
{
  return(!(EF1==EF2));
}

//====================================================
template <class T> 
vector<T> tetrap0<T>::Fk(T x,T y,T z, int i) const
//====================================================
{

  matrix<T> MAT(3,3);
  vector<T> A1A2(3), A1A3(3), A1A4(3), A1(3), VEC(3);
  VEC[1]=x ; VEC[2]=y;  VEC[3]=z;
  
  A1=vector<T>(element[i-1][1]);
  
   
  A1A2=vector<T>(element[i-1][2])-A1;
  A1A3=vector<T>(element[i-1][3])-A1;
  A1A4=vector<T>(element[i-1][4])-A1;

  MAT(1,1)=A1A2[1]; MAT(2,1)=A1A2[2]; MAT(3,1)=A1A2[3]; 
  MAT(1,2)=A1A3[1]; MAT(2,2)=A1A3[2]; MAT(3,2)=A1A3[3];
  MAT(1,3)=A1A4[1]; MAT(2,3)=A1A4[2]; MAT(3,3)=A1A4[3];

  return MAT*VEC+A1;  
}

//====================================================
template <class T>
T tetrap0<T>::Integrate_f(T (*f)(T,T,T),int num_element,integration<T> itg) const
//====================================================
{

  T xk,yj,zi,omegak,omegaj,omegai,valeurfonction;
  T  valeurint;
 
  valeurint=0.0;
 
  for (int k=1; k<=itg.NbIntPts(); k++) 
   {
    omegak=itg.weight(k);
    xk=itg.point(k); 
    for (int j=1; j<=itg.NbIntPts(); j++)
     {
      omegaj=itg.weight(j);
      yj=itg.point(j); 
      for (int i=1; i<=itg.NbIntPts(); i++)
       {
	omegai=itg.weight(i);
        zi=itg.point(i);
              
        valeurfonction=f(Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[1],
			 Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[2],
                         Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[3]);
        valeurint+=omegai*omegaj*omegak*abs((1.0-xk)*(1.0-xk-yj+xk*yj))*valeurfonction;
       }
     }
   }
  
   valeurint*=element[num_element-1].AbsDet();

   return(valeurint);
}

//======================================================
template <class T>
T tetrap0<T>::error_L2 (T (*u)(T,T,T),const vector<T>& uh,int NbIntPts) const
//======================================================
{
  T GlobalError=0;
  T LocalError;

  integration<T> itg;
  itg.Gauss(NbIntPts,0.0,1.0);

  T xk,yj,zi,omegak,omegaj,omegai;
 
  GlobalError=0.0;

  for(int num_element=1;num_element<=nbelement_;num_element++)
  {
   LocalError=0;
   for (int k=1; k<=itg.NbIntPts(); k++) 
   {
    omegak=itg.weight(k);
    xk=itg.point(k); 
    for (int j=1; j<=itg.NbIntPts(); j++)
     {
      omegaj=itg.weight(j);
      yj=itg.point(j); 
      for (int i=1; i<=itg.NbIntPts(); i++)
       {
	omegai=itg.weight(i);
        zi=itg.point(i);
              
        LocalError+=omegai*omegaj*omegak*abs((1.0-xk)*(1.0-xk-yj+xk*yj))*
	  pow((u(Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[1],
                Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[2],
                Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[3])
	       -uh[num_element]),2);
       }
     }
   }
  
   GlobalError+=element[num_element-1].AbsDet()*LocalError;
  }
   
return(sqrt(GlobalError));
}
//======================================================
template <class T>
T tetrap0<T>::error_L2_loc_2 (T (*u)(T,T,T),const vector<T>& uh,int NbIntPts,int num_element) const
//======================================================
{
  T LocalError=0;

  integration<T> itg;
  itg.Gauss(NbIntPts,0.0,1.0);

  T xk,yj,zi,omegak,omegaj,omegai;

   for (int k=1; k<=itg.NbIntPts(); k++) 
   {
    omegak=itg.weight(k);
    xk=itg.point(k); 
    for (int j=1; j<=itg.NbIntPts(); j++)
     {
      omegaj=itg.weight(j);
      yj=itg.point(j); 
      for (int i=1; i<=itg.NbIntPts(); i++)
       {
	omegai=itg.weight(i);
        zi=itg.point(i);
              
        LocalError+=omegai*omegaj*omegak*abs((1.0-xk)*(1.0-xk-yj+xk*yj))*
	  pow((u(Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[1],
                Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[2],
                Fk(xk,(1-xk)*yj,(1-xk-yj+xk*yj)*zi,num_element)[3])
	       -uh[num_element]),2);
       }
     }
   }
   
return(element[num_element-1].AbsDet()*LocalError);
}
       
#endif
