/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class tetrahedre : declaration for tetrahedre     *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.0.6	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2003 CREUSE Emmanuel
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class tetrahedre
    \brief tetrahedre library \n

    \htmlonly 
    <FONT color="#838383">

    tetrahedre belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2003 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.0.6
    \date 2003
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type tetrahedre
#endif

#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../../MOL++/matrix.h"
#endif

#if !defined(__NODE_H)
#include "../meshes/node.h"
#endif

#ifndef _tetrahedre_h
#define _tetrahedre_h

using namespace std;

/*
Let A,B,C,D 4 nodes of R3, we define
the template class tetrahedre as follows:

       (*)--->&B
       /\\
      / \ \
     /  \  \
    /  (*)---->&D (Tetrahedre<<==>>set of pointers on nodes)
   / /    \  \
  //________\_\
(*)--->&C    (*)--->&A

*/





//==============================================
template <class T> class tetrahedre
//==============================================
{
  protected :
  node<T>** vertex;// ....................define an array of pointers
  int alive;// ...........................1 if node exists else O
  public:
  tetrahedre();// ..........................default constructor
  ~tetrahedre();//......................... destructor
  tetrahedre(node<T>&,node<T>&,node<T>&,node<T>&);
    tetrahedre(node<T> * A[4]);
  // .....................................constructor(node,node, node,node)
  tetrahedre(const tetrahedre<T>&);//........ copy constructor
  tetrahedre<T>& operator=(const tetrahedre<T>&);
  friend  ostream& operator << <T> (ostream&, const tetrahedre<T>&);
  node<T>& operator [] (int)const;// .....return the i-th node of tetrahedre
  friend  int operator == <T> (const tetrahedre<T>&, const tetrahedre<T>&);
  T AbsDet()const;//...................return the |det| of the affine transfo.
  matrix<T> Mk()const;//...........matrix of the affine transfo.
  matrix<T> InvMk()const;//.........inverse matrix of the affine transfo
  int nb_nodes() const;//........returns the number of nodes of a tetra
};

/*!
  \brief Default Constructor 
  \n \n
  alive is set to 0 but the memory is not allocated to store the adresses of the nodes of the tetrahedron in vertex
  */
//==============================================
template <class T> tetrahedre<T>::tetrahedre()
//==============================================
{
  alive=0;
}
/*!
  \brief Destructor
  \n \n
  Deletes vertex and set alive to 0 
*/
//==============================================
template <class T> tetrahedre<T>::~tetrahedre()
//==============================================
{
  if(alive)
      delete [] vertex;
  alive=0;
}

/*!
  \brief Constructor 
  \n \n
  \param A node \n
  \param B node \n
  \param C node \n
  \param D node \n \n
  alive is set to 1 and vertex is built with pointers towards A,B,C and D
  */
//========================================================================
template <class T> 
tetrahedre<T>::tetrahedre(node<T>&A,node<T> &B,node<T> &C,node<T> &D)
//========================================================================
{
  alive=1;
  vertex=new node<T>*[4];
  vertex[0]=&A;
  vertex[1]=&B;
  vertex[2]=&C;
  vertex[3]=&D;
}

/*!
  \brief Constructor 
  \n \n
  \param A an array containing 4 adresses of nodes \n \n
  alive is set to 1 and vertex is built with the 4 adresses contained in A
  */
//========================================================================
template <class T> 
tetrahedre<T>::tetrahedre(node<T>* A[4])
//========================================================================
{
// The entry arguments are the adresses of the nodes, but not the nodes themselves.
  alive=1;
  vertex=new node<T>*[4];
  vertex[0]=A[0];
  vertex[1]=A[1];
  vertex[2]=A[2];
  vertex[3]=A[3];
}


/*!
  \brief Copy constructor
  \param K tetrahedron to duplicate
*/
//========================================================================
template <class T> tetrahedre<T>::tetrahedre(const tetrahedre<T>& K)
//========================================================================
{
  vertex=new node<T>*[4];
  alive=K.alive;
  vertex[0]=K.vertex[0];
  vertex[1]=K.vertex[1];
  vertex[2]=K.vertex[2];
  vertex[3]=K.vertex[3];
}
/*!
  \brief Standard operator = for tetrahedre \n \n
  \param K tetrahedron to copy \n
  \return reference of the left hand size tetrahedron (for multiple equalities)
  \n\n
*/
//=========================================================================
template <class T> tetrahedre<T>& tetrahedre<T>::operator=(const tetrahedre<T>& K)
//=========================================================================
{
  assert(K.alive);
  if(!alive)
      vertex=new node<T>*[4];
  alive=K.alive;
  vertex[0]=K.vertex[0];
  vertex[1]=K.vertex[1];
  vertex[2]=K.vertex[2];
  vertex[3]=K.vertex[3];
  return (*this);
}
/*!
  \brief  overloads output stream for tetrahedron \n \n
  \param exit output flow \n
  \param K tetrahedron
*/
//==========================================================================
template <class T> ostream& operator <<(ostream& exit, const tetrahedre<T>& K)
//==========================================================================
{
  if(!K.alive)
      exit<<"undefined tetrahedre";
  else
      {
      exit<< "tetrahedre:"<<endl;
      exit<< *(K.vertex[0]) << *(K.vertex[1]) <<*(K.vertex[2]) <<*(K.vertex[3]);
      }
  return exit;
}
/*!
  \brief Returns one node of the tetrahedron \n \n
  Warning : we use the mathematical notations so the nodes adresses are in the set [1;4], and not [0:3] \n \n
  \param i node number \n \n
  \return the node i of the tetrahedron \n \n
*/
//================================================================
template <class T> node<T>& tetrahedre<T>::operator [] (int i)const
//================================================================
{
  assert((i>0)&&(i<=4)&&alive);
  return *(vertex[i-1]);
}

/*!
  \brief  Check the equality of two tetrahedra \n \n
  \param K tetrahedron \n
  \param L tetrahedron \n
  \return 1 if equal, 0 if not
*/
//================================================================
template <class T> int operator ==(const tetrahedre<T> &K,const tetrahedre<T> &L)
//================================================================
{
  int boolean=1;
  boolean*=((K[1]==L[1])&&(K[2]==L[2])&&(K[3]==L[3])&&(K[4]==L[4])&&(K.alive==L.alive));
  return(boolean);
}
/*!
  \brief  Returns the absolute value of the matrix of the affine application which
  send the reference triangle to the actual triangle.   \n \n
  \return The value of the absolute value of the determinant
*/
//================================================================
template <class T> T tetrahedre<T>::AbsDet()const
//================================================================
{
return abs(det(Mk()));
}

/*!
  \brief  Returns the 3*3 matrix of the affine application which
  send the reference tetrahedron to the actual tetrahedron.   \n \n
  \return The 3*3 matrix
*/
//================================================================
template <class T> matrix<T> tetrahedre<T>::Mk()const
//================================================================
{
matrix<T> M(3,3);
M(1,1)=(*this)[2][1]-(*this)[1][1]; 
M(1,2)=(*this)[3][1]-(*this)[1][1];
M(1,3)=(*this)[4][1]-(*this)[1][1];
M(2,1)=(*this)[2][2]-(*this)[1][2]; 
M(2,2)=(*this)[3][2]-(*this)[1][2];
M(2,3)=(*this)[4][2]-(*this)[1][2];
M(3,1)=(*this)[2][3]-(*this)[1][3]; 
M(3,2)=(*this)[3][3]-(*this)[1][3];
M(3,3)=(*this)[4][3]-(*this)[1][3];
return(M);
}

/*!
  \brief  Returns the inverse of the 3*3 matrix of the affine application which
  send the reference tetrahedron to the actual tetrahedron.   \n \n
  \return The 3*3 matrix
*/
//================================================================
template <class T> matrix<T> tetrahedre<T>::InvMk()const
//================================================================
{
return(Mk().inv());
}
/*!
  \brief  Returns the number of nodes in a tetrahedron (always 4).   \n \n
  \return 4
*/
//================================================================
template <class T> int tetrahedre<T>::nb_nodes()const
//================================================================
{
return(4);
}

#endif
