/***************************************************************
 *                 Finite Element Method Object Library        *
 *                             tutorial 3                      *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 	0.2.0                  *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2006 CREUSE Emmanuel
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class tuto3
    \brief tuto3 example \n

    \htmlonly 
    <FONT color="#838383">

    tuto3 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.2.0
    \date 2006
    \bug none
    \warning none
*/

#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__STRING_H)
#include <string>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__ASSERT_H)
#include <time.h>
#endif

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATHS_H)
#include "../../MOL++/maths.h"
#endif

#if !defined(__INTEGRATION_H)
#include "../../MOL++/integration.h"
#endif

#if !defined(__CONVERT_H)
#include "../../FEMOL++/meshes/convert.h"
#endif

#if !defined(__MESH_TRI_2D_H)
#include "../../FEMOL++/meshes/mesh_tri_2d.h"
#endif

#if !defined(__TRIANGLE_H)
#include "../../FEMOL++/meshes/triangle.h"
#endif

#if !defined(__AFFICHE_H)
#include "../../tests/affiche.h"
#endif

#if !defined(__P1C2D_H)
#include "../../FEMOL++/elements/p1c2d.h"
#endif

#if !defined(__TETRART0_H)
#include "../../FEMOL++/elements/tetraRT0.h"
#endif

#if !defined(__TETRAP0_H)
#include "../../FEMOL++/elements/tetrap0.h"
#endif

//-------------------------------------------------
// Parameter for reffinement around the singularity
//-------------------------------------------------
long double tau=0.67;

//-------------------------------------
// Nb points for numerical integration
//------------------------------------
int NbIntPts=3;

//------------------------------------------
// Relaxation Parameter for Uzawa Algorithm
//------------------------------------------
long double rho=1e-04;

//----------------------------------
// Stop test for conjugate gradient
//----------------------------------
long double eps1=1e-06;

//------------------------------
// Stop test for Uzawa Algorithm
//------------------------------
long double eps2=1e-04;

//------------------------------------------------
// Frquency (in nb iterations) for error checkout
//------------------------------------------------
int nsauv=50;

//----------------
// Right-Hand-Side
//----------------
long double f(long double x, long double y, long double z)
{
 double r=sqrt(powl(x,2.0)+powl(y,2.0));
   
   if (r<1e-12) 
	{
	  cout << "Be carefull, f undefined for x=0 and y=0 !!!" << "\n";
          assert(r>1e-12);
        }
      else
        {
         double theta=acos(x/r);
         if (y<0) {theta=2*3.14159264-theta;}
          return( 2.0*powl(r,2.0/3.0)*sin(2.0*theta/3.0)*(0.1-r)
		  +7.0*(0.1-z)*z*powl(r,-1.0/3.0)*sin(2*theta/3)/3 );
        }
}

//----------------
// Exact solution
//----------------
long double u(long double x, long double y, long double z)
{
double r=sqrt(powl(x,2.0)+powl(y,2.0));
     if (r<1e-12)
       {
	 return 0;
       }
     else
       {
         double theta=acos(x/r);
         if (y<0) {theta=2.0*3.14159264-theta;}
         return(powl(r,2.0/3.0)*sin(2.0*theta/3.0)*z*(0.1-z)*(0.1-r));
       }
}

//---------------------------
// Gradient of exact solution
//---------------------------
vector<long double> p(long double x, long double y, long double z)
{
  vector<long double> vec(3);
  double r=sqrt(powl(x,2.0)+powl(y,2.0));
     if (r<1e-12) 
	{
	  cout << "attention, p non dfini en x=0 et y=0 !!!" << "\n";
          assert(r>1e-12);
        }
      else
        {
	 double theta=acos(x/r);
         if (y<0) {theta=2.0*3.14159264-theta;}
          vec[1]=cos(theta)*z*(0.1-z)*sin(2.0*theta/3.0)*
                 (2.0*0.1*powl(r,-1.0/3.0)/3.0-5.0*powl(r,2.0/3.0)/3.0)
	         -2.0*sin(theta)*powl(r,-1.0/3.0)*(0.1-r)*z*(0.1-z)
	         *cos(2.0*theta/3.0)/3.0;
          vec[2]=sin(theta)*z*(0.1-z)*sin(2.0*theta/3.0)*
                 (2.0*0.1*powl(r,-1.0/3.0)/3.0-5.0*powl(r,2.0/3.0)/3.0)
	         +2.0*cos(theta)*powl(r,-1.0/3.0)*(0.1-r)*z*(0.1-z)
	         *cos(2.0*theta/3.0)/3.0;
          vec[3]=powl(r,2.0/3.0)*(0.1-r)*sin(2.0*theta/3.0)*(0.1-2*z);
        }    
  return(vec);
}

//--------------------------------------
// Divergence of exact solution gradient
//--------------------------------------
long double  div_p(long double x, long double y, long double z)
{ // So it is -Laplacian !!!
  return(-f(x,y,z));
}
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//==========================
main()
//==========================
{
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";
cout << " BEGINNING tuto3 \n";
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";
cout << endl;

//---------------------------------
// tetraRT0 finite element loading
//---------------------------------
make_convertion3dtetra("../../tests/FEMOL++/data/PartOfCyl2/PartOfCylSmall0.dat","tampon.dat");
tetraRT0<long double> Th("tampon.dat");

//---------------------------------------------
// Nodes positions modification for refinement
//---------------------------------------------
Th.make_anisotropic(tau,4);


//-------------------------------
// tetrap0 finite element loading
//-------------------------------
tetrap0<long double>  Sh("tampon.dat");

//--------------------------------------------------------------------
// Nodes positions modification for refinement
//--------------------------------------------------------------------
Sh.make_anisotropic(tau,4);

//--------------------------------------------------
// Gauss points and weight for numerical integration
//--------------------------------------------------
integration<long double> itg;
itg.Gauss(NbIntPts,0.0,1.0);


int nb_tetra=Th.nbelement();
int nb_faces=Th.nbfaces();
cout << endl;
cout << "Nb DoF for ph= " << nb_faces << endl;
cout << "Nb DoF for uh= " << nb_tetra << endl;

//-----------------------------------
// Descriptor copy in local variables
//-----------------------------------
matrix<int> face_of_element_local(nb_tetra,4);
face_of_element_local=Th.face_of_element();
matrix<int> direction_of_normal_local(nb_tetra,4);
direction_of_normal_local=Th.direction_of_normal();

//-----------------------------------
// Approximate solutions declarations
//-----------------------------------
vector<long double> uh(nb_tetra),uhAnc(nb_tetra);
vector<long double> ph(nb_faces),phAnc(nb_faces);

//------------------------------------
// Constitution de la matrice de masse
//------------------------------------
 spmatrix<long double> A(Th.UV());

//-----------------------------------------
// Computation of the fixed part of the RHS
//------------------------------------------
 vector<long double> B;
 
 int iterations=0;
 int iter_grad_conj=0;
 int nb_iterations;
 vector<long double> Int_f(nb_tetra),determinantdiv6(nb_tetra);

 for (int i=1; i<=nb_tetra; i++)
   {
   Int_f[i]=Sh.Integrate_f(f,i,itg);
   determinantdiv6[i]=Sh[i].AbsDet()/6.0;
   }

//--------------------------------------------------------------------------------   
// Optional part 
// uh is initialized by the projection of the exact solution to increase the
// convergence speed   
//-------------------------------------------------------------------------------- 
  for (int i=1; i<=nb_tetra; i++)
   {
   uh[i]=Sh.Integrate_f(u,i,itg)/(Sh[i].AbsDet()/6.0);
   }
   
//-----------------------------------
// RHS computation and ph computation
//-----------------------------------      
 B=Th.B_PB_MIXTE(uh);
 ph=conjugate_gradient(A,B,phAnc,iter_grad_conj,eps1);

//---------------- 
// Local variables
//---------------- 
  clock_t start, end;
  start = clock();
  long double sumalpha,error_L2_u,error_Hdvi_p;
  
//----------------  
// Loop beginning
//----------------  
   for (; (uh-uhAnc).norm() / uhAnc.norm() > eps2; iterations++) // precision test
   {
    nb_iterations=iterations;
    uhAnc=uh;
    phAnc=ph;
//----------------------------------------
// RHS computation and ph computation
//----------------------------------------     
    B=Th.B_PB_MIXTE(uh);
    ph=conjugate_gradient(A,B,phAnc,iter_grad_conj,eps1);
//---------------
// uh computation
//---------------       
    for (int numele=1; numele<=nb_tetra; numele++)
     {
      sumalpha=0.0;
      for (int i=1; i<=4; i++)
      sumalpha+=ph[face_of_element_local(numele,i)]*direction_of_normal_local(numele,i);
      uh[numele]+=rho*(sumalpha+Int_f[numele])/determinantdiv6[numele];
     }
//------------------
// Error computation
//------------------       
    if (iterations%nsauv==0)
      {
    cout << "------------------------------------------------------------------\n"; 
    cout << "ITERATION= " << iterations;
    cout << " nb iter conjugate gradient= " << iter_grad_conj ;
    cout << " stop test : " << (uh-uhAnc).norm() / uhAnc.norm() << "\n";
    error_L2_u=Sh.error_L2(u,uh,NbIntPts);
    error_Hdvi_p=sqrt(pow(Th.error_L2(p,ph,NbIntPts),2)+pow(Th.error_div(div_p,ph,NbIntPts),2));
    cout << "GLOBAL ERROR = " << error_L2_u+error_Hdvi_p << "\n" ;
    cout << "\n";
      }
   }
//------------   
// End of loop
//------------   
 end = clock();
 cout << "------------------------------------------------------------------\n";
 cout << "------------------------------------------------------------------\n"; 
     cout << "CONVERGENCE OK" << "\n";
     cout<<"CPU TIME : " << (long double)(end-start)/CLOCKS_PER_SEC << endl;
     cout<<"nb iter= " << nb_iterations << "\n";
 cout << "------------------------------------------------------------------\n";
 cout << "------------------------------------------------------------------\n"; 
 
//--------------------------------------
// Error computation when convergence OK
//---------------------------------------      
 cout << "ERROR COMPUTATION...";   
 error_L2_u=Sh.error_L2(u,uh,NbIntPts);
 error_Hdvi_p=sqrt(pow(Th.error_L2(p,ph,NbIntPts),2)+pow(Th.error_div(div_p,ph,NbIntPts),2)); cout << "END : \n";
 cout << "L2(Omega)-norm for u-uh= " << error_L2_u << "\n";
 cout << "H(div,Omega)-norm for p-ph= " << error_Hdvi_p << "\n" ;
 cout << "GLOBAL ERROR= " << error_L2_u+error_Hdvi_p << "\n" ;
 cout << "\n";

 //------------------------- 
 // Destruction des fichiers
 //-------------------------
 remove("tampon.dat");
 
   
cout << endl;
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";
cout << " END tuto3 \n";
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";
}


