/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class triangle : declaration for triangle         *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.1.8	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2002,2003,2004,2005,2006 CREUSE Emmanuel
 * copyright  2002,2003,2004,2005,2006 SOUALEM Nadir
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class triangle
    \brief triangle library \n

    \htmlonly 
    <FONT color="#838383">

    triangle belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2002,2003,2004,2005,2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2002,2003,2004,2005,2006 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.1.8
    \date 2002-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type triangle
#endif

#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../../MOL++/matrix.h"
#endif

#if !defined(__NODE_H)
#include "../meshes/node.h"
#endif

#ifndef _triangle_h
#define _triangle_h

using namespace std;

/*
Let A,B,C 3 nodes of R2, we define
the template class triangle as follows:

       (*)--->&B
       /\
      /  \
     /    \
    /      \ (Triangle<<==>>set of pointers on nodes)
   /        \
  /__________\
(*)--->&C    (*)--->&A

*/

/*!
Let A,B,C 3 nodes of R2, we define
the template class triangle as a set of 3 pointers on these nodes.
*/
//==============================================
template <class T> class triangle
//==============================================
{
  protected :
  node<T>** vertex;// ....................define an array of pointers
  int alive;// ...........................1 if node exists else O
  public:
  triangle();// ..........................default constructor
  ~triangle();//......................... destructor
  triangle(node<T>&,node<T>&,node<T>&);// constructor
  triangle(node<T>* A[3]);// constructor
  triangle(const triangle<T>&);//........ copy constructor
  triangle<T>& operator=(const triangle<T>&);//defines equal operator for triangles
  template <class Tf> friend  ostream& operator << (ostream&, const triangle<Tf>&);//  overloads output stream for triangle
  node<T>& operator [] (int)const;// .....return the i-th node of triangle
  template <class Tf> friend  int operator ==  (const triangle<Tf>&, const triangle<Tf>&); // compares 2 triangles
  T AbsDet()const;//...............................returns the det of the 2*2 matrix of the affine transfo.
  matrix<T> Mk()const;//...........................returns the 2*2 matrix of the affine transfo.
  matrix<T> InvMk()const;//........................returns the inverse of the 2*2 matrix of the affine transfo.
  int nb_nodes() const ; // returns the number of nodes of a triangle
  T diam() const; // returns the diameter of the circonscript circle to the triangle
};


/*!
  \brief Default Constructor 
  \n \n
  alive is set to 0 but the memory is not allocated to store the adresses of the nodes of the triangle in vertex
  */
//==============================================
template <class T> triangle<T>::triangle()
//==============================================
{
  alive=0;
}

/*!
  \brief Destructor
  \n \n
  Deletes vertex and set alive to 0 
*/
//==============================================
template <class T> triangle<T>::~triangle()
//==============================================
{
  if(alive)
      delete [] vertex;
  alive=0;
}

/*!
  \brief Constructor 
  \n \n
  \param A node \n
  \param B node \n
  \param C node \n \n
  alive is set to 1 and vertex is built with pointers towards A,B and C
  */
//========================================================================
template <class T> 
triangle<T>::triangle(node<T>&A,node<T> &B,node<T> &C)
//========================================================================
{
  alive=1;
  vertex=new node<T>*[3];
  vertex[0]=&A;
  vertex[1]=&B;
  vertex[2]=&C;
}


/*!
  \brief Constructor 
  \n \n
  \param A an array containing 3  adresses of nodes \n \n
  alive is set to 1 and vertex is built with the 3 adresses contained in A
  */
//========================================================================
template <class T> 
triangle<T>::triangle(node<T>* A[3])
//========================================================================
// The entry arguments are the adresses of the nodes, but not the nodes themselves.
{
  alive=1;
  vertex=new node<T>*[3];
  vertex[0]=A[0];
  vertex[1]=A[1];
  vertex[2]=A[2];
}

/*!
  \brief Copy constructor
  \param K triangle to duplicate
*/
//========================================================================
template <class T> triangle<T>::triangle(const triangle<T>& K)
//========================================================================
{
  vertex=new node<T>*[3];
  alive=K.alive;
  vertex[0]=K.vertex[0];
  vertex[1]=K.vertex[1];
  vertex[2]=K.vertex[2];
}

/*!
  \brief Standard operator = for triangle \n \n
  \param K triangle to copy \n
  \return reference of the left hand size triangle (for multiple equalities)
  \n\n
*/
//=========================================================================
template <class T> triangle<T>& triangle<T>::operator=(const triangle<T>& K)
//=========================================================================
{
  assert(K.alive);
  if(!alive)
      vertex=new node<T>*[3];
  alive=K.alive;
  vertex[0]=K.vertex[0];
  vertex[1]=K.vertex[1];
  vertex[2]=K.vertex[2];
  return (*this);
}

/*!
  \brief  overloads output stream for triangle \n \n
  \param exit output flow \n
  \param K triangle
*/
//==========================================================================
template <class Tf> ostream& operator <<(ostream& exit, const triangle<Tf>& K)
//==========================================================================
{
  if(!K.alive)
      exit<<"undefined triangle";
  else
      {
      exit<< "triangle:"<<endl;
      exit<< *(K.vertex[0]) << *(K.vertex[1]) <<*(K.vertex[2]);
      }
  return exit;
}

/*!
  \brief Returns one node of the triangle \n \n
  Warning : we use the mathematical notations so the nodes adresses are in the set [1;3], and not [0:2] \n \n
  \param i node number \n \n
  \return the node i of the triangle \n \n
*/
//================================================================
template <class T> node<T>& triangle<T>::operator [] (int i)const
//================================================================
{
  assert((i>0)&&(i<=3)&&alive);
  return *(vertex[i-1]);
}
/*!
  \brief  Check the equality of two triangles \n \n
  \param K triangle \n
  \param L triangle \n
  \return 1 if equal, 0 if not
*/
//================================================================
template <class Tf> int operator ==(const triangle<Tf> &K,const triangle<Tf> &L)
//================================================================
{
  int boolean=1;
  boolean*=((K[1]==L[1])&&(K[2]==L[2])&&(K[3]==L[3])&&(K.alive==L.alive));
  return(boolean);
}

/*!
  \brief  Returns the absolute value of the matrix of the affine application which
  send the reference triangle to the actual triangle. So, it is twice the area of the triangle  \n \n
  \return The value of the absolute value of the determinant
*/
//================================================================
template <class T> T triangle<T>::AbsDet()const
//================================================================
{
vector<T> A1A2(2), A1A3(2);
A1A2=(*this)[2]-(*this)[1];
A1A3=(*this)[3]-(*this)[1];
return abs(A1A2[1]*A1A3[2]-A1A2[2]*A1A3[1]);
}

/*!
  \brief  Returns the 2*2 matrix of the affine application which
  send the reference triangle to the actual triangle.   \n \n
  \return The 2*2 matrix
*/
//================================================================
template <class T> matrix<T> triangle<T>::Mk()const
//================================================================
{
matrix<T> M(2,2);
M(1,1)=(*this)[2][1]-(*this)[1][1]; M(1,2)=(*this)[3][1]-(*this)[1][1];
M(2,1)=(*this)[2][2]-(*this)[1][2]; M(2,2)=(*this)[3][2]-(*this)[1][2];
return(M);
}


/*!
  \brief  Returns the inverse of the 2*2 matrix of the affine application which
  send the reference triangle to the actual triangle.   \n \n
  \return The 2*2 matrix
*/
//================================================================
template <class T> matrix<T> triangle<T>::InvMk()const
//================================================================
{
return(Mk().inv());
}

/*!
  \brief  Returns the number of nodes in a triangle (always 3).   \n \n
  \return 3
*/
//================================================================
template <class T> int triangle<T>::nb_nodes()const
//================================================================
{
return(3);
}


/*!
  \brief  returns the diameter of the circonscript circle to the triangle \n \n
  \return the diameter
*/
//================================================================
template <class T> T triangle<T>::diam()const
//================================================================
{
T xA,yA,xB,yB,xC,yC;
xA=(*this)[1][1];
xB=(*this)[2][1];
xC=(*this)[3][1];
yA=(*this)[1][2];
yB=(*this)[2][2];
yC=(*this)[3][2];

T alpha=0.5*((xB-xC)*(xB-xA)+(yB-yC)*(yB-yA))/
            ((yA-yC)*(xB-xA)+(xC-xA)*(yB-yA));
return(2*sqrt(
             pow(0.5*(xC-xA)+alpha*(-yC+yA),2)
            +pow(0.5*(yC-yA)+alpha*(xC-xA),2)
	      )
       );
}



#endif
