/***************************************************************
 *             Finite Element Method Object Library            *
 *          class triangle : declaration for rectangle         *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.6.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2004 CREUSE Emmanuel
 * copyright © 2004 KALETA Olivia
 * copyright © 2012 COLLARD Christophe
 * copyright © 2004, 2012 Centre National de la Recherche Scientifique
 * copyright © 2004 Arts et Métiers ParisTech
 * copyright © 2004 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2004 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2004 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \class rectangle
    \brief rectangle library \n

    \htmlonly 
    <FONT color="#838383">

    triangle belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly   2004 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly   2004 KALETA Olivia \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2004, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 2.6.0
    \date 2004-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error must use C++ for the type rectangle
#endif

#ifndef _rectangle_h
#define _rectangle_h


#if !defined(__IOSTREAM_H)
#include<iostream>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__NODE_H)
#include "FEMOL++/meshes/node.h"
#endif

/*
let A,B,C,D 4 nodes of R2, we define
the template class rectangle as follows:



  &D <---(*)_______________________(*)---->&C
           |                       |
           |                       |
           |                       |               (Rectangle<<-->>set of pointers on nodes)
           |                       |
           | ______________________/
   &A<---(*)                      (*)--->&B
       

*/       

using namespace std;
using namespace mol;

namespace femol
{


//================================
template <class T> class rectangle
//================================
{
  protected:
    node<T>** vertex; // define an array of pointers
    int alive; // return 1 if rectangle exists 0 otherwise

  public:
    rectangle (); // default constuctor
    rectangle (node<T>&, node<T>&, node<T>&, node<T>&);
    rectangle (node<T>* A[4]);
    rectangle (const rectangle<T>&); // copy constructor
    ~rectangle (); // destructor

    rectangle<T>& operator = (const rectangle<T>&);
    template <class Tf> friend ostream& operator << (ostream&, const rectangle<Tf>&);
    node<T>& operator [] (int) const; // return the i-th node of rectangle
    template <class Tf> friend bool operator == (const rectangle<Tf>&, const rectangle<Tf>&);
    T AbsDet () const; // return the det of the affine transfo.
    matrix<T> Mk () const;
    matrix<T> InvMk () const;
    int nb_nodes () const; // returns the number of nodes of a rectangle
};


//=====Private methods for rectangle========================================


//=====Public  methods for rectangle========================================


/*!
  \brief default constructor of the rectangle class
*/

//===========================================
template <class T> rectangle<T>::rectangle ()
//===========================================
{
  alive = 0;
}


/*!
  \brief constructor of the rectangle class.the arguments are 4 points. A, B, C, D are the nodes of the rectangle.
  \param A node
  \param B node
  \param C node
  \param D node
*/

//=========================================================================================
template <class T> rectangle<T>::rectangle (node<T>& A, node<T>& B, node<T>& C, node<T>& D)
//=========================================================================================
{
  alive = 1;
  vertex = new node<T>* [4];
  vertex[0] = &A;
  vertex[1] = &B;
  vertex[2] = &C;
  vertex[3] = &D;
}


/*!
  \brief constructor of the rectangle class. The arguments are table of node.
  \param A table of  four nodes
*/

//=======================================================
template<class T> rectangle<T>::rectangle (node<T>* A[4])
//=======================================================
{
  alive = 1;
  vertex = new node<T>* [4];
  vertex[0] = A[0];
  vertex[1] = A[1];
  vertex[2] = A[2];
  vertex[3] = A[3];
}


/*!
  \brief copy constructor of the rectangle class
  \param K type const rectangle<T>
*/

//================================================================
template <class T> rectangle<T>::rectangle (const rectangle<T>& K)
//================================================================
{
  vertex = new node<T>* [4];
  alive = K.alive;
  vertex[0] = K.vertex[0];
  vertex[1] = K.vertex[1];
  vertex[2] = K.vertex[2];
  vertex[3] = K.vertex[3];
}


/*!
  \brief destructor of the rectangle class
*/

//============================================
template <class T> rectangle<T>::~rectangle ()
//============================================
{
  if (alive)
    delete [] vertex;
  alive = 0;
}


/*!
  \brief affectation operator for the rectangle class
  \param K type const rectangle<T>
  \return reference on rectangle object
*/

//===============================================================================
template <class T> rectangle<T>& rectangle<T>::operator = (const rectangle<T>& K)
//===============================================================================
{
  assert (K.alive);
  if (!alive)
    vertex = new node<T>* [4];
  alive = K.alive;
  vertex[0] = K.vertex[0];
  vertex[1] = K.vertex[1];
  vertex[2] = K.vertex[2];
  vertex[3] = K.vertex[3];

  return (*this);
}


/*!
  \brief output operator for the rectangle class
  \param K type const rectangle<T>
  \param s type ostream
  \return reference on ostream
*/

//=========================================================================
template <class T> ostream& operator << (ostream& s, const rectangle<T>& K)
//=========================================================================
{
  if (!K.alive)
    s << "undefined rectangle";
  else 
    { s << "rectangle:" << endl;
      s << *(K.vertex[0]) << *(K.vertex[1]) << *(K.vertex[2]) << *(K.vertex[3]);
    }

  return s;
}


/*!
  \brief hook operator for the rectangle class
  \param i int
  \return reference on node object
*/

//=================================================================
template <class T> node<T>& rectangle<T>::operator [] (int i) const
//=================================================================
{
  assert ((i > 0) && (i <= 4) && alive);
  return *(vertex[i-1]);
}


/*!
  \brief comparaison operator for the rectangle class
  \param K type const rectangle<T>&
  \param L type const rectangle<T>&
  \return 1 if  K and L are equal, 0 otherwise
*/

//================================================================================
template <class T> bool operator == (const rectangle<T> &K, const rectangle<T> &L)
//================================================================================
{
  bool boolean = 1;
  boolean *= ((K[1] == L[1]) && (K[2] == L[2]) && (K[3] == L[3]) && (K[4] == L[4]) && (K.alive == L.alive));

  return boolean;
}


/*!
  \return the matrix of the affine transformation from the reference element to the actual one
*/

//====================================================
template <class T> matrix<T> rectangle<T>::Mk () const
//====================================================
{
  matrix<T> M(2,2);
  M(1,1) = (*this)[2][1] - (*this)[1][1];
  M(1,2) = (*this)[3][1] - (*this)[2][1];
  M(2,1) = (*this)[2][2] - (*this)[1][2];
  M(2,2) = (*this)[3][2] - (*this)[2][2];

  return M;
}


/*!
  \return the inverse of the matrix of the affine transformation from the reference element to the actual one
*/

//=======================================================
template <class T> matrix<T> rectangle<T>::InvMk () const
//=======================================================
{
  return Mk().inv();
}


/*!
  \brief declaration of absolute value of the det of the affine transformation
  \return the absolute value of the det of the affine transformation
*/

//================================================
template <class T> T rectangle<T>::AbsDet () const
//================================================
{
  vector<T> A1A2 = (*this)[2] - (*this)[1];
  vector<T> A2A3 = (*this)[3] - (*this)[1];

  return abs (A1A2[1] * A2A3[2] - A1A2[2] * A2A3[1]);
}


/*!
  \return the number of rectangle's nodes: four here
*/

//====================================================
template <class T> int rectangle<T>::nb_nodes () const
//====================================================
{
  return 4;
}


}


#endif
