/***************************************************************
 *             Finite Element Method Object Library            *
 *             class rt02d : declaration for rt02d             *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.0.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2005,2006 CREUSE Emmanuel
 * copyright © 2012 COLLARD Christophe
 * copyright © 2005,2006,2012 Centre National de la Recherche Scientifique
 * copyright © 2005,2006 Arts et Métiers ParisTech
 * copyright © 2005,2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2005,2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - UMR 7554)
 * copyright © 2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace femol
    \brief Finite Element Method Object Libraries
*/

/*! \class femol::rt02d
    \brief rt02d library \n

    \htmlonly 
    <FONT color="#838383">

    rt02d belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2005 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2005, 2006, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2005, 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2005, 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2005-2012
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type rt02d
#endif

#ifndef __rt02d_hpp
#define __rt02d_hpp

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __spmatrix_hpp
#include "MOL++/spmatrix.hpp"
#endif

#ifndef __integration_hpp
#include "MOL++/integration.hpp"
#endif

#ifndef __mesh_tri_2d_hpp
#include "FEMOL++/meshes/mesh_tri_2d.hpp"
#endif

using namespace std;
using namespace mol;

namespace femol
{


//=====================================================
template <class T> class rt02d : public mesh_tri_2d <T>
//=====================================================
{
  using mesh<triangle<T>,T>::alive;
  using mesh<triangle<T>,T>::dimspace_;
  using mesh<triangle<T>,T>::nbvertex_;
  using mesh<triangle<T>,T>::nbelement_;
  using mesh<triangle<T>,T>::vertices;
  using mesh<triangle<T>,T>::element;

  using mesh_tri_2d<T>::nbsegments_;
  using mesh_tri_2d<T>::SetOfEdges_;
  using mesh_tri_2d<T>::Type_of_edge_;
  using mesh_tri_2d<T>::EdgesPerTriangle_;
  using mesh_tri_2d<T>::NbTrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerTriangle_;
  using mesh_tri_2d<T>::NumberOfNeighboorOfTriangles_;
  using mesh_tri_2d<T>::NumLocNodeInTriangle_;
  using mesh_tri_2d<T>:: NormalDirection_;
  using mesh_tri_2d<T>::hminT_;
  using mesh_tri_2d<T>::hE_;
  using mesh_tri_2d<T>::hminE_;
  using mesh_tri_2d<T>::NormEdge_;
  using mesh_tri_2d<T>::NormalUnitVector_;
  using mesh_tri_2d<T>::NbMaxSegmentsPerNode_;
  using mesh_tri_2d<T>::NbSegmentsPerNode_;

  private:
    int DofNumber_;
    int GlobalDofNumber_;

  public:
    rt02d ();
    rt02d (const char*, const char* = "");
    rt02d(const rt02d<T>&);

    rt02d<T>& operator = (const rt02d<T>&);
    template <class Tf> friend bool operator == (const rt02d<Tf>&, const rt02d<Tf>&);
    template <class Tf> friend bool operator != (const rt02d<Tf>&, const rt02d<Tf>&);
    int operator () (int, int) const;
    int DofNumber () const {return DofNumber_;}
    int GlobalDofNumber () const {return GlobalDofNumber_;}
    matrix<T> RefBasisFunction (T, T) const;
    vector<T> Fk (T, T, int) const;
    matrix<T> Ak (int, integration<T>, const vector<T>&, T) const;
    matrix<T> Bk (int) const;
    matrix<T> Ck (int) const;
    T int_K_f (T (*f)(T,T), int, integration<T> itg) const;
    template <template <class> class matrix_type> void Assemble (matrix_type<T>&, vector<T>&, T (*f)(T,T), const vector<T>&, T, int);
    template <template <class> class matrix_type> void Dirichlet (matrix_type<T>&, vector<T>&, T (*u)(T,T), T);
    void Compute_u (vector<T>&, const vector<T>&, T (*f)(T,T), const vector<T>&, T, int);
    void Compute_p (vector<T>&, const vector<T>&, const vector<T>&, const vector<T>&, T, int);
    T error_p (const vector<T>&, T (*px)(T,T), T (*py)(T,T), T, int);
    T error_u (const vector<T>&, T, T (*u)(T,T), int);
    T error_u_loc2 (const vector<T>&, T, T (*u)(T,T), int, int);
    T error_loc_u (T, T (*u)(T,T), int, int);
    T eta_11_K_p (const vector<T>&, T, int, int);
    T eta_13_K_p (const vector<T>&, T, int);
    T eta_1 (const vector<T>&, T, int);
    T eta_2_K_q (T (*f)(T,T), T, int, int);
    T eta_2 (T (*f)(T,T), T, int);
    void refine_FE (vector<int>&); // refine the Finite Element
    void refine_FE_angle (vector<int>&, T angle); // refine the Finite Element
    matrix<T> LocalMassMatrix (int) const;
    spmatrix<T> UV () const;
    T error_div (T (*divp)(T,T), const vector<T>&, int) const;
    T error_L2 (const vector<T>&, T (*px)(T,T), T (*py)(T,T), int);
    T error_L2_u (const vector<T>&, T (*u)(T,T), int);
    T error_L2_loc2 (const vector<T>&, T (*px)(T,T), T (*py)(T,T), int, int);
};


//=====Private methods for rt02d=============================================


//=====Public methods for rt02d==============================================


/*!
  \brief default constructor of the rt02d class
*/

//=====================================
template <class T>
rt02d<T>::rt02d () : mesh_tri_2d <T> ()
//=====================================
{
  DofNumber_ = GlobalDofNumber_ = 0;
}


/*!
  \brief constructor of the rt02d class. The arguments are 2 files file1 and file2. file1 must be documented by the number of elements, node 's numerotation of each element. file2 indicate the dimension (2 or 3), number of nodes and each node coordinates with a boundary condition (1 if node is on the boundary, 0 otherwise).
  \param file1 string
  \param file2 string (if not merged with file1)
*/

//====================================================================================
template <class T>
rt02d<T>::rt02d (const char* file1, const char* file2) : mesh_tri_2d<T> (file1, file2)
//====================================================================================
{
  DofNumber_ = 3;
  GlobalDofNumber_ = nbsegments_;
}


/*!
  \brief copy constructor of the rt02d class
  \param EF  reference on rt02d object
*/

//========================================================
template <class T>
rt02d<T>::rt02d (const rt02d<T>& EF) : mesh_tri_2d<T> (EF)
//========================================================
{
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
}


/*!
  \brief affectation operator for the rt02d class.
  \param EF reference on rt02d object
  \return  reference on rt02d object
*/

//=================================================
template <class T>
rt02d<T>& rt02d<T>::operator = (const rt02d<T>& EF)
//=================================================
{
  (*this).mesh_tri_2d<T>::operator = (EF);
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;

  return (*this);
}


/*!
  \param xref type T
  \param yref type T
  \return matrix \f$(\vec{\varphi_1}(xref,yref),\vec{\varphi_2}(xref,yref),\vec{\varphi_3}(xref,yref))\f$ where \f$(\vec{\varphi_i)}_{1\leq i\leq 3}\f$ is the i-th basis function.
*/

//=========================================================
template <class T>
matrix<T> rt02d<T>::RefBasisFunction (T xref, T yref) const
//=========================================================
{
  matrix<T> P (2, DofNumber_);
  P(1,1) = xref;
  P(1,2) = xref - 1;
  P(1,3) = xref;
  P(2,1) = yref;
  P(2,2) = yref;
  P(2,3) = yref - 1;

  return P;
}


/*!
  \param num_element element's number
  \param j the j-th dof
  \return global number of the j-th of the element num_element
*/

//======================================================
template <class T>
int rt02d<T>::operator () (int num_element, int j) const
//======================================================
{
  return EdgesPerTriangle_ (num_element, j);
}


/*!
  \param EF1 type const rt02d<T>&
  \param EF2 type const rt02d<T>&
  \return 1 if EF1 and  EF2 are equal, 0 otherwise
*/

//===========================================================
template <class Tf>
bool operator == (const rt02d<Tf>& EF1, const rt02d<Tf>& EF2)
//===========================================================
{
  bool boolean = true;
  boolean *= (mesh_tri_2d<Tf> (EF1) == mesh_tri_2d<Tf> (EF2));
  boolean *= (EF1.DofNumber_ == EF2.DofNumber_);
  boolean *= (EF1.GlobalDofNumber_ == EF2.GlobalDofNumber_);

  return boolean;
}


/*!
  \param EF1 type const rt02d<T>&
  \param EF2 type const rt02d<T>&
  \return 1 if EF1 and  EF2 are different, 0 otherwise
*/

//===========================================================
template <class Tf>
bool operator != (const rt02d<Tf>& EF1, const rt02d<Tf>& EF2)
//===========================================================
{
  return !(EF1 == EF2);
}


/*!
  \param num_element type int
  \param itg type integration<T>
  \param P type vector<T>& (contains the ddl associated to each segment of the mesh, representing the value of \f$ \vec{p}=|\nabla u|^{p-2} \vec{\nabla u} \f$).)
  \param q type T
  \return matrix \f$ \int_{K_{num_{element}}} |\vec{p}|^{q-2} \vec{\phi}_j^{K_{num_{element}}}.\vec{\phi}_i^{K_{num_{element}}} d \Omega , \quad 1 \leq i,j \leq 3\f$.
*/

//=========================================================================================
template <class T>
matrix<T> rt02d<T>::Ak (int num_element, integration<T> itg, const vector<T>& P, T q) const
//=========================================================================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  T omegak, omegal, xk, xl;
  T sumx, sumy;

  matrix<T> A (3,3);
  matrix<T> Mk (element[num_element-1].Mk());

  matrix<T> NormalDir (3,3);
  for (int i=1; i<=3; i++)
    NormalDir(i,i) = (T) NormalDirection_(num_element,i);

  matrix<T> LocalBasisFunction (2,3);

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  LocalBasisFunction = Mk * RefBasisFunction (xk,xl) * NormalDir;
	  sumx = sumy = 0;
	  for (int i=1; i<=3; i++)
	    { sumx += P[(*this)(num_element,i)] * LocalBasisFunction(1,i);
	      sumy += P[(*this)(num_element,i)]*LocalBasisFunction(2,i);
	    }
	  A += omegak * omegal * (1-xk) * t(LocalBasisFunction) * LocalBasisFunction * (T) powf (sumx * sumx + sumy * sumy, (q-2) * 0.5);
	}
    }

  return A / (T) pow (element[num_element-1].AbsDet(), q-1);
}


//===========================================================================================================
template <class T>
template <template <class> class matrix_type>
void rt02d<T>::Assemble (matrix_type<T>& A, vector<T>& B, T (*f)(T,T), const vector<T>& P, T q, int NbIntPts)
//===========================================================================================================
{
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  matrix<T> LocalA (3,3);
  matrix<T> invAk (3,3), invAkmod (3,3), matloc (3,3);
  matrix<T> FullOne (3,3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      FullOne (i,j) = 1;
  T int_f_K, sum;
  vector<T> sum_line (3);
  matrix<T> vecBk (3,1);
  matrix<T> tvecBk (1,3);
  matrix<T> coeff (1,1);

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { invAk = gauss (Ak (num_element, itg, P, q)); // inverse of Ak
      vecBk = Bk (num_element);
      tvecBk = t(vecBk);
      coeff = tvecBk * invAk * vecBk;
      int_f_K = int_K_f (f, num_element, itg);

      for (int i=1; i<=3; i++)
	for (int j=1; j<=3; j++)
	  invAkmod(i,j) = invAk(i,j) * (T) NormalDirection_(num_element,i) * (T) NormalDirection_(num_element,j);

      matloc = invAk * vecBk * tvecBk * invAk;
      for (int i=1; i<=3; i++)
	for (int j=1; j<=3; j++)
	  LocalA(i,j) = invAkmod(i,j) - (T) NormalDirection_(num_element,i) * matloc(i,j) * (T) NormalDirection_(num_element,j) / coeff(1,1);

      for (int i=1; i<=3; i++)
	{ B[(*this)(num_element,i)] += (T) NormalDirection_(num_element,i) * (invAk * vecBk)(i,1) * int_f_K / coeff(1,1);
	  for (int j=1; j<=3; j++)
	    A ((*this)(num_element,i), (*this)(num_element,j)) += LocalA (i,j);
	}
    }
}


/*!
  \param A type matrix_creuse<T>
  \param B type vector<T>
  \param dudx type (*f)(T,T)
  \param dudy type (*f)(T,T)
  \param p type T
  \return the matrix \f$ A \f$ and the vector \f$ B \f$ in which the degrees of freedom associated to boundary segments are set to zero. In this case, the parameters \f$(*dudx)\f$ , \f$(*dudy)\f$ and \f$p\f$ are not needed. This function can be modified to specify non homogeneous boundary conditions (the lines of the code are put into //), and in this case we can choose between rectangle or simpson numerical integration.
*/

//=========================================================================
template <class T>
template <template <class> class matrix_type>
void rt02d<T>::Dirichlet (matrix_type<T>& A, vector<T>& B, T(*u)(T,T), T p) 
//=========================================================================
{
  vector<T> VECM(2), VECA(2), VECB(2);
  T uM;
  T valeur_M;
  T valeur_ddl;
  for (int num_edge=1; num_edge<=GlobalDofNumber_; num_edge++)
    if (Type_of_edge_[num_edge])
      { VECA = vertices [SetOfEdges_(num_edge,1) - 1];
	VECB = vertices [SetOfEdges_(num_edge,2) - 1];
	VECM = (VECA + VECB) * (T) 0.5;
	uM = u (VECM[1], VECM[2]);
	valeur_ddl = uM;
	B[num_edge] = valeur_ddl;
	for (int j=1; j<=GlobalDofNumber_; j++)
	  A (num_edge, j) = 0;
	A (num_edge, num_edge) = 1;
	for (int k=1; k<=GlobalDofNumber_; k++)
	  if (k != num_edge)
	    { B[k] += - A (k, num_edge) * B[num_edge];
	      A (k, num_edge) = 0;
	    }
      }
}


/*!
  \param f type (*f)(T,T)
  \param num_element type int
  \param itg type integration<T>
  \return the scalar T \f$ \int_{K_{num_{element}}} f(x,y) \, d \Omega \f$.
*/

//==========================================================================
template <class T>
T rt02d<T>::int_K_f (T (*f)(T,T), int num_element, integration<T> itg) const
//==========================================================================
{
  T res = 0;
  T omegak, omegal, xk, xl;
  vector<T> Fxkl;

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Fxkl = Fk (xk, xl, num_element);
	  res += omegak * omegal * (1-xk) * f (Fxkl[1], Fxkl[2]);
	}
    }
  res *= element[num_element-1].AbsDet();

  return res;
}


/*!
  \param num_element type int
  \return the matrix<T> (in fact the vector of size 3) \f$ Bk[i]=|K|_{num_{element}} div \; \vec{\Phi_i}^{K_{num_{element}}}, \quad 1 \leq i \leq 3 \f$.
*/

//============================================
template <class T>
matrix<T> rt02d<T>::Bk (int num_element) const
//============================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  matrix<T> u (3,1);
  for (int i=1; i<=3; i++)
    u(i,1) = NormalDirection_(num_element, i);

  return u;
}


/*!
  \param num_element type int
  \return the matrix<T> of size \f$3 \times 3\f$ \f$ Ck(i,j)=\int_{e_j}\vec{\Phi_i}^{K_{num_{element}}}.\, \vec{n_{e_j}}^{K_{num_{element}}} \, d \Omega,  \quad 1 \leq i,j \leq 3 \f$, where \f$e_j\f$ is the jth segment of the triangle \f$K_{num_{element}}\f$, opposed to the jth node.
*/

//============================================
template <class T>
matrix<T> rt02d<T>::Ck (int num_element) const
//============================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  matrix<T> M (3,3);
  for (int i=1; i<=3; i++)
    M(i,i) = NormalDirection_(num_element, i);

  return M;
}


/*!
  \param x type T
  \param y type T
  \param i type integer
  \return vector \f$F_k(x,y,i)\f$
*/

//============================================
template <class T>
vector<T> rt02d<T>::Fk (T x, T y, int i) const
//============================================
{
  matrix<T> mat (2,2);
  vector<T> vec (2);
  vec[1] = x;
  vec[2] = y;

  vector<T> A1A2 = element[i-1][2] - element[i-1][1];
  vector<T> A1A3 = element[i-1][3] - element[i-1][1];
  mat(1,1) = A1A2[1];
  mat(1,2) = A1A3[1];
  mat(2,1) = A1A2[2];
  mat(2,2) = A1A3[2];

  return mat * vec + element[i-1][1];
}


//==================================================================================================================
template <class T>
void rt02d<T>::Compute_u (vector<T>& U, const vector<T>& lambda, T (*f)(T,T), const vector<T>& P, T q, int NbIntPts)
//==================================================================================================================
{
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  matrix<T> invAk (3,3);
  vector<T> sum_col (3);
  matrix<T> vecBk (3,1);
  matrix<T> tvecBk (1,3);
  matrix<T> coeff (1,1);
  matrix<T> C(3,3), line (1,3);
  T sum;

  for (int i=1; i<=nbelement_; i++)
    U[i] = 0;
  for (int num_element=1; num_element<=nbelement_; num_element++)
    { invAk = gauss (Ak (num_element, itg, P, q)); // inverse of Ak
      vecBk = Bk (num_element);
      tvecBk = t(vecBk);
      coeff = tvecBk * invAk * vecBk;
      for (int i=1; i<=3; i++)
	C(i,i) = NormalDirection_(num_element, i);
      line = tvecBk * invAk * C;

      for (int j=1; j<=3; j++)
	U[num_element] += line (1,j) * lambda[(*this)(num_element, j)];
      U[num_element] += int_K_f (f, num_element, itg);
      U[num_element] /= coeff (1,1);
    }
}


//===========================================================================================================================
template <class T>
void rt02d<T>::Compute_p (vector<T>& P, const vector<T>& U, const vector<T>& lambda, const vector<T>& Pm1, T q, int NbIntPts)
//===========================================================================================================================
{
  // Here for the internal segment, the value of P associate to the segment is computed twice :
  // one time for each element containing the segment. It is nevertheless not necessary !
  matrix<T> invAk (3,3);
  vector<T> vec(3), res(3);

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { invAk = gauss (Ak (num_element, itg, Pm1, q)); // inverse of Ak
      for (int i=1; i<=3; i++)
	vec[i] = NormalDirection_(num_element, i) * (lambda[(*this)(num_element,i)] - U[num_element]);
      res = invAk * vec;
      for (int i=1; i<=3; i++)
	P[(*this)(num_element, i)] = res[i];
    }
}


/*!
  \param P type vector<T>& (contains the ddl associated to each segment of the mesh, representing the value of \f$ \vec{p}=|\nabla u|^{p-2} \vec{\nabla u} \f$).
  \param px type (*f)(T,T)
  \param py type (*f)(T,T)
  \param q type T
  \param NbIntPts type int
  \return \f$ |P-pexact|_{0,q,\Omega} \f$
*/

//=====================================================================================
template <class T>
T rt02d<T>::error_p (const vector<T>& P, T (*px)(T,T), T (*py)(T,T), T q, int NbIntPts)
//=====================================================================================
{
  T omegak, omegal, xk, xl;
  T valeur_px, valeur_py;
  vector<T> Fxkl(2), vec(2);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  matrix<T> Mk(2,2), BasisFunction(2,3);

  T error = 0;
  T error_loc;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { error_loc = 0;
      Mk = element[num_element-1].Mk();
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1-xk) * itg.point (l);
	      Fxkl = Fk (xk, xl, num_element);
	      BasisFunction = Mk * RefBasisFunction (xk, xl);
	      vec[1] = vec[2] = 0;
	      for (int j=1; j<=3; j++)
		{ vec[1] += P[(*this)(num_element,j)] * BasisFunction(1,j) * NormalDirection_(num_element,j);
		  vec[2] += P[(*this)(num_element,j)] * BasisFunction(2,j) * NormalDirection_(num_element,j);
		}
	      vec /= element[num_element-1].AbsDet();
//	  if (NormGradu(Fxkl[1],Fxkl[2])<epsilon)
//	   {
//	    valeur_px=0;
//	    valeur_py=0;
//	    vec[1]=0;
//	    vec[2]=0;
//	   }
//	  else
//	   {  
	      valeur_px = px (Fxkl[1], Fxkl[2]);
	      valeur_py = py (Fxkl[1], Fxkl[2]);
//	   }
	      error_loc += omegak * omegal * (1-xk) * pow (sqrt (pow (valeur_px - vec[1], 2) + pow (valeur_py - vec[2], 2)), q);
	    }
	}
      error_loc *= element[num_element-1].AbsDet();
      error += error_loc;
    }

  return pow (error, 1./q);
}


/*!
  \param US type vector<T>& (contains the value of \f$u\f$ on each triangle of the mesh)
  \param u type (*f)(T,T)
  \param NbIntPts type int
  \return \f$ |US-uexact|_{0,2,\Omega} \f$
*/

//==========================================================================================
template <class T> T rt02d<T>::error_u (const vector<T>& US, T s, T (*u)(T,T), int NbIntPts)
//==========================================================================================
{
  T omegak, omegal, xk, xl;
  vector<T> Fxkl (2);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T error = 0;
  T error_loc;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { error_loc = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1 - xk) * itg.point (l);
	      Fxkl = Fk (xk, xl, num_element);
	      error_loc += omegak * omegal * (1-xk) * pow (abs (u (Fxkl[1], Fxkl[2]) - US[num_element]), s);
	    }
	}
      error_loc *= element[num_element-1].AbsDet();
      error += error_loc;
    }

  return pow (error, 1./s);
}


//=============================================================================================
template <class T>
T rt02d<T>::error_u_loc2 (const vector<T>& US, T s, T (*u)(T,T), int NbIntPts, int num_element)
//=============================================================================================
{
  T omegak, omegal, xk, xl;
  T error_loc = 0;
  vector<T> Fxkl (2);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Fxkl = Fk(xk, xl, num_element);
	  error_loc += omegak * omegal * (1-xk) * pow (abs (u (Fxkl[1], Fxkl[2]) - US[num_element]), s);
	}
    }
  error_loc *= element[num_element-1].AbsDet();

  return error_loc;
}


/*!
  \param Umaille type vector<T>& (contains the value of \f$u\f$ on each triangle of the mesh)
  \param u type (*f)(T,T)
  \param NbIntPts type int
  \param num_element type int
  \return \f$ |Umaille-uexact|_{0,2,K_{num_{element}}} \f$
*/

//=============================================================================
template <class T>
T rt02d<T>::error_loc_u (T Umaille, T (*u)(T,T), int NbIntPts, int num_element)
//=============================================================================
{
  T omegak, omegal, xk, xl;
  vector<T> Fxkl (2);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T error_loc = 0;

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Fxkl = Fk (xk, xl, num_element);
	  error_loc += omegak * omegal * (1-xk) * pow (u (Fxkl[1], Fxkl[2]) - Umaille, 2);
	}
    }
  error_loc *= element[num_element-1].AbsDet();

  return sqrt (error_loc);
}


/*!
  \param TABTRI type vector<int>& (the size of TABTRI is the number of triangle of the mesh. TABTRI[i]=-1 if triangle i must be refined, else TABTRI[i]=0)
  \return the finite element, with the new mesh
*/

//============================================
template <class T>
void rt02d<T>::refine_FE (vector<int>& TABTRI)
//============================================
{
  (*this).refine_mesh (TABTRI);
  GlobalDofNumber_ = nbsegments_;
}


/*!
  \param TABTRI type vector<int>& (the size of TABTRI is the number of triangle of the mesh. TABTRI[i]=-1 if triangle i must be refined, else TABTRI[i]=0)
  \return the finite element, with the new mesh (and angle restriction)
*/

//============================================================
template <class T>
void rt02d<T>::refine_FE_angle (vector<int> & TABTRI, T angle)
//============================================================
{
  (*this).refine_mesh_angle (TABTRI, angle);
  GlobalDofNumber_ = nbsegments_;
}


//==========================================================================
template <class T>
T rt02d<T>::eta_11_K_p (const vector<T>& P, T s, int NbPts, int num_element)
//==========================================================================
{
  T hK = hminT_[num_element];
  T omegak, omegal, xk, xl;
  vector<T> Fxkl(2), vec(2);
  integration<T> itg;
  itg.Gauss (NbPts, 0, 1);
  matrix<T> Mk(2,2), BasisFunction(2,3);
  T value_loc = 0;
  Mk = element[num_element-1].Mk();

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1-xk) * itg.point (l);
	  Fxkl = Fk (xk, xl, num_element);
	  BasisFunction = Mk * RefBasisFunction (xk, xl);
	  vec[1] = vec[2] = 0;
	  for (int j=1; j<=3; j++)
	    { vec[1] += P[(*this)(num_element,j)] * BasisFunction(1,j) * NormalDirection_(num_element,j);
	      vec[2] += P[(*this)(num_element,j)] * BasisFunction(2,j) * NormalDirection_(num_element,j);
	    }
	  vec /= element[num_element-1].AbsDet();
	  value_loc += omegak * omegal * (1-xk) * pow (sqrt (pow (vec[1], 2) + pow (vec[2], 2)), s/(s-1));
	}
    }
  value_loc *= element[num_element-1].AbsDet();

  return pow (hK, s) * value_loc;
}


//===============================================================
template <class T>
T rt02d<T>::eta_13_K_p (const vector<T>& P, T s, int num_element)
//===============================================================
{
  T value = 0;
  int num_seg, NumMaille1, NumMaille2;
  int NumNoeudA, NumNoeudB;
  T value_i;

  for (int i=1; i<=3; i++)
    { num_seg = EdgesPerTriangle_(num_element, i);
      NumNoeudA = SetOfEdges_(num_seg, 1);
      NumNoeudB = SetOfEdges_(num_seg, 2);
      NumMaille1 = TrianglesPerEdge_(num_seg, 1); // c'est K+
      NumMaille2 = TrianglesPerEdge_(num_seg, 2); // c'est K-
      vector<T> AB (vertices[NumNoeudB-1] - vertices[NumNoeudA-1]);
      vector<T> M ((vertices[NumNoeudB-1] + vertices[NumNoeudA-1]) * (T) 0.5);
      // K+ est à droite du segment quand on le parcourt dans le sens A->B
      T E = AB.norm ();
     vector<T> n(AB.ortho2D()/E); // c'est donc la normale unitaire sortante à K+
     vector<T> p_M_Maille1(2), p_M_Maille2(2);

     if (NumMaille1 && NumMaille2) // segment interne
       { p_M_Maille1 *= 0;
	 p_M_Maille2 *= 0;
	 for (int j=1; j<=3; j++)
	   { p_M_Maille1 += P[EdgesPerTriangle_(NumMaille1,j)] * (M - element[NumMaille1-1][j]) * (T) NormalDirection_(NumMaille1,j);
	     p_M_Maille2 += P[EdgesPerTriangle_(NumMaille2,j)] * (M - element[NumMaille2-1][j]) * (T) NormalDirection_(NumMaille2,j);
	   }
	 p_M_Maille1 /= element[NumMaille1-1].AbsDet();
	 p_M_Maille2 /= element[NumMaille2-1].AbsDet();
	 value_i = pow (abs ( pow (sqrt ( pow (p_M_Maille1[1], 2) + pow (p_M_Maille1[2], 2)), (2-s)/(s-1)) * (p_M_Maille1 * AB / E)
			      - pow (sqrt ( pow (p_M_Maille2[1], 2) + pow (p_M_Maille2[2], 2)), (2-s)/(s-1)) * (p_M_Maille2 * AB / E)), s) * E;
	 value += value_i * E;
       }
    }

  return value;
}


//====================================================
template <class T>
T rt02d<T>::eta_1 (const vector<T>& P, T s, int NbPts)
//====================================================
{
  T value = 0;
  for (int num_element=1; num_element<=nbelement_; num_element++)
    //value+=eta_11_K_p(P,s,NbPts,num_element);
    value += eta_11_K_p (P, s, NbPts, num_element) + eta_13_K_p (P, s, num_element);

  return powf (value, 1./s);
}


//==================================================================
template <class T>
T rt02d<T>::eta_2_K_q (T (*f)(T,T), T s, int NbPts, int num_element)
//==================================================================
{
  T omegak, omegal, xk, xl;
  vector<T> Fxkl (2);
  integration<T> itg;
  itg.Gauss (NbPts, 0, 1);

  T error_loc = 0;
  vector<T> M (Fk (1/3., 1/3., num_element)); // Barycentre de la maille

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Fxkl = Fk (xk, xl, num_element);
	  error_loc += omegak * omegal * (1 - xk) * pow (f (Fxkl[1], Fxkl[2]) - f (M[1], M[2]), 2);
	}
    }
  error_loc *= element[num_element-1].AbsDet();

  return pow (error_loc, s*0.5);
}


//=============================================
template <class T>
T rt02d<T>::eta_2 (T (*f)(T,T), T s, int NbPts)
//=============================================
{
  T value = 0;
  for (int num_element=1; num_element<=nbelement_; num_element++)
    value += eta_2_K_q (f, s, NbPts,num_element);

  return powf (value, 1/s);
}


/*!
  \param num_element element's number
  \return local mass matrix of the element num_element
*/

//=========================================================
template <class T>
matrix<T> rt02d<T>::LocalMassMatrix (int num_element) const
//=========================================================
{
  // ici on a choisi 5 points d'intégration ce qui est trop important
  // compte tenu du degré des fonctions à integrer... A faire baisser à
  // la valeur tout juste suffisante.

  int NbIntPts = 5;
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  assert ((num_element >= 1) && (num_element <= nbelement_));
  matrix<T> matloc (3,3);
  matrix<T> B = t(element[num_element-1].Mk()) * element[num_element-1].Mk() / element[num_element-1].AbsDet();
  T valeurint;
  T xk, yj, omegak, omegaj;
  vector<T> phim(2), phip(2);
  matrix<T> valeurfonctions (2,3);

  for (int m=1; m<=3; m++)
    for (int p=1; p<=3; p++)
      { valeurint = 0;
        for (int k=1; k<=NbIntPts; k++)
	  { omegak = itg.weight (k);
	    xk = itg.point (k);
	    for (int j=1; j<=NbIntPts; j++)
	      { omegaj = itg.weight (j);
		yj = itg.point (j);
		valeurfonctions = RefBasisFunction (xk, (1-xk) * yj);
		for (int l=1; l<=2; l++)
		  { phim[l] = valeurfonctions (l,m);
		    phip[l] = valeurfonctions (l,p);
		  }
		valeurint += omegaj * omegak * abs (1-xk) * B * phim * phip;
	      }
	  }
	matloc(m,p) = NormalDirection_(num_element, m) * NormalDirection_(num_element, p) * valeurint;
      }

  return matloc;
}


/*!
  \return the global matrix of the tetraRT0 element
*/


//==============================
template <class T>
spmatrix<T> rt02d<T>::UV() const
//==============================
{
  matrix<T> Mass (DofNumber_, DofNumber_);
  spmatrix<T> UV_(GlobalDofNumber_, GlobalDofNumber_, 7);
  for (int numele=1; numele<=nbelement_; numele++)
    { Mass = LocalMassMatrix (numele);
      for (int i=1; i<=DofNumber_; i++)
	for (int j=1; j<=DofNumber_; j++)
	  UV_(EdgesPerTriangle_(numele, i), EdgesPerTriangle_(numele, j)) += Mass (i,j);
    }

  return UV_;
}


//=============================================================================
template <class T>
T rt02d<T>::error_div (T (*divp)(T,T), const vector<T>& uh, int NbIntPts) const
//=============================================================================
{
  T GlobalError = 0;
  T LocalError2;
  T Somme;
  T  Determinant;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T xk, yj, omegak, omegaj;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { Determinant = element[num_element-1].AbsDet();
      LocalError2 = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int j=1; j<=itg.NbIntPts(); j++)
	    { omegaj = itg.weight (j);
	      yj = itg.point (j);
	      Somme = 0;
	      for (int l=1; l<=3; l++)
		Somme += uh[EdgesPerTriangle_(num_element,l)] * NormalDirection_(num_element,l);
	      LocalError2 += omegaj * omegak * abs (1-xk) * pow (divp (Fk (xk, (1-xk) * yj, num_element)[1], Fk (xk, (1-xk) * yj, num_element)[2]) - 2 * Somme / Determinant, 2);
	    }
	}
      GlobalError += Determinant * LocalError2;
    }

  return sqrt (GlobalError);
}


//=================================================================================
template <class T>
T rt02d<T>::error_L2 (const vector<T>& P, T (*px)(T,T), T (*py)(T,T), int NbIntPts)
//=================================================================================
{
  T omegak, omegal, xk, xl;
  T valeur_px, valeur_py;
  vector<T> Fxkl(2), vec(2);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  matrix<T> Mk(2,2), BasisFunction(2,3);
  T error = 0;
  T error_loc;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { error_loc = 0;
      Mk = element[num_element-1].Mk();
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1 - xk) * itg.point (l);
	      Fxkl = Fk (xk, xl, num_element);
	      BasisFunction = Mk * RefBasisFunction (xk, xl);
	      vec[1] = vec[2] = 0;
	      for (int j=1; j<=3; j++)
		{ vec[1] += P[(*this)(num_element,j)] * BasisFunction(1,j) * NormalDirection_(num_element,j);
		  vec[2] += P[(*this)(num_element,j)] * BasisFunction(2,j) * NormalDirection_(num_element,j);
		}
	      vec = vec / element[num_element-1].AbsDet();
	      valeur_px = px (Fxkl[1], Fxkl[2]);
	      valeur_py = py (Fxkl[1], Fxkl[2]);
	      error_loc += omegak * omegal * (1-xk) * (pow (valeur_px - vec[1], 2) + pow (valeur_py - vec[2], 2));
	    }
	}
      error_loc *= element[num_element-1].AbsDet();
      error += error_loc;
    }

  return sqrt (error);
}


//=======================================================================================================
template <class T>
T rt02d<T>::error_L2_loc2 (const vector<T>& P, T (*px)(T,T), T (*py)(T,T), int NbIntPts, int num_element)
//=======================================================================================================
{
  T omegak, omegal, xk, xl;
  T valeur_px, valeur_py;
  vector<T> Fxkl(2), vec(2);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  matrix<T> Mk(2,2), BasisFunction(2,3);
  T error_loc2 = 0;

  Mk = element[num_element-1].Mk();
  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Fxkl = Fk (xk, xl, num_element);
	  BasisFunction = Mk * RefBasisFunction (xk, xl);
	  vec[1] = vec[2] = 0;
	  for (int j=1; j<=3; j++)
	    { vec[1] += P[(*this)(num_element,j)] * BasisFunction(1,j) * NormalDirection_(num_element,j);
	      vec[2] += P[(*this)(num_element,j)] * BasisFunction(2,j) * NormalDirection_(num_element,j);
	    }
	  vec = vec / element[num_element-1].AbsDet();
	  valeur_px = px (Fxkl[1], Fxkl[2]);
	  valeur_py = py (Fxkl[1], Fxkl[2]);
	  error_loc2 += omegak * omegal * (1-xk) * (pow (valeur_px - vec[1], 2) + pow (valeur_py - vec[2], 2));
	}
    }
  error_loc2 *= element[num_element-1].AbsDet();

  return error_loc2;
}


//=====================================================================
template <class T>
T rt02d<T>::error_L2_u (const vector<T>& uh, T (*u)(T,T), int NbIntPts)
//=====================================================================
{
  T omegak, omegal, xk, xl;
  T valeur_u;
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);
  T error = 0;
  T error_loc;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { error_loc = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1-xk) * itg.point (l);
	      valeur_u = u (Fk (xk, xl, num_element)[1], Fk (xk, xl, num_element)[2]);
	      error_loc += omegak * omegal * (1-xk) * pow (valeur_u - uh[num_element],2);
	    }
	}
      error_loc *= element[num_element-1].AbsDet();
      error += error_loc;
    }

  return sqrt (error);
}


}


#endif
