/***************************************************************
 *             Finite Element Method Object Library            *
 *           class tetrap0 : declaration for tetrap0           *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.0.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2004 CREUSE Emmanuel
 * copyright © 2011,2012 COLLARD Christophe
 * copyright © 2004,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2004 Arts et Métiers ParisTech
 * copyright © 2004 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2004 Laboratoire de Physique et Mécanique des Matériaux (LPMM - UMR 7554)
 * copyright © 2004 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace femol
    \brief Finite Element Method Object Libraries
*/

/*! \class femol::tetrap0
    \brief tetrap0 library \n

    \htmlonly
    <FONT color="#838383">

    tetrap0 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2004 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly 2011, 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2004, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2004 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	     copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2004-2012
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type tetrap0
#endif

#ifndef __tetrap0_hpp
#define __tetrap0_hpp

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __tetrahedron_hpp
#include "FEMOL++/meshes/tetrahedron.hpp"
#endif

#ifndef __mesh_hpp
#include "FEMOL++/meshes/mesh.hpp"
#endif

using namespace std;
using namespace mol;

namespace femol
{


//==============================================================
template <class T> class tetrap0 : public mesh<tetrahedron<T>,T>
//==============================================================
{
  using mesh<tetrahedron<T>,T>::alive;
  using mesh<tetrahedron<T>,T>::dimspace_;
  using mesh<tetrahedron<T>,T>::nbvertex_;
  using mesh<tetrahedron<T>,T>::nbelement_;
  using mesh<tetrahedron<T>,T>::vertices;
  using mesh<tetrahedron<T>,T>::element;

  private:
    int DofNumber_;
    int GlobalDofNumber_;

  public:
    tetrap0 ();
    tetrap0 (const char*, const char* = "");
    tetrap0 (const tetrap0<T>&);
    tetrap0<T>& operator = (const tetrap0<T>&); //tetrap0 affectation
    int DofNumber () const {return DofNumber_;}
    int GlobalDofNumber () const {return GlobalDofNumber_;}
    vector<T> BasisFunction (T, T, T) const;
    matrix<T> DiffBasisFunction (T xref=0, T yref=0, T zref=0) const;
    virtual int operator () (int,int) const;
    template <class Tf> friend bool operator == (const tetrap0<Tf>&, const tetrap0<Tf>&);
    template <class Tf> friend bool operator != (const tetrap0<Tf>&, const tetrap0<Tf>&);
    vector<T> Fk (T, T, T, int) const;
    T Integrate_f (T (*f)(T, T, T), int, integration<T>) const; //compute int_numele(f)dx
    T error_L2 (T (*u)(T, T, T), const vector<T>&, int) const;
    T error_L2_loc_2 (T (*u)(T, T, T), const vector<T>&, int, int) const;
};


//=====Private methods for tetrap0===========================================


//=====Public methods for tetrap0============================================


//================================================
template <class T>
tetrap0<T>::tetrap0 () : mesh<tetrahedron<T>,T> ()
//================================================
{
  DofNumber_ = GlobalDofNumber_ = 0;
}


//==============================================================================================
template <class T>
tetrap0<T>::tetrap0 (const char*file1, const char*file2) : mesh<tetrahedron<T>,T> (file1, file2)
//==============================================================================================
{
  DofNumber_ = 1;
  GlobalDofNumber_ = nbelement_;
}


//======================================================================
template <class T> 
tetrap0<T>::tetrap0 (const tetrap0<T>& EF) : mesh<tetrahedron<T>,T> (EF)
//======================================================================
{
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
}


//=======================================================
template <class T>
tetrap0<T>& tetrap0<T>::operator = (const tetrap0<T>& EF)
//=======================================================
{
  (*this).mesh<tetrahedron<T>,T>::operator = (EF);
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;

  return (*this);
}


//================================================================
template <class T>
vector<T> tetrap0<T>::BasisFunction (T xref, T yref, T zref) const
//================================================================
{
  vector<T> P (DofNumber_);
  P[1] = 1;

  return P;
}


//====================================================================
template <class T>
matrix<T> tetrap0<T>::DiffBasisFunction (T xref, T yref, T zref) const
//====================================================================
{
  matrix<T> DP (3, DofNumber_);
  DP(1,1) = 0;
  DP(2,1) = 0;
  DP(3,1) = 0;

  return DP;
}


//========================================================
template <class T>
int tetrap0<T>::operator () (int num_element, int j) const
//========================================================
{
  assert ((j >= 1) && (j <= DofNumber_) && (num_element >= 1) && (num_element <= nbelement_));
  return num_element;
}


//===============================================================
template <class Tf>
bool operator == (const tetrap0<Tf>& EF1, const tetrap0<Tf>& EF2)
//===============================================================
{
  bool boolean = true;
  boolean *= (mesh<tetrahedron<Tf>,Tf> (EF1) == mesh<tetrahedron<Tf>,Tf> (EF2));
  boolean *= (EF1.DofNumber_ == EF2.DofNumber_);
  boolean *= (EF1.GlobalDofNumber_ == EF2.GlobalDofNumber_);

  return boolean;
}


//==============================================================
template <class Tf>
int operator != (const tetrap0<Tf>& EF1, const tetrap0<Tf>& EF2)
//==============================================================
{
  return !(EF1 == EF2);
}


//===================================================
template <class T>
vector<T> tetrap0<T>::Fk (T x, T y, T z, int i) const
//===================================================
{
  matrix<T> mat (3,3);
  vector<T> A1A2(3), A1A3(3), A1A4(3), A1(3), vec(3);
  vec[1] = x ;
  vec[2] = y;
  vec[3] = z;

  A1 = vector<T> (element[i-1][1]);
  A1A2 = vector<T> (element[i-1][2]) - A1;
  A1A3 = vector<T> (element[i-1][3]) - A1;
  A1A4 = vector<T> (element[i-1][4]) - A1;

  mat(1,1) = A1A2[1];
  mat(2,1) = A1A2[2];
  mat(3,1) = A1A2[3];
  mat(1,2) = A1A3[1];
  mat(2,2) = A1A3[2];
  mat(3,2) = A1A3[3];
  mat(1,3) = A1A4[1];
  mat(2,3) = A1A4[2];
  mat(3,3) = A1A4[3];

  return mat * vec + A1;
}


//==================================================================================
template <class T>
T tetrap0<T>::Integrate_f (T (*f)(T,T,T), int num_element, integration<T> itg) const
//==================================================================================
{
  T xk, yj, zi, omegak, omegaj, omegai, valeurfonction;
  T  valeurint = 0;

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int j=1; j<=itg.NbIntPts(); j++)
	{ omegaj = itg.weight (j);
	  yj = itg.point (j);
	  for (int i=1; i<=itg.NbIntPts(); i++)
	    { omegai = itg.weight (i);
	      zi = itg.point (i);
	      valeurfonction = f (Fk (xk, (1 - xk) * yj, (1 - xk - yj + xk * yj) * zi, num_element)[1],
				  Fk (xk, (1 - xk) * yj, (1 - xk - yj + xk * yj) * zi, num_element)[2],
				  Fk (xk, (1 - xk) * yj, (1 - xk - yj + xk * yj) * zi, num_element)[3]);
	      valeurint += omegai * omegaj * omegak * abs ((1 - xk) * (1 - xk - yj + xk * yj)) * valeurfonction;
	    }
	}
    }
  valeurint *= element[num_element-1].AbsDet();

  return valeurint;
}


//=============================================================================
template <class T>
T tetrap0<T>::error_L2 (T (*u)(T,T,T), const vector<T>& uh, int NbIntPts) const
//=============================================================================
{
  T GlobalError = 0;
  T LocalError;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T xk, yj, zi, omegak, omegaj, omegai;
  GlobalError = 0;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { LocalError = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int j=1; j<=itg.NbIntPts(); j++)
	    { omegaj = itg.weight (j);
	      yj = itg.point (j);
	      for (int i=1; i<=itg.NbIntPts(); i++)
		{ omegai = itg.weight (i);
		  zi = itg.point (i);
		  LocalError += omegai * omegaj * omegak * abs ((1 - xk) * (1 - xk - yj + xk * yj)) * pow (u (Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj) * zi, num_element)[1],
													      Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj) * zi, num_element)[2],
													      Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj) * zi, num_element)[3])
													   - uh[num_element], 2);
		}
	    }
	}
      GlobalError += element[num_element-1].AbsDet() * LocalError;
    }

  return sqrt (GlobalError);
}


//====================================================================================================
template <class T>
T tetrap0<T>::error_L2_loc_2 (T (*u)(T,T,T), const vector<T>& uh, int NbIntPts, int num_element) const
//====================================================================================================
{
  T LocalError = 0;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T xk, yj, zi, omegak, omegaj, omegai;

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int j=1; j<=itg.NbIntPts(); j++)
	{ omegaj = itg.weight (j);
	  yj = itg.point (j);
	  for (int i=1; i<=itg.NbIntPts(); i++)
	    { omegai = itg.weight (i);
	      zi = itg.point (i);
	      LocalError += omegai * omegaj * omegak * abs ((1-xk) * (1-xk-yj+xk*yj)) * pow (u (Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[1],
												Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[2],
												Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[3])
											     - uh[num_element], 2);
	    }
	}
    }

  return element[num_element-1].AbsDet() * LocalError;
}


}


#endif
