/***************************************************************
 *             Finite Element Method Object Library            *
 *       class tetrahedron : declaration for tetrahedron       *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.0.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2003 CREUSE Emmanuel
 * copyright © 2012 COLLARD Christophe
 * copyright © 2003, 2012 Centre National de la Recherche Scientifique
 * copyright © 2003 Arts et Métiers ParisTech
 * copyright © 2003 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2003 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2003 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace femol
    \brief Finite Element Method Object Libraries
*/

/*! \class femol::tetrahedron
    \brief tetrahedron library \n

    \htmlonly 
    <FONT color="#838383">

    tetrahedron belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2003 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2003, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2003 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2003 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2003 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2003 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2003-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type tetrahedron
#endif

#ifndef __tetrahedron_hpp
#define __tetrahedron_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __node_hpp
#include "FEMOL++/meshes/node.hpp"
#endif


/*
Let A,B,C,D 4 nodes of R3, we define
the template class tetrahedron as follows:

       (*)--->&B
       /\\
      / \ \
     /  \  \
    /  (*)---->&D (Tetrahedron<<==>>set of pointers on nodes)
   / /    \  \
  //________\_\
(*)--->&C    (*)--->&A

*/


using namespace std;
using namespace mol;

namespace femol
{


//==================================
template <class T> class tetrahedron
//==================================
{
  protected:
    node<T>** vertex; // define an array of pointers
    int alive; // 1 if node exists O otherwise

  public:
    tetrahedron ();
    tetrahedron (node<T>&, node<T>&, node<T>&, node<T>&);
    tetrahedron (node<T> * A[4]);
    tetrahedron (const tetrahedron<T>&);
    ~tetrahedron ();

    tetrahedron<T>& operator = (const tetrahedron<T>&);
    template <class Tf> friend ostream& operator << (ostream&, const tetrahedron<Tf>&);
    node<T>& operator [] (int) const; // return the i-th node of tetrahedron
    template <class Tf> friend bool operator == (const tetrahedron<Tf>&, const tetrahedron<Tf>&);
    T AbsDet () const; // return the |det| of the affine transfo.
    matrix<T> Mk () const; // matrix of the affine transfo.
    matrix<T> InvMk () const; // inverse matrix of the affine transfo
    int nb_nodes () const; // returns the number of nodes of a tetra
};


//=====Private methods for tetrahedron======================================


//=====Public  methods for tetrahedron======================================


/*!
  \brief Default Constructor
  \n \n
  alive is set to 0 but the memory is not allocated to store the adresses of the nodes of the tetrahedron in vertex
*/

//===============================================
template <class T> tetrahedron<T>::tetrahedron ()
//===============================================
{
  alive = 0;
}


/*!
  \brief Constructor
  \n \n
  \param A node \n
  \param B node \n
  \param C node \n
  \param D node \n \n
  alive is set to 1 and vertex is built with pointers towards A,B,C and D
*/

//=============================================================================================
template <class T> tetrahedron<T>::tetrahedron (node<T>& A, node<T>& B, node<T>& C, node<T>& D)
//=============================================================================================
{
  alive = 1;
  vertex = new node<T>* [4];
  vertex[0] = &A;
  vertex[1] = &B;
  vertex[2] = &C;
  vertex[3] = &D;
}


/*!
  \brief Constructor
  \n \n
  \param A an array containing 4 adresses of nodes \n \n
  alive is set to 1 and vertex is built with the 4 adresses contained in A
*/

//===========================================================
template <class T> tetrahedron<T>::tetrahedron(node<T>* A[4])
//===========================================================
{
  alive = 1;
  vertex = new node<T>* [4];
  vertex[0] = A[0];
  vertex[1] = A[1];
  vertex[2] = A[2];
  vertex[3] = A[3];
}


/*!
  \brief Copy constructor
  \param K tetrahedron to duplicate
*/

//======================================================================
template <class T> tetrahedron<T>::tetrahedron (const tetrahedron<T>& K)
//======================================================================
{
  vertex = new node<T>* [4];
  alive = K.alive;
  vertex[0] = K.vertex[0];
  vertex[1] = K.vertex[1];
  vertex[2] = K.vertex[2];
  vertex[3] = K.vertex[3];
}


/*!
  \brief Destructor
  \n \n
  Deletes vertex and set alive to 0
*/

//================================================
template <class T> tetrahedron<T>::~tetrahedron ()
//================================================
{
  if(alive)
      delete [] vertex;
  alive=0;
}


/*!
  \brief Standard operator = for tetrahedron \n \n
  \param K tetrahedron to copy \n
  \return reference of the left hand size tetrahedron (for multiple equalities)
  \n\n
*/

//=====================================================================================
template <class T> tetrahedron<T>& tetrahedron<T>::operator = (const tetrahedron<T>& K)
//=====================================================================================
{
  assert (K.alive);
  if (!alive)
    vertex = new node<T>* [4];
  alive = K.alive;
  vertex[0] = K.vertex[0];
  vertex[1] = K.vertex[1];
  vertex[2] = K.vertex[2];
  vertex[3] = K.vertex[3];

  return (*this);
}


/*!
  \brief  overloads output stream for tetrahedron \n \n
  \param s output flow \n
  \param K tetrahedron
*/

//===========================================================================
template <class T> ostream& operator << (ostream& s, const tetrahedron<T>& K)
//===========================================================================
{
  if (!K.alive)
    s << "undefined tetrahedron";
  else
    { s << "tetrahedron:" << endl;
      s << *(K.vertex[0]) << *(K.vertex[1]) <<*(K.vertex[2]) <<*(K.vertex[3]);
    }

  return s;
}


/*!
  \brief Returns one node of the tetrahedron \n \n
  Warning : we use the mathematical notations so the nodes adresses are in the set [1;4], and not [0:3] \n \n
  \param i node number \n \n
  \return the node i of the tetrahedron \n \n
*/

//===================================================================
template <class T> node<T>& tetrahedron<T>::operator [] (int i) const
//===================================================================
{
  assert ((i > 0) && (i <= 4) && alive);
  return *(vertex[i-1]);
}


/*!
  \brief  Check the equality of two tetrahedra \n \n
  \param K tetrahedron \n
  \param L tetrahedron \n
  \return 1 if equal, 0 if not
*/

//====================================================================================
template <class T> bool operator == (const tetrahedron<T> &K, const tetrahedron<T> &L)
//====================================================================================
{
  bool boolean = 1;
  boolean *= ((K[1] == L[1]) && (K[2] == L[2]) && (K[3] == L[3]) && (K[4] == L[4]) && (K.alive == L.alive));

  return boolean;
}


/*!
  \brief  Returns the absolute value of the matrix of the affine application which sends the reference triangle to the actual triangle.   \n \n
  \return The value of the absolute value of the determinant
*/

//==================================================
template <class T> T tetrahedron<T>::AbsDet () const
//==================================================
{
  return abs (det (Mk()));
}


/*!
  \brief  Returns the 3*3 matrix of the affine application which sends the reference tetrahedron to the actual tetrahedron.   \n \n
  \return The 3*3 matrix
*/

//======================================================
template <class T> matrix<T> tetrahedron<T>::Mk () const
//======================================================
{
  matrix<T> M(3,3);
  M(1,1) = (*this)[2][1] - (*this)[1][1];
  M(1,2) = (*this)[3][1] - (*this)[1][1];
  M(1,3) = (*this)[4][1] - (*this)[1][1];
  M(2,1) = (*this)[2][2] - (*this)[1][2];
  M(2,2) = (*this)[3][2] - (*this)[1][2];
  M(2,3) = (*this)[4][2] - (*this)[1][2];
  M(3,1) = (*this)[2][3] - (*this)[1][3];
  M(3,2) = (*this)[3][3] - (*this)[1][3];
  M(3,3) = (*this)[4][3] - (*this)[1][3];

  return M;
}


/*!
  \brief  Returns the inverse of the 3*3 matrix of the affine application which sends the reference tetrahedron to the actual tetrahedron.   \n \n
  \return The 3*3 matrix
*/

//=========================================================
template <class T> matrix<T> tetrahedron<T>::InvMk () const
//=========================================================
{
  return Mk().inv();
}

/*!
  \brief  Returns the number of nodes in a tetrahedron (always 4).   \n \n
  \return 4
*/

//======================================================
template <class T> int tetrahedron<T>::nb_nodes () const
//======================================================
{
  return 4;
}


}


#endif
