/***************************************************************
 *             Finite Element Method Object Library            *
 *                          tutorial 2                         *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 	2.6.0                  *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2006 BOULAAJINE Lahcen
 * copyright © 2006 COCHEZ Sarah
 * copyright © 2006 CREUSE Emmanuel
 * copyright © 2012 COLLARD Christophe
 * copyright © 2006,2012 Centre National de la Recherche Scientifique
 * copyright © 2006 Arts et Métiers ParisTech
 * copyright © 2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \class tuto2
    \brief tuto2 example \n

    \htmlonly 
    <FONT color="#838383">

    tuto2 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2006 BOULAAJINE Lahcen \n
             copyright \htmlonly &#169; \endhtmlonly  2006 COCHEZ Sarah \n
	     copyright \htmlonly &#169; \endhtmlonly  2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2006, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 2.6.0
    \date 2006-2012
    \bug none
    \warning none
*/


#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__STRING_H)
#include <string>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__TIME_H)
#include <time.h>
#endif

#if !defined(__MATHS_H)
#include "MOL++/maths.h"
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MESH_TRI_2D_H)
#include "FEMOL++/meshes/mesh_tri_2d.h"
#endif

#if !defined(__CONVERT_H)
#include "FEMOL++/meshes/convert.h"
#endif

#if !defined(__TRIANGLE_H)
#include "FEMOL++/meshes/triangle.h"
#endif

#if !defined(__P1C2D_H)
#include "FEMOL++/elements/p1c2d.h"
#endif

#if !defined(__AFFICHE_H)
#include "tests/affiche.h"
#endif

// Exact solution (harmonic)
long double u_p1c (long double x, long double y)
{
  long double r2;
  long double theta;
  r2 = pow (x, 2) + pow (y, 2);
  long double prec = 1e-12;

  if (abs (x) < prec && abs (y) < prec) theta = 0;
  else if (abs (x) < prec && y > 0) theta = pi() * 0.5;
  else if (abs (x) < prec && y < 0) theta = 1.5 * pi();
  else if (x < 0) theta = atan (y / x) + pi();
  else theta = atan (y / x);

  return powf(r2, (1/3.)) * sin (2 * theta / 3.);
}

using namespace std;
using namespace mol;
using namespace femol;


//====
main()
//====
{
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << " BEGINNING tuto2 \n";
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << endl;

  //----------
  // Mesh file
  //----------
  make_conversion2dtri<long double> ("../../tests/FEMOL++/data/Lshape2d/2dLshape2.dat", "tampon.dat", "lshape2d");

  //-----------------------------------
  //Nb points for numerical integration
  //-----------------------------------
  int NbIntPts = 3;

  //----------------
  // Refinement data
  //----------------
  int nb_refine_1 = 5; // Nb Refinements for angle=1.0
  int nb_refine_2 = 20; // Nb Refinements for angle=0.5
  int nb_refine_3 = 20; // Nb Refinements for angle=0.0

  //---------------------------------
  // Mesh visualisation (1=YES, 0=NO)
  //---------------------------------
  int visu_meshes = 1;

  //----------------
  // Local variables
  //----------------
  long double L2_error, L2_error_anc;
  int ndof, ndof_anc;
  vector<long double> s, x0, uh, local_error;
  vector<int> TABTRI;
  spmatrix<long double> DuDv;

  //-----------------------------------------------
  //		angle=1.0
  // ----------------------------------------------
  cout << endl; 
  long double angle = 1;
  cout << "-----------------------"<< endl;
  cout << "-----------------------"<< endl;
  cout << "angle = " << angle << endl;
  cout << "-----------------------"<< endl;
  cout << "-----------------------"<< endl;

  //------------------
  // p1c2d declaration
  //------------------
  p1c2d<long double> Th1 ("tampon.dat");
  if (visu_meshes) Th1.visu2d ();

  //-----------------------------------
  // Local files (destroyed at the end)
  //-----------------------------------
  ofstream file11 ("tuto21.dat");

  //----------
  // Main loop
  //----------
  for (int num_refine=1; num_refine<=nb_refine_1; num_refine++)
    {  ndof = Th1.GlobalDofNumber ();
      cout << endl;
      cout << "Refinement "  << num_refine << "/" << nb_refine_1 << " ndof = " << ndof;

      //------------------------------------------
      // Global Stifness Matrix Declaration
      // Computation of the Global Stifness Matrix 
      // Declaration of the Right-Hand-Side
      // Dirichlet Boundary conditions
      //------------------------------------------
      DuDv &=& spmatrix<long double> (ndof, ndof, Th1.NbMaxSegmentsPerNode()+1);
      Th1.DUDV (DuDv);
      vector<long double> s (ndof);
      Th1.Dirichlet (DuDv, s, u_p1c);

      //-------------------------
      // Linear system resolution
      //-------------------------
      x0 &=& vector<long double> (ndof);
      int iter;
      uh &=& conjugate_gradient (DuDv, s, x0, iter, 1.e-08);

      //---------------------------------------------
      // Error computations and orders of convergence
      //---------------------------------------------
      L2_error_anc = L2_error;
      L2_error = Th1.error_L2 (u_p1c, uh, NbIntPts);
      cout << " L2_error = " << L2_error;
      if (num_refine > 1)
	cout << " Order of convergence for L2-norm = " << -2 * ((log (L2_error) - log (L2_error_anc)) / (log ((long double) ndof) - log ((long double) ndof_anc)));
      cout << endl;

      //------------------------
      // Refinement mesh process
      //------------------------
      if (num_refine<nb_refine_1)
	{ local_error &=& vector<long double> (Th1.nbelement());
	  long double erreur_max = 0;
	  for (int num_element=1; num_element<=Th1.nbelement(); num_element++)
	    { local_error[num_element] = Th1.error_L2_loc (u_p1c, uh, NbIntPts, num_element);
	      if (local_error[num_element] > erreur_max) erreur_max = local_error[num_element];
	    }
	  TABTRI &=& vector<int> (Th1.nbelement());
	  for (int num_element=1; num_element<=Th1.nbelement(); num_element++)
	    if (local_error[num_element] > 0.75 * erreur_max) TABTRI[num_element] = -1;
	  Th1.refine_FE_angle (TABTRI, angle);
	  if (visu_meshes) Th1.visu2d ();

	  //------------------------------
	  // Results are written in a file
	  //------------------------------
	  file11 << ndof << "     " << L2_error << endl;
	  ndof_anc = ndof;
	}
    }

  // In the following : the same program with angle=0.5 and then with angle=0.0

  //----------------------------
  //		angle=0.5
  // ---------------------------
  cout << endl;
  angle = 0.5;
  cout << "-----------------------"<< endl;
  cout << "-----------------------"<< endl;
  cout << "angle = " << angle << endl;
  cout << "-----------------------"<< endl;
  cout << "-----------------------"<< endl;

  p1c2d<long double> Th2 ("tampon.dat");
  if (visu_meshes) Th2.visu2d ();

  ofstream file12 ("tuto22.dat");

  //----------
  // Main loop
  //----------
  for (int num_refine=1; num_refine<=nb_refine_2; num_refine++)
    { ndof = Th2.GlobalDofNumber ();
      cout << endl;
      cout << "Reffinement "  << num_refine << "/" << nb_refine_2 <<" ndof = "<<ndof;

      DuDv &=& spmatrix<long double> (ndof, ndof, Th2.NbMaxSegmentsPerNode()+1);
      Th2.DUDV (DuDv);
      vector<long double> s (ndof);
      Th2.Dirichlet (DuDv, s, u_p1c);

      x0 &=& vector<long double> (ndof);
      int iter;
      uh &=& conjugate_gradient (DuDv, s, x0, iter, 1.e-08);

      L2_error_anc = L2_error;
      L2_error = Th2.error_L2 (u_p1c, uh, NbIntPts);
      cout << " L2_error = " << L2_error;
      if (num_refine > 1)
	cout << " Order of convergence for L2-norm = " << -2 * ((log (L2_error) - log (L2_error_anc)) / (log ((long double) ndof) - log ((long double) ndof_anc)));
      cout << endl;
      if (num_refine < nb_refine_2)
	{ local_error &=& vector<long double> (Th2.nbelement());
	  long double erreur_max = 0;
	  for (int num_element=1; num_element<=Th2.nbelement(); num_element++)
	    { local_error[num_element] = Th2.error_L2_loc (u_p1c, uh, NbIntPts, num_element);
	      if (local_error[num_element] > erreur_max) erreur_max = local_error[num_element];
	    }
	  TABTRI &=& vector<int> (Th2.nbelement());
	  for (int num_element=1; num_element<=Th2.nbelement(); num_element++)
	    if (local_error[num_element] > 0.75 * erreur_max) TABTRI[num_element]=-1;
	  Th2.refine_FE_angle (TABTRI, angle);
	  if (visu_meshes) Th2.visu2d ();

	  file12 << ndof << "     " << L2_error << endl;
	  ndof_anc = ndof;
	}
    }

  //----------------------------------------
  //		angle=0.0
  // ---------------------------------------
  cout << endl;
  angle = 0;
  cout << "-----------------------"<< endl;
  cout << "-----------------------"<< endl;
  cout << "angle = " << angle << endl;
  cout << "-----------------------"<< endl;
  cout << "-----------------------"<< endl;
  p1c2d<long double> Th3 ("tampon.dat");
  if (visu_meshes) Th3.visu2d ();

  ofstream file13 ("tuto23.dat");

  //----------
  // Main loop
  //----------
  for (int num_refine=1; num_refine<=nb_refine_3; num_refine++)
    { ndof = Th3.GlobalDofNumber();
      cout << endl;
      cout << "Reffinement "  << num_refine << "/" << nb_refine_3 <<" ndof = "<<ndof;

      DuDv &=& spmatrix<long double> (ndof, ndof, Th3.NbMaxSegmentsPerNode()+1);
      Th3.DUDV (DuDv);
      vector<long double> s (ndof);
      Th3.Dirichlet (DuDv, s, u_p1c);

      x0 &=& vector<long double> (ndof);
      int iter;
      uh &=& conjugate_gradient (DuDv, s, x0, iter, 1.e-08);

      L2_error_anc = L2_error;
      L2_error = Th3.error_L2 (u_p1c, uh, NbIntPts);
      cout << " L2_error = " << L2_error;
      if (num_refine > 1)
	cout << " Order of convergence for L2-norm = " << -2 * ((log (L2_error) - log (L2_error_anc)) / (log ((long double) ndof) - log ((long double) ndof_anc)));
      cout << endl;
      if (num_refine < nb_refine_3)
	{ local_error &=& vector<long double> (Th3.nbelement());
	  long double erreur_max = 0;
	  for (int num_element=1; num_element<=Th3.nbelement(); num_element++)
	    { local_error[num_element] = Th3.error_L2_loc (u_p1c, uh, NbIntPts, num_element);
	      if (local_error[num_element] > erreur_max) erreur_max = local_error[num_element];
	    }
	  TABTRI &=& vector<int> (Th3.nbelement());
	  for (int num_element=1; num_element<=Th3.nbelement(); num_element++)
	    if (local_error[num_element] > 0.75*erreur_max) TABTRI[num_element] = -1;
	  Th3.refine_FE_angle (TABTRI, angle);
	  if (visu_meshes) Th3.visu2d();

	  file13 << ndof << "     " << L2_error << endl;
	  ndof_anc = ndof;
	}
    }
  //----------------------------
  //Visualisation of the results
  //----------------------------
  ofstream file23 ("visu23.gnu");
  file23 << "set logscale xy" << endl;
  //file23 << "set terminal postscript" << endl;
  //file23 << "set output \" figure.ps \" " << endl;
  file23 << "plot [1:80000] [0.0000015:50] \"tuto21.dat\"  using 1:2 title \"alpha=1.0\" with linespoints, \"tuto22.dat\" using 1:2 title \"alpha=0.5\" with linespoints, \"tuto23.dat\" using 1:2 title \"alpha=0.0\" with linespoints" << endl;
  file23 << "pause -1 \"Press Enter\" " << endl;
  file23.close ();

  system ("gnuplot visu23.gnu");

  //----------------------------------
  // Closing and deleting of the files
  //----------------------------------
  remove ("tuto23.dat");
  remove ("tuto22.dat");
  remove ("tuto21.dat");
  remove ("visu23.gnu");
  remove ("tampon.dat");

  //-----------------------------------------
  cout << endl;
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << " END tuto2 \n";
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  //-----------------------------------------
}
