/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class p2c2d : declaration for p2c2d               *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 	3.2.0                  *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2006 CREUSE Emmanuel
 * copyright © 2006 SOUALEM Nadir
 * copyright © 2012 COLLARD Christophe
 * copyright © 2006,2012 Centre National de la Recherche Scientifique
 * copyright © 2006 Arts et Métiers ParisTech
 * copyright © 2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace femol
    \brief Finite Element Method Object Libraries
*/

/*! \class femol::p2c2d
    \brief p2c2d library \n

    \htmlonly 
    <FONT color="#838383">

    p2c2d belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2006 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2006, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2005-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type p2c2d
#endif

#ifndef __p2c2d_hpp
#define __p2c2d_hpp

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __integration_hpp
#include "MOL++/integration.hpp"
#endif

#ifndef __mesh_tri_2d_hpp
#include "FEMOL++/meshes/mesh_tri_2d.hpp"
#endif

long double f_p2c2d (long double x, long double y)
{
  return -exp(x) - exp(y);
}

long double u_p2c2d (long double x, long double y)
{
  return exp(x) + exp(y);
}

long double dudx_p2c2d (long double x, long double y)
{
  return exp(x);
}

long double dudy_p2c2d (long double x, long double y)
{
  return exp(y);
}

using namespace std;
using namespace mol;

namespace femol
{


//====================================================
template <class T> class p2c2d : public mesh_tri_2d<T>
//====================================================
{
  using mesh<triangle<T>,T>::alive;
  using mesh<triangle<T>,T>::dimspace_;
  using mesh<triangle<T>,T>::nbvertex_;
  using mesh<triangle<T>,T>::nbelement_;
  using mesh<triangle<T>,T>::vertices;
  using mesh<triangle<T>,T>::element;

  using mesh_tri_2d<T>::nbsegments_;
  using mesh_tri_2d<T>::SetOfEdges_;
  using mesh_tri_2d<T>::Type_of_edge_;
  using mesh_tri_2d<T>::EdgesPerTriangle_;
  using mesh_tri_2d<T>::NbTrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerTriangle_;
  using mesh_tri_2d<T>::NumberOfNeighboorOfTriangles_;
  using mesh_tri_2d<T>::NumLocNodeInTriangle_;
  using mesh_tri_2d<T>:: NormalDirection_;
  using mesh_tri_2d<T>::hminT_;
  using mesh_tri_2d<T>::hE_;
  using mesh_tri_2d<T>::hminE_;
  using mesh_tri_2d<T>::NormEdge_;
  using mesh_tri_2d<T>::NormalUnitVector_;
  using mesh_tri_2d<T>::NbMaxSegmentsPerNode_;
  using mesh_tri_2d<T>::NbSegmentsPerNode_;

  private:
    int DofNumber_;
    int GlobalDofNumber_;
    matrix<T> RefMassMatrix_;
    vector<T> BuildBloc (T (*f)(T,T), int, integration<T>) const;
  public:
    p2c2d ();
    p2c2d (const char*, const char* = "");
    p2c2d (const p2c2d<T>&);
    p2c2d<T>& operator = (const p2c2d<T>&);
    int DofNumber () const {return DofNumber_;}
    int GlobalDofNumber () const {return GlobalDofNumber_;}
    matrix<T> RefMassMatrix () const {return RefMassMatrix_;}
    vector<T> BasisFunction (T, T) const;
    matrix<T> DiffBasisFunction (T xref=0, T yref=0) const;
    matrix<T> Diff2BasisFunction (T xref=0, T yref=0) const;
    virtual int operator () (int, int) const;
    template <class Tf> friend bool operator == (const p2c2d<Tf>&, const p2c2d<Tf>&);
    template <class Tf> friend bool operator != (const p2c2d<Tf>&, const p2c2d<Tf>&);
    matrix<T> LocalMassMatrix (int) const;
    matrix<T> LocalStiffnessMatrix (int) const;
    vector<T> Fk (T, T, int) const;
    template <template <class> class matrix_type> void DUDV (matrix_type<T>&) const; // Global Stiffness matrix
    vector<T> B (T (*f)(T,T), int) const;
    template <template <class> class matrix_type> void Dirichlet (matrix_type<T>&, vector<T>&, T (*g)(T,T)) const; // Dirichlet BC for matrix
    T error_L2 (T (*u)(T,T), const vector<T>&, int) const ;
    T error_semi_H1 (T (*dudx)(T,T), T (*dudy)(T,T), const vector<T>&, int) const;
    /*  T Grad_Normal_Jump_L2 (const vector<T>&, int) const;
	T Grad_Tangential_Jump_L2 (const vector<T>&, int) const;
	T Laplacian_Residual_L2 (T (*f)(T,T), int, const vector<T>&) const*/;
};


//=====Private methods for p2c2d=============================================


//=====Public methods for p2c2d==============================================


/*!
  \brief default constructor of the p2c2d class
*/

//===================================
template <class T>
p2c2d<T>::p2c2d() : mesh_tri_2d<T> ()
//===================================
{
  DofNumber_ = GlobalDofNumber_ = 0;
}


/*!
  \brief constructor of the p2c2d class. The arguments are 2 files file1 and file2. file1 must be documented by the number of elements, node 's numerotation of each element. file2 indicate the dimension (2 or 3), number of nodes and each node coordinates with a boundary condition (1 if node is on the boundary, 0 otherwise).
  \param file1 string
  \param file2 string (if not merged with file1)
*/

//====================================================================================
template <class T>
p2c2d<T>::p2c2d (const char* file1, const char* file2) : mesh_tri_2d<T> (file1, file2)
//====================================================================================
{
  GlobalDofNumber_ = nbsegments_+nbvertex_;
  DofNumber_ = 6;
  RefMassMatrix_ = matrix<T> (DofNumber_, DofNumber_);

  RefMassMatrix_(1,1) = 1 /60.;
  RefMassMatrix_(1,2) = -1/360.;
  RefMassMatrix_(1,3) = -1/360.;
  RefMassMatrix_(1,4) = -1/90.;

  RefMassMatrix_(2,1) = -1/360.;
  RefMassMatrix_(2,2) = 1/60.;
  RefMassMatrix_(2,3) = -1/360.;
  RefMassMatrix_(2,5) = -1/90.;

  RefMassMatrix_(3,1) = -1/360.;
  RefMassMatrix_(3,2) = -1/360.;
  RefMassMatrix_(3,3) = 1/60.;
  RefMassMatrix_(3,6) = -1/90.;

  RefMassMatrix_(4,1) = -1/90.;
  RefMassMatrix_(4,4) = 4/45.;
  RefMassMatrix_(4,5) = 2/45.;
  RefMassMatrix_(4,6) = 2/45.;

  RefMassMatrix_(5,2) = -1/90.;
  RefMassMatrix_(5,4) = 2/45.;
  RefMassMatrix_(5,5) = 4/45.;
  RefMassMatrix_(5,6) = 2/45.;

  RefMassMatrix_(6,3) = -1/90.;
  RefMassMatrix_(6,4) = 2/45.;
  RefMassMatrix_(6,5) = 2/45.;
  RefMassMatrix_(6,6) = 4/45.;
}


/*!
  \brief copy constructor of the p1nc2 class
  \param EF  reference on p2c2d object
*/

//========================================================
template <class T>
p2c2d<T>::p2c2d (const p2c2d<T>& EF) : mesh_tri_2d<T> (EF)
//========================================================
{
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  RefMassMatrix_ = EF.RefMassMatrix_;
}


/*!
  \brief affectation operator for the p2c2d class.
  \param EF reference on p2c2d object
  \return  reference on p2c2d object
*/

//=================================================
template <class T>
p2c2d<T>& p2c2d<T>::operator = (const p2c2d<T>& EF)
//=================================================
{
  (*this).mesh_tri_2d<T>::operator = (EF);
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  RefMassMatrix_ = EF.RefMassMatrix_;

  return (*this);
}


/*!
  \param xref type T
  \param yref type T
  \return vector \f$(\varphi_1(xref,yref),\varphi_2(xref,yref),...,\varphi_6(xref,yref))\f$ where \f$(\varphi_i)_{1\leq i\leq 6}\f$ is the i-th basis function.
*/

//======================================================
template <class T>
vector<T> p2c2d<T>::BasisFunction (T xref, T yref) const
//======================================================
{
  vector<T> P (DofNumber_);

  P[1] = 2 * (1 - xref - yref) * (0.5 - xref - yref);
  P[2] = -2 * xref * (0.5 - xref);
  P[3] = -2 * yref * (0.5 - yref);
  P[4] = 4 * xref * yref;
  P[5] = 4 * (1 - xref - yref) * yref;
  P[6] = 4 * (1 - xref - yref) * xref;

  return P;
}


/*!
  \param xref type T
  \param yref type T
  \return matrix 
  \f$ \left[\begin{array}{ccc}
            \frac{\partial\varphi_1}{\partial x_1} & ... & \frac{\partial\varphi_6}{\partial x_1} \\
	    \frac{\partial\varphi_1}{\partial x_2} & ... & \frac{\partial\varphi_6}{\partial x_2}
	    \end{array} \right]
  \f$
  where \f$(\varphi_i)_{1\leq i\leq 6}\f$ is the i-th basis function.
*/

//==========================================================
template <class T>
matrix<T> p2c2d<T>::DiffBasisFunction (T xref, T yref) const
//==========================================================
{
  matrix<T> DP (2, DofNumber_);

  DP(1,1) = 4 * xref + 4 * yref - 3;
  DP(2,1) = 4 * xref + 4 * yref - 3;
  DP(1,2) = 4 * xref - 1;
  DP(2,3) = 4 * yref - 1;
  DP(1,4) = 4 * yref;
  DP(2,4) = 4 * xref;
  DP(1,5) = -4 * yref;
  DP(2,5) = 4 - 8 * yref - 4 * xref;
  DP(1,6) = 4 - 8 * xref - 4 * yref;
  DP(2,6) = -4 * xref;

  return DP;
}


/*!
  \param xref type T
  \param yref type T
  \return matrix 
  \f$ \left[\begin{array}{ccc}
            \frac{\partial^2\varphi_1}{\partial x_1^2} & ... & \frac{\partial^2\varphi_6^2}{\partial x_1^2} \\
	    \frac{\partial^2\varphi_1}{\partial x_1 \partial x_2} & ... & \frac{\partial^2\varphi_6^2}{\partial x_1 \partial x_2} \\
	    \frac{\partial^2\varphi_1}{\partial x_2^2} & ... & \frac{\partial^2\varphi_6^2}{\partial x_2^2}
	    \end{array} \right]
  \f$ 
  where \f$(\varphi_i)_{1\leq i\leq 6}\f$ is the i-th basis function.
*/

//===========================================================
template <class T>
matrix<T> p2c2d<T>::Diff2BasisFunction (T xref, T yref) const
//===========================================================
{
  matrix<T> DDP (3, DofNumber_);

  DDP(1,1) = 4;
  DDP(2,1) = 4;
  DDP(3,1) = 4;
  DDP(1,2) = 4;
  DDP(3,3) = 4;
  DDP(2,4) = 4;
  DDP(2,5) = -4;
  DDP(3,5) = -8;
  DDP(1,6) = -8;
  DDP(2,6) = -4;

  return DDP;
}


/*!
  \param x type T
  \param y type T
  \param i type integer
  \return vector \f$F_k(x,y,i)\f$
*/

//============================================
template <class T>
vector<T> p2c2d<T>::Fk (T x, T y, int i) const
//============================================
{
  matrix<T> mat (2,2);
  vector<T> vec (2);
  vec[1] = x;
  vec[2] = y;

  vector<T> A1A2 = element[i-1][2] - element[i-1][1];
  vector<T> A1A3 = element[i-1][3] - element[i-1][1];
  mat(1,1) = A1A2[1];
  mat(1,2) = A1A3[1];
  mat(2,1) = A1A2[2];
  mat(2,2) = A1A3[2];

  return mat * vec + element[i-1][1];
}


/*!
  \param num_element element's number
  \return local mass matrix of the element num_element
*/

//=========================================================
template <class T>
matrix<T> p2c2d<T>::LocalMassMatrix (int num_element) const
//=========================================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));

  return RefMassMatrix_ * element[num_element-1].AbsDet();
}


/*!
  \param num_element element's number
  \return local stifness  matrix of the element num_element
*/

//==============================================================
template <class T>
matrix<T> p2c2d<T>::LocalStiffnessMatrix (int num_element) const
//==============================================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  matrix<long double> inv = element[num_element-1].InvMk();
  matrix<long double> Stiff (DofNumber_, DofNumber_);

  integration<T> itg;
  itg.Gauss (2, 0, 1);
  T omegak, omegal, xk, xl;
  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Stiff += omegak * omegal * (1-xk) * (t(DiffBasisFunction (xk, xl)) * inv * t(inv) * DiffBasisFunction (xk, xl));
	}
    }

  return (element[num_element-1].AbsDet()) * Stiff;
}


/*!
  \param num_element element's number
  \param j the j-th dof
  \return global number of the j-th of the element num_element
*/

//======================================================
template <class T>
int p2c2d<T>::operator () (int num_element, int j) const
//======================================================
{
  assert ((j >= 1) && (j <= DofNumber_) && (num_element >= 1) && (num_element <= nbelement_));

  if (j <= 3)
    return this -> number (element[num_element-1][j]);
  else
    return EdgesPerTriangle_(num_element, j-3) + nbvertex_;
}


/*!
  \param EF1 type const p2c2d<T>&
  \param EF2 type const p2c2d<T>&
  \return 1 if EF1 and  EF2 are equal, 0 otherwise
*/

//===========================================================
template <class Tf>
bool operator == (const p2c2d<Tf>& EF1, const p2c2d<Tf>& EF2)
//===========================================================
{
  bool boolean = true;
  boolean *= (mesh_tri_2d<Tf> (EF1) == mesh_tri_2d<Tf> (EF2));
  boolean *= (EF1.DofNumber_ == EF2.DofNumber_);
  boolean *= (EF1.GlobalDofNumber_ == EF2.GlobalDofNumber_);
  boolean*= (EF1.RefMassMatrix_ == EF2.RefMassMatrix_);

  return boolean;
}


/*!
  \param EF1 type const p2c2d<T>&
  \param EF2 type const p2c2d<T>&
  \return 1 if EF1 and  EF2 are different, 0 otherwise
*/

//===========================================================
template <class Tf>
bool operator != (const p2c2d<Tf>& EF1, const p2c2d<Tf>& EF2)
//===========================================================
{
  return !(EF1 == EF2);
}


/*!
  \return the global stifness matrix of the p2c2d element
*/

//==============================================
template <class T>
template <template <class> class  matrix_type>
void p2c2d<T>::DUDV (matrix_type<T> & mat) const
//==============================================
{
  matrix<T> Stiff (DofNumber_, DofNumber_);
  for (int k=1; k<=nbelement_; k++)
    { Stiff = LocalStiffnessMatrix (k);
      for (int i=1; i<=3; i++)
	for (int j=1; j<=3; j++)
	  mat (this -> number (element[k-1][i]), this ->number (element[k-1][j])) += Stiff (i,j);
      for (int i=1; i<=3; i++)
	for (int j=4; j<=6; j++)
	  mat (this -> number (element[k-1][i]), EdgesPerTriangle_(k,j-3) + nbvertex_) += Stiff (i,j);
      for (int i=4; i<=6; i++)
	for (int j=1; j<=3; j++)
	  mat (EdgesPerTriangle_(k,i-3) + nbvertex_, this -> number (element[k-1][j])) += Stiff (i,j); 
      for(int i=4; i<=6; i++)
	for (int j=4; j<=6; j++)
	  mat (EdgesPerTriangle_(k,i-3) + nbvertex_, EdgesPerTriangle_(k,j-3) + nbvertex_) += Stiff (i,j);  
    }
}


/*!
  \brief Value of the contribution to the right-hand side of the variational formulation of a given element \n \n
  \param \f$ f \f$ the function   \n
  \param num_element the element on which we want to integrate \n
  \param itg the Gauss points and weight used for the numerical integration
  \return vector \f$(\int_{num_{element}} f(x,y) \varphi_1(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_2(x,y) dx dy,...,\int_{num_{element}} f(x,y) \varphi_6(x,y) dx dy)\f$ where \f$(\varphi_i)_{1\leq i\leq 6}\f$ is the i-th basis function associated to the element num_element.
  \n\n
*/

//====================================================================================
template <class T>
vector<T> p2c2d<T>::BuildBloc (T (*f)(T,T), int num_element, integration<T> itg) const
//====================================================================================
{
  vector<T> res (DofNumber_);
  vector<T> Basis (DofNumber_);
  T omegak, omegal, xk, xl;
  vector<T> Fxkl;
  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Basis = BasisFunction (xk, xl);
	  Fxkl = Fk (xk, xl, num_element);
          for (int i=1; i<=DofNumber_; i++)
	    res[i] += omegak * omegal * (1-xk) * f (Fxkl[1], Fxkl[2]) * Basis[i];
	}
    }
  res *= element[num_element-1].AbsDet();

  return res;
}


/*!
  \brief Value of the right-hand side of the variational formulation \n \n
  \param \f$ f \f$ the function   \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return vector v with v_i=\f$(\int_{\Omega} f(x,y) \varphi_i(x,y) dx dy \f$ where \f$(\varphi_i)_{1\leq i\leq nbvertex+nbsegments}\f$ is the i-th global basis function.
  \n\n
*/

//=====================================================
template <class T>
vector<T> p2c2d<T>::B (T (*f)(T,T), int NbIntPts) const
//=====================================================
{
  assert (NbIntPts > 0);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  vector<T> B_ (GlobalDofNumber_);
  vector<T> Bloc (DofNumber_);

  for (int k=1; k<=nbelement_; k++)
    { Bloc = BuildBloc (f, k, itg);
      for (int i=1; i<=3; i++)
	B_[this->number(element[k-1][i])] += Bloc[i];
      for (int i=4; i<=6; i++)
	B_[ EdgesPerTriangle_(k,i-3)+nbvertex_] += Bloc[i];
    }

return B_;
}


//============================================================================
template <class T>
template <template <class> class matrix_type>
void p2c2d<T>::Dirichlet (matrix_type<T> & mat, vector<T>&F,T (*g)(T,T)) const
//============================================================================
{
  for (int i=1; i<=nbvertex_; i++)
    { if (mesh_tri_2d<T>::operator () (i).boundary_condition())
	{ F[i] = (*g) (vertices[i-1][1], vertices[i-1][2]);
	  for (int j=1; j<=GlobalDofNumber_; j++)
	    mat(i,j) = 0;
	  mat(i,i) = 1;
	  for (int k=1; k<=GlobalDofNumber_; k++)
	    if (k != i)
	      { F[k] += -mat(k,i) * F[i];
		mat(k,i) = 0;
	      }
	}
    }

  for (int i=nbvertex_+1; i<=GlobalDofNumber_; i++)
    { if (Type_of_edge_[i-nbvertex_])
	{ F[i] = (*g) (0.5 * (vertices[SetOfEdges_(i-nbvertex_,1) - 1][1] + vertices[SetOfEdges_(i-nbvertex_,2) - 1][1]),
		       0.5 * (vertices[SetOfEdges_(i-nbvertex_,1) - 1][2] + vertices[SetOfEdges_(i-nbvertex_,2) - 1][2]));
	  for (int j=1; j<=GlobalDofNumber_; j++)
	    mat(i,j) = 0;
	  mat(i,i) = 1;
	  for (int k=1; k<=GlobalDofNumber_; k++)
	    if (k != i)
	      { F[k] += -mat(k,i) * F[i];
		mat(k,i) = 0;
	      }
	}
    }
}


/*!
  \param u exact solution
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \return error \f$||u-u_h||_{0,\Omega}\f$
*/

//=========================================================================
template <class T>
T p2c2d<T>::error_L2 (T (*u)(T,T), const vector<T>& uh, int NbIntPts) const
//=========================================================================
{
  T GlobalError = 0;
  T LocalError;
  T omegak, omegal, xk, xl, SumK;
  vector<T> Fxkl;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  vector<T> Basis(DofNumber_);

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { LocalError = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1-xk) * itg.point (l);
	      Basis = BasisFunction (xk, xl);
	      SumK = 0;
	      for (int i=1; i<=3; i++)
		SumK += uh [this -> number (element[num_element-1][i])] * Basis[i];
	      for (int i=4; i<=6; i++)
		SumK += uh [EdgesPerTriangle_(num_element, i-3) + nbvertex_] * Basis[i];
	      Fxkl = Fk (xk, xl, num_element);
	      LocalError += omegak * omegal * (1-xk) * pow (u (Fxkl[1], Fxkl[2]) - SumK, 2);
	    }
	}
      GlobalError += element[num_element-1].AbsDet() * LocalError;
    }

  return sqrt(GlobalError);
}


/*!
  \param dudx first derivative of the exact solution
  \param dudy second  derivative of the exact solution
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \return l'erreur \f$|u-u_h|_{0,\Omega}\f$
*/

//=================================================================================================
template <class T>
T p2c2d<T>::error_semi_H1 (T (*dudx)(T,T), T (*dudy)(T,T), const vector<T>& uh, int NbIntPts) const
//=================================================================================================
{
  T GlobalError = 0;
  T LocalError;
  T omegak, omegal, xk, xl;

  T duhdx, duhdy;
  T duhdxhat, duhdyhat;
  vector<T> Fxkl;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  matrix<T> DiffBasis (2, DofNumber_);
  matrix<T> InvMk;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { LocalError = 0;
      InvMk = element[num_element-1].InvMk();
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1 - xk) * itg.point (l);
	      DiffBasis = DiffBasisFunction (xk, xl);
	      duhdxhat = 0;
	      duhdyhat = 0;
	      for (int i=1; i<=3; i++)
		{ duhdxhat += uh [this -> number (element[num_element-1][i])] * DiffBasis (1,i);
		  duhdyhat += uh [this -> number (element[num_element-1][i])] * DiffBasis (2,i);
		}
	      for (int i=4; i<=6; i++)
		{ duhdxhat += uh [EdgesPerTriangle_(num_element,i-3) + nbvertex_ ] * DiffBasis (1,i);
		  duhdyhat += uh [EdgesPerTriangle_(num_element,i-3) + nbvertex_ ] * DiffBasis (2,i);
		}
	      duhdx = duhdxhat * InvMk (1,1) + duhdyhat * InvMk (2,1);
	      duhdy = duhdxhat * InvMk (1,2) + duhdyhat * InvMk (2,2);
	      Fxkl = Fk (xk, xl, num_element);
	      LocalError += omegak * omegal * (1-xk) * (pow (dudx (Fxkl[1], Fxkl[2]) - duhdx, 2) + pow (dudy (Fxkl[1], Fxkl[2]) - duhdy, 2));
	    }
	}
      GlobalError += LocalError * element[num_element-1].AbsDet();
    }

  return sqrt(GlobalError);
}


}


#endif
