/***************************************************************
 *             Finite Element Method Object Library            *
 *           class tetraRT0 : declaration for tetraRT0         *
 *                    simula.plus@cemes                        *
 *	             GNU/linux version 3.2.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2003,2004 CREUSE Emmanuel
 * copyright © 2004 PIRAUT Frédéric
 * copyright © 2011,2012 COLLARD Christophe
 * copyright © 2003,2004,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2003,2004 Arts et Métiers ParisTech
 * copyright © 2003,2004 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2003,2004 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2003,2004 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace femol
    \brief Finite Element Method Object Libraries
*/

/*! \class femol::tetraRT0
    \brief tetraRT0 library \n

    \htmlonly 
    <FONT color="#838383">

    tetraRT0 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2003, 2004 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2004 PIRAUT Frédéric \n
	     copyright \htmlonly &#169; \endhtmlonly 2011, 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2003, 2004, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	     copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2003-2012
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type tetraRT0
#endif

#ifndef __tetraRT0_hpp
#define __tetraRT0_hpp

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __spmatrix_hpp
#include "MOL++/spmatrix.hpp"
#endif

#ifndef __integration_hpp
#include "MOL++/integration.hpp"
#endif

#ifndef __tetrahedron_hpp
#include "FEMOL++/meshes/tetrahedron.hpp"
#endif

#ifndef __mesh_hpp
#include "FEMOL++/meshes/mesh.hpp"
#endif

using namespace std;
using namespace mol;

namespace femol
{


//===============================================================
template <class T> class tetraRT0 : public mesh<tetrahedron<T>,T>
//===============================================================
{
  using mesh<tetrahedron<T>,T>::alive;
  using mesh<tetrahedron<T>,T>::dimspace_;
  using mesh<tetrahedron<T>,T>::nbvertex_;
  using mesh<tetrahedron<T>,T>::nbelement_;
  using mesh<tetrahedron<T>,T>::vertices;
  using mesh<tetrahedron<T>,T>::element;

  private:
    int DofNumber_;
    int GlobalDofNumber_;

    // nbfaces_ is the number of faces of the mesh (equal to GlobalDofNumber)
    int nbfaces_;

    // face_of_element_(i,j) is the global number of the jth face of the ith tetrahedra (1<=i<=nbelement_, 1<=j<=4). The jth face of the element is in front of the jth node of the element (ie the jth node does not belong to the jth face).
    matrix<int> face_of_element_;

    // node_of_face_(i,j) is the global number of jth node of the ith face (1<=i<=nbfaces_, 1<=j<=3). These numbers are ordered from the smallest to the biggest.
    matrix<int> node_of_face_;

  // for each face, the unit normal n_i (1<=i<=nbfaces_) is defined, in the direct sens with respect to the three nodes of the face node_of_face_(i,:). direction_of_normal(i,j) is equal to 1 if the unit normal of face_of_element_(i,j) previously defined goes outside the ith tetrahedra, and to -1 if the unit normal of face_of_element_(i,j) previously defined goes inside the ith tetrahedra.
    matrix<int> direction_of_normal_;

    vector<int> type_of_face_; // type_of_face_[i]=0 if internal face, 0 if not (1<=i<=nbfaces_)

    // 27/01/04 : The following descriptors are needed for a posteriori estimators, but not for the computation of the solution for each face i, the components of the unit normal n_i are in the descriptor normal_(i,1:3).
    matrix<T> normal_;

    // for each face i, the elements (two if the face is internal and one if not) that  contain the face i are mentionned in the descriptor element_of_face_(i,1:2)
    matrix<int> element_of_face_;

    // for each tetrahedron i, the jth neighboored tetrahedra (j<=4) is  mentionned in voisins_tetra_(i,j). For a given tetrahedron i, the total number of neighboored tetrahedra is in n_vois_tetra_(i).
    matrix<int> voisins_tetra_;
    vector<int> n_vois_tetra_;

    // for anisotropic error estimators : we need to define some anisotropic parameter
    vector<T> hminT_;
    vector<T> hE_;  

  public:
    tetraRT0 ();
    tetraRT0 (const char*, const char* = "");
    tetraRT0 (const tetraRT0<T>&);
    tetraRT0<T>& operator = (const tetraRT0<T>&); //tetraRT0 affectation

    /*! \return the number of node of the object */
    int DofNumber () const {return DofNumber_;}
    /*! \return the number of node of meshes */
    int GlobalDofNumber () const {return GlobalDofNumber_;}
    /*! \return the faces number of mesh */
    int nbfaces () const {return nbfaces_;}
    /*! \return global number of faces of each tetrahedra */
    matrix<int> face_of_element () const {return face_of_element_;}

    /*! \return global number of nodes of each face */
    matrix<int> node_of_face () const {return node_of_face_;}

    /*! \return matrix(nbelement_,4) with the direction_of_normal_ of ech faces */
    matrix<int> direction_of_normal () const {return direction_of_normal_;}

    /*! \return the type of each face: 1 if face is on the boundary , 0 otherwise) */
    vector<int> type_of_face () const {return type_of_face_;}
    matrix<T> normal () const {return normal_;}
    matrix<int> element_of_face () const {return element_of_face_;}
    matrix<int> voisins_tetra () const {return voisins_tetra_;}
    vector<int> n_vois_tetra () const {return n_vois_tetra_;}
    vector<T> hminT () const {return hminT_;}
    vector<T> hE () const {return hE_;}
    matrix<T> BasisFunction (T, T, T) const;
    matrix<T> LocalMassMatrix (int) const;
    //  int boundary(vector<int>& numero_des_noeuds,const char*geometrie)const;
    virtual int operator () (int, int) const;
    template <class Tf> friend bool operator == (const tetraRT0<Tf>&, const tetraRT0<Tf>&);
    template <class Tf> friend bool operator != (const tetraRT0<Tf>&, const tetraRT0<Tf>&);
    vector<T> Fk (T, T, T, int) const;
    vector<T> Fkm1 (T, T, T, int) const;
    vector<T> vect (const vector<T>&, const vector<T>&) const; // vectorial product (this function should be developped in the vector.h class, but the problem is to find a meaning if the vectors are not of dimension 3...). For the moment, it is so developped in the class tetraRT0.h
    spmatrix<T> UV () const;
    vector<T> B_PB_MIXTE (const vector<T>&) const;
    T error_L2 (vector<T>(*u)(T,T,T), const vector<T>&, int) const;
    T error_div (T (*divu)(T,T,T), const vector<T>&, int) const;
    T error_L2_loc_2 (vector<T> (*u)(T,T,T), const vector<T>&, int, int) const;
    T error_div_loc_2 (T (*divu)(T,T,T), const vector<T>&, int, int) const;
    void DirichletHom (matrix<T>&) const;
};


//=====Private methods for tetraRT0==========================================


//=====Public methods for tetraRT0===========================================


/*!
  \brief default constructor of the tetraRT0 class
*/

//==================================================
template <class T> 
tetraRT0<T>::tetraRT0 () : mesh<tetrahedron<T>,T> ()
//==================================================
{
  DofNumber_ = GlobalDofNumber_ = nbfaces_ = 0;
}


/*!
  \brief constructor of the tetraRT0 class. The arguments are 2 files:file1 and file2.numerotation of each element. file1 indicates the dimension (2 or 3), the number of nodes and each node coordinate with a boundary condition (1 if face is on the boundary , 0 otherwise).file2 must be documented by the number of elements,node's numerotation of each tetrahedron.
  \param file1 string
  \param file2 string (if not merged with file1)
*/

//==================================================================================================
template <class T>
tetraRT0<T>::tetraRT0 (const char* file1, const char* file2) : mesh<tetrahedron<T>,T> (file1, file2)
//==================================================================================================
{
  DofNumber_ = 4;
  int num_face;
  int found;
  int num_neu_in_front_of_face;
  vector<int> num_neu_face (3);
  face_of_element_ = matrix<int> (nbelement_, 4);
  int nb_min_faces = nbvertex_ + 2 * (nbelement_ - 1);
  vector<int> head (nbvertex_);
  vector<int> next (nb_min_faces);
  matrix<int> hyp_node_of_face_ (nb_min_faces, 3);
  direction_of_normal_ = matrix<int> (nbelement_, 4);
  vector<T> vecteur_normal (3);
  int tetra1, tetra2;
  matrix<T> hyp_normal_ (nb_min_faces, 3);
  matrix<int> hyp_element_of_face_ (nb_min_faces, 2);
  vector<int> hyp_type_of_face_ (nb_min_faces);
  voisins_tetra_ = matrix<int> (nbelement_, 4);
  n_vois_tetra_ = vector<int> (nbelement_);
  for (int i=1; i<=nb_min_faces; i++)
    hyp_type_of_face_[i] = 1;
  num_face = 0;
  for (int i=1; i<=nbelement_; i++)
    for (int j=1; j<=4; j++)
      { for (int k=1; k<=3; k++)
	  num_neu_face[k] = (*this).number ((*this)[i][1+(j+k-1)%4]);
	num_neu_in_front_of_face = (*this).number ((*this)[i][j]);
	order(num_neu_face);
	int exist = 0;
	for (int e=head[num_neu_face[1]]; e!=0; e=next[e])
	  if ((hyp_node_of_face_(e,2) == num_neu_face[2]) && (hyp_node_of_face_(e,3) == num_neu_face[3]))
	    { exist = 1;
	      face_of_element_(i,j) = e;
	      hyp_type_of_face_[e] = 0;
	      if (vect (vertices[num_neu_face[2]-1] - vertices[num_neu_face[1]-1], vertices[num_neu_face[3]-1] - vertices[num_neu_face[1]-1]) * (vertices[num_neu_in_front_of_face-1] - vertices[num_neu_face[1]-1]) > 0)
		direction_of_normal_(i,j) = -1;
	      else
		direction_of_normal_(i,j) = 1;
	    }
	if (!exist)
	  { num_face++;
	    for (int l=1; l<=3; l++)
	      hyp_node_of_face_(num_face,l) = num_neu_face[l];
	    next[num_face] = head[num_neu_face[1]];
	    face_of_element_(i,j) = num_face;
	    head[num_neu_face[1]] = num_face;
	    if (vect (vertices[num_neu_face[2]-1] - vertices[num_neu_face[1]-1], vertices[num_neu_face[3]-1] - vertices[num_neu_face[1]-1]) * (vertices[num_neu_in_front_of_face-1] - vertices[num_neu_face[1]-1]) > 0)
	      direction_of_normal_(i,j) = -1;
	    else
	      direction_of_normal_(i,j) = 1;
	  }
      }
  nbfaces_ = num_face;
  GlobalDofNumber_ = nbfaces_;
  type_of_face_ = vector<int> (nbfaces_);
  node_of_face_ = matrix<int> (nbfaces_, 3);
  normal_ = matrix<T> (nbfaces_, 3);
  element_of_face_ = matrix<int> (nbfaces_, 2);
  for (int i=1; i<=nbfaces_; i++)
    { for (int j=1; j<=3; j++)
	{ node_of_face_(i,j) = hyp_node_of_face_(i,j);
	  normal_(i,j) = hyp_normal_(i,j);
	}
      for (int j=1; j<=2; j++)
	element_of_face_(i,j) = hyp_element_of_face_(i,j);

      if (!element_of_face_(i,1))
	{ element_of_face_(i,1) = element_of_face_(i,2);
	  element_of_face_(i,2) = 0;
	}
      type_of_face_[i] = hyp_type_of_face_[i];
    }

  for (int i=1; i<=nbfaces_; i++)
    { tetra1 = element_of_face_(i,1);
      tetra2 = element_of_face_(i,2);

     if (tetra2)
       { n_vois_tetra_[tetra1]++;
	 n_vois_tetra_[tetra2]++;
	 voisins_tetra_ (tetra1, n_vois_tetra_[tetra1]) = tetra2;
       voisins_tetra_ (tetra2, n_vois_tetra_[tetra2]) = tetra1;
       }
    }

  // Anisotropic quantities...
  hminT_ = vector<T> (nbelement_);
  hE_ = vector<T> (nbfaces_);
  T volume_tetra, Surface_Face_max, Surface_Face;
  int n_face;
  vector<T> VecA1(3), VecA2(3), VecA3(3);
  for (int num_element=1; num_element<=nbelement_; num_element++)
    { // calcul du volume du tetrahedron
      volume_tetra = element[num_element-1].AbsDet() / 6.;
      Surface_Face_max = 0;
      for (int j=1; j<=4; j++)
	{ // calcul de la surface de la face
	  n_face = face_of_element_ (num_element, j);

	  VecA1 = vertices[node_of_face_(n_face,1)-1];
	  VecA2 = vertices[node_of_face_(n_face,2)-1];
	  VecA3 = vertices[node_of_face_(n_face,3)-1];

	  Surface_Face = vect (VecA2-VecA1, VecA3 - VecA1).norm() / 2.;

	  if (Surface_Face > Surface_Face_max) Surface_Face_max = Surface_Face;

	  if (type_of_face_[n_face])
	    hE_[n_face] = 3 * volume_tetra / Surface_Face;
	  else
	    hE_[n_face] += 3 * volume_tetra / (2 * Surface_Face);
	}
      hminT_[num_element] = 3 * volume_tetra / Surface_Face_max;
    }
  // End of Anisotropic quantities
}


/*!
  \brief copy constructor of the tetraRT0 class
  \param EF reference on tetraRT0 object
*/

//=========================================================================
template <class T> 
tetraRT0<T>::tetraRT0 (const tetraRT0<T>& EF) : mesh<tetrahedron<T>,T> (EF)
//=========================================================================
{
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  nbfaces_ = EF.nbfaces_;
  face_of_element_ = matrix<int> (EF.face_of_element_);
  node_of_face_ = matrix<int> (EF.node_of_face_);
  direction_of_normal_ = matrix<int> (EF.direction_of_normal_);
  type_of_face_ = vector<int> (EF.type_of_face_);
}


/*!
  \brief affectation operator for the tetraRT0 class.
  \param EF reference on tetraRT0 object.
  \param reference on tetraRT0 object.
*/

//==========================================================
template <class T>
tetraRT0<T>& tetraRT0<T>::operator = (const tetraRT0<T>& EF)
//==========================================================
{
  (*this).mesh <tetrahedron<T>,T>::operator = (EF);
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  nbfaces_ = EF.nbfaces_;
  face_of_element_ = EF.face_of_element_;
  node_of_face_ = EF.node_of_face_;
  direction_of_normal_ = EF.direction_of_normal_;
  type_of_face_ = EF.type_of_face_;
  return (*this);
}


/*!
  \param xref type T
  \param yref type T
  \param zref type T
  \return matrix container the basis functions of the reference element tetraRT0.
*/

//=================================================================
template <class T>
matrix<T> tetraRT0<T>::BasisFunction (T xref, T yref, T zref) const
//=================================================================
{
  matrix<T> P (DofNumber_,3);
  P(1,1) = 2 * xref;        P(1,2) = 2 * yref;       P(1,3) = 2 * zref;
  P(2,1) = 2 * (xref - 1);  P(2,2) = 2 * yref;       P(2,3) = 2 * zref;
  P(3,1) = 2 * xref;        P(3,2) = 2 * (yref - 1); P(3,3) = 2 * zref;
  P(4,1) = 2 * xref;        P(4,2) = 2 * yref;       P(4,3) = 2 * (zref - 1);

  return P;
}


/*!
  \param num_element element's number
  \return local mass matrix of the element num_element
*/

//============================================================
template <class T>
matrix<T> tetraRT0<T>::LocalMassMatrix (int num_element) const
//============================================================
{
// ici on a choisi 5 points d'intégration ce qui est trop important compte tenu du degré des fonctions à integrer... A faire baisser à la valeur tout juste suffisante.

  int NbIntPts = 5;
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  assert ((num_element >= 1) && (num_element <= nbelement_));
  matrix<T> matloc (4,4);
  matrix<T> B = t(element[num_element-1].Mk()) * element[num_element-1].Mk() / element[num_element-1].AbsDet();
  T valeurint;
  T xk, yj, zi, omegai, omegaj, omegak;
  vector<T> phim(3), phip(3);
  matrix<T> valeurfonctions (4,3);

  for (int m=1; m<=4; m++)
    { for (int p=1; p<=4; p++)
      { valeurint = 0;
        for (int k=1; k<=NbIntPts; k++)
	  { omegak = itg.weight (k);
	    xk = itg.point (k);
	    for (int j=1; j<=NbIntPts; j++)
	      { omegaj = itg.weight (j);
		yj = itg.point (j);
		for (int i=1; i<=NbIntPts; i++)
		  { omegai = itg.weight (i);
		    zi = itg.point (i);
		    valeurfonctions = BasisFunction (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi);
		    for (int l=1; l<=3; l++)
		      { phim[l] = valeurfonctions (m,l);
			phip[l] = valeurfonctions (p,l);
		      }
		    valeurint += omegai * omegaj * omegak * abs ((1 - xk) * (1 - xk - yj + xk * yj)) * ((B * phim) * phip);
		  }
	      }
	  }
	matloc (m,p) = direction_of_normal_(num_element, m) * direction_of_normal_(num_element, p) * valeurint;
      }
    }

  return matloc;
}


/*!
   \param num_element element's number
   \param j the j-th face
   \return global number of the j-th of the element num_element
*/

//=========================================================
template <class T>
int tetraRT0<T>::operator () (int num_element, int j) const
//=========================================================
{
  assert ((j >= 1) && (j <= DofNumber_) && (num_element >= 1) && (num_element <= nbelement_));
  return face_of_element_(num_element, j);
}


/*!
   \param EF1 type const tetraRT0<T>&
   \param EF2 type const tetraRT0<T>&
   \return 1 if EF1 and EF2 are equal, 0 otherwise
*/

//=================================================================
template <class Tf>
bool operator == (const tetraRT0<Tf>& EF1, const tetraRT0<Tf>& EF2)
//=================================================================
{
  bool boolean= true;
  boolean *= (mesh<tetrahedron<Tf>,Tf> (EF1) == mesh<tetrahedron<Tf>,Tf> (EF2));
  boolean *= (EF1.DofNumber_ == EF2.DofNumber_);
  boolean *= (EF1.GlobalDofNumber_ == EF2.GlobalDofNumber_);
  boolean *= (EF1.nbfaces_ == EF2.nbfaces_);
  boolean *= (EF1.face_of_element_ == EF2.face_of_element_);
  boolean *= (EF1.node_of_face_ == EF2.node_of_face_);
  boolean *= (EF1.direction_of_normal_ == EF2.direction_of_normal_);
  boolean *= (EF1.type_of_face_ == EF2.type_of_face_);

  return boolean;
}


/*!
  \param EF1 type const tetraRT0<T>&
  \param EF2 type const tetraRT0<T>&
  \return 1 if EF1 and EF2 are different, 0 otherwise
*/

//=================================================================
template <class Tf>
bool operator != (const tetraRT0<Tf>& EF1, const tetraRT0<Tf>& EF2)
//=================================================================
{
  return !(EF1==EF2);
}


/*
  Emmanuel Creusé
  060104 : Cette fonction n'est plus utilisée car pour déterminer si une face est frontalière ou pas, on compte le nombre de fois qu'elle a été repérée dans le maillage. Mais on la laisse, car il pourrait être utile d'écrire quelquechose de ce type si l'on désire appliquer des conditions aux limites différentes sur des parties distinctes de la frontière d'un polyhèdre.
//==================================================================================
template <class T> 
int tetraRT0<T>::boundary(vector<int>& numero_des_noeuds,const char*geometrie) const
//==================================================================================
{
  int frontalier=1;
  if (geometrie=="cubeunite")
    {
      node<T> A(vertices[numero_des_noeuds[1]-1]);
      node<T> B(vertices[numero_des_noeuds[2]-1]);
      node<T> C(vertices[numero_des_noeuds[3]-1]);
      // a necessary condition for the face to be on the boundary is that
      // each of its node is on the boundary :
      frontalier*=A.boundary_condition()*B.boundary_condition()*C.boundary_condition();
      if (frontalier)
	{ 
          for (int k=1; k<=3; k++)
	    {
              if ((abs(A[k])<epsilon)&&(abs(B[k])<epsilon)&&(abs(C[k])<epsilon))
	       return 1;
	      else if ((abs(A[k]-1.0)<epsilon)&&(abs(B[k]-1.0)<epsilon)&&(abs(C[k]-1.0)<epsilon))
	       return 1;
            }
          return 0;
	} 
    }

  else
    {
      cout << "quelle est la géométrie ?";
      assert(geometrie=="cubeunite");
    }
}
*/


/*!
   \param x type T
   \param y type T
   \param z type T
   \param i type integer
   \return vector by the application Fk (reference element -> i-th element)
*/

//====================================================
template <class T>
vector<T> tetraRT0<T>::Fk (T x, T y, T z, int i) const
//====================================================
{
  matrix<T> MAT (3,3);
  vector<T> A1A2(3), A1A3(3), A1A4(3), A1(3), VEC(3);
  VEC[1] = x;
  VEC[2] = y;
  VEC[3] = z;

  A1 = vector<T> (element[i-1][1]);

  A1A2 = vector<T> (element[i-1][2]) - A1;
  A1A3 = vector<T> (element[i-1][3]) - A1;
  A1A4 = vector<T> (element[i-1][4]) - A1;

  MAT(1,1) = A1A2[1]; MAT(2,1) = A1A2[2]; MAT(3,1) = A1A2[3];
  MAT(1,2) = A1A3[1]; MAT(2,2) = A1A3[2]; MAT(3,2) = A1A3[3];
  MAT(1,3) = A1A4[1]; MAT(2,3) = A1A4[2]; MAT(3,3) = A1A4[3];

  return MAT * VEC + A1;
}


/*!
  \param x type T
  \param y type T
  \param z type T
  \param i type integer
  \return the vector by the application Fkm1=(Fk)^(-1)
*/

//======================================================
template <class T>
vector<T> tetraRT0<T>::Fkm1 (T x, T y, T z, int i) const
//======================================================
{
  matrix<T> MAT (3,3);
  vector<T> A1A2(3), A1A3(3), A1A4(3), A1(3), VEC(3);
  VEC[1] = x;
  VEC[2] = y;
  VEC[3] = z;

  A1 = vector<T> (element[i-1][1]);

  A1A2 = vector<T> (element[i-1][2]) - A1;
  A1A3 = vector<T> (element[i-1][3]) - A1;
  A1A4 = vector<T> (element[i-1][4]) - A1;

  MAT(1,1) = A1A2[1]; MAT(2,1) = A1A2[2]; MAT(3,1) = A1A2[3];
  MAT(1,2) = A1A3[1]; MAT(2,2) = A1A3[2]; MAT(3,2) = A1A3[3];
  MAT(1,3) = A1A4[1]; MAT(2,3) = A1A4[2]; MAT(3,3) = A1A4[3];

  return MAT.inv() * (VEC - A1);
}


/*!
  \param a type const<T>&
  \param b type const<T>&
  \return vector type T equal to the vectorial produit of a and b
*/
//========================================================================
template <class T>
vector<T> tetraRT0<T>::vect (const vector<T>& a, const vector<T>& b) const
//========================================================================
{
  assert ((a.dim()) && (a.dim() == b.dim()));
  vector<T> c;
  c = a.dim(); // allocates memory without initializing the vector

  for (int i=1; i<=a.dim()-2 ; i++)
    c[i] = a[i+1] * b[i+2] - a[i+2] * b[i+1];
  c[a.dim()-1] = a[a.dim()] * b[1] - a[1] * b[a.dim()];
  c[a.dim()] = a[1] * b[2] - a[2] * b[1];

  return c;
}


/*!
  \return the global matrix of the tetraRT0 element
*/

//==================================
template <class T>
spmatrix<T> tetraRT0<T>::UV () const
//==================================
{
  matrix<T> Mass (DofNumber_, DofNumber_);
  spmatrix<T> UV_ (GlobalDofNumber_, GlobalDofNumber_, 7);
  for (int numele=1; numele<=nbelement_; numele++)
    { Mass = LocalMassMatrix (numele);
      for (int i=1; i<=DofNumber_; i++)
	for (int j=1; j<=DofNumber_; j++)
	  UV_ (face_of_element_(numele,i), face_of_element_(numele,j)) += Mass(i,j);
    }

  return UV_;
}


/*!
  \param beta type const vector<T>&
  \return the second member
*/

//=============================================================
template <class T>
vector<T> tetraRT0<T>::B_PB_MIXTE (const vector<T>& beta) const
//=============================================================
{
  assert (beta.dim() && (beta.dim() == nbelement_));
  vector<T> B_ (GlobalDofNumber_);
  for (int numele=1; numele<=nbelement_; numele++)
    { for (int i=1; i<=DofNumber_; i++)
	B_[face_of_element_(numele,i)] -= beta[numele] * direction_of_normal_(numele,i);
    }

  return B_;
}


/*!
  \param u exact solution
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \return error_L2
*/

//======================================================================================
template <class T>
T tetraRT0<T>::error_L2 (vector<T> (*u)(T,T,T), const vector<T>& uh, int NbIntPts) const
//======================================================================================
{
  T GlobalError = 0;
  vector<T> LocalError2 (3);
  vector<T> Somme (3);
  matrix<T> Mk(3,3), tMk(3,3);
  T Determinant;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T xk, yj, zi, omegak, omegaj, omegai;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { Mk = element[num_element-1].Mk();
      tMk = t(Mk);
      Determinant = element[num_element-1].AbsDet();
      LocalError2[1] = LocalError2[2] = LocalError2[3] = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int j=1; j<=itg.NbIntPts(); j++)
	    { omegaj = itg.weight (j);
	      yj = itg.point (j);
	      for (int i=1; i<=itg.NbIntPts(); i++)
		{ omegai = itg.weight (i);
		  zi = itg.point (i);
		  Somme[1] = Somme[2]= Somme[3] = 0;
		  for (int m=1; m<=3; m++)
		    { for (int l=1; l<=4; l++)
			Somme[m] += uh[face_of_element_(num_element,l)] * (BasisFunction (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi) * tMk)(l,m) * direction_of_normal_(num_element,l) / Determinant;
		      LocalError2[m] += omegai * omegaj * omegak * abs ((1 - xk) * (1 - xk - yj + xk * yj)) * pow (u (Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[1],
														      Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[2],
														      Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[3])[m] - Somme[m], 2);
		    }
		}
	    }
	}
      GlobalError += Determinant * (LocalError2[1] + LocalError2[2] + LocalError2[3]);
    }

  return sqrt(GlobalError);
}


/*!
  \param u exact solution
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \param num_element element's number
  \return error_L2_loc_2
*/

//=============================================================================================================
template <class T>
T tetraRT0<T>::error_L2_loc_2 (vector<T> (*u)(T,T,T), const vector<T>& uh, int NbIntPts, int num_element) const
//=============================================================================================================
{
  vector<T> LocalError2 (3);
  vector<T> Somme (3);
  matrix<T> Mk(3,3), tMk(3,3);
  T  Determinant;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T xk, yj, zi, omegak, omegaj, omegai;

  Mk = element[num_element-1].Mk();
  tMk = t(Mk);
  Determinant = element[num_element-1].AbsDet();

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int j=1; j<=itg.NbIntPts(); j++)
	{ omegaj = itg.weight (j);
	  yj = itg.point (j);
	  for (int i=1; i<=itg.NbIntPts(); i++)
	    { omegai = itg.weight (i);
	      zi = itg.point (i);
	      Somme[1] = Somme[2] = Somme[3] = 0;
	      for (int m=1; m<=3; m++)
		{ for (int l=1; l<=4; l++)
		    Somme[m] += uh[face_of_element_(num_element,l)] * (BasisFunction (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi)*tMk)(l,m) * direction_of_normal_(num_element,l) / Determinant;
		  LocalError2[m] += omegai * omegaj * omegak * abs ((1-xk) * (1-xk-yj+xk*yj)) * pow (u (Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[1],
													Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[2],
													Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[3])[m] - Somme[m], 2);
		}
	    }
	}
    }

  return Determinant * (LocalError2[1] + LocalError2[2] + LocalError2[3]);
}


/*!
  \param divu
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \return error_div
*/

//==================================================================================
template <class T>
T tetraRT0<T>::error_div (T (*divu)(T,T,T), const vector<T>& uh, int NbIntPts) const
//==================================================================================
{
  T GlobalError = 0;
  T LocalError2;
  T Somme;
  T Determinant;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T xk, yj, zi, omegak, omegaj, omegai;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { Determinant = element[num_element-1].AbsDet();
      LocalError2 = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int j=1; j<=itg.NbIntPts(); j++)
	    { omegaj = itg.weight (j);
	      yj = itg.point (j);
	      for (int i=1; i<=itg.NbIntPts(); i++)
		{ omegai = itg.weight (i);
		  zi = itg.point (i);
		  Somme = 0;
		  for (int l=1; l<=4; l++)
		    Somme += uh[face_of_element_(num_element,l)] * direction_of_normal_(num_element,l);
		  LocalError2 += omegai * omegaj * omegak * abs ((1-xk) * (1-xk-yj+xk*yj)) * pow (divu (Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[1],
													Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[2],
													Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[3]) - 6 * Somme / Determinant, 2);
		}
	    }
	}
      GlobalError += Determinant * LocalError2;
    }

  return sqrt (GlobalError);
}


/*!
  \param divu
  \param uh vector approximation solution
  \param NbIntPts number of points of the integration
  \param num_element element's number
  \return error_div_loc_2
*/

//=========================================================================================================
template <class T>
T tetraRT0<T>::error_div_loc_2 (T (*divu)(T,T,T), const vector<T>& uh, int NbIntPts, int num_element) const
//=========================================================================================================
{
  T LocalError2 = 0;
  T Somme;
  T Determinant;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  T xk, yj, zi, omegak, omegaj, omegai;

  Determinant = element[num_element-1].AbsDet();

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int j=1; j<=itg.NbIntPts(); j++)
	{ omegaj = itg.weight (j);
	  yj = itg.point (j);
	  for (int i=1; i<=itg.NbIntPts(); i++)
	    { omegai = itg.weight (i);
	      zi = itg.point (i);
	      Somme = 0;
	      for (int l=1; l<=4; l++)
		Somme += uh[face_of_element_(num_element,l)] * direction_of_normal_(num_element,l);
	      LocalError2 += omegai * omegaj * omegak * abs ((1-xk) * (1-xk-yj+xk*yj)) * pow (divu (Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[1],
												    Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[2],
												    Fk (xk, (1-xk)*yj, (1-xk-yj+xk*yj)*zi, num_element)[3]) - 6 * Somme / Determinant, 2);
	    }
	}
    }

  return Determinant * LocalError2;
}


/*!
  \param MAT type matrix
  \return matrix with DirichletHom conditions
*/

//===================================================
template <class T>
void tetraRT0<T>::DirichletHom (matrix<T>& MAT) const
//===================================================
{
  for (int i=1; i<=GlobalDofNumber_; i++)
    if (type_of_face_[i] == 1)
      { for (int j=1; j<=GlobalDofNumber_; j++)
	  { MAT(i,j) = 0;
	    MAT(j,i) = 0;
	  }
	MAT(i,i) = 1;
      }
}


}


#endif
