/***************************************************************
 *             Finite Element Method Object Library            *
 *      class convert : declaration for conversion tools       *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.2.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2003,2004,2005,2006 CREUSE Emmanuel
 * copyright © 2003 MERCIER Denis
 * copyright © 2011,2012 COLLARD Christophe
 * copyright © 2003,2004,2005,2006,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2003,2004,2005,2006 Arts et Métiers ParisTech
 * copyright © 2003,2004,2005,2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2003,2004,2005,2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2003,2004,2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace femol
    \brief Finite Element Method Object Libraries
*/

/*! \class femol::convert
    \brief convert library \n

    \htmlonly 
    <FONT color="#838383">

    convert belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2003, 2004, 2005, 2006 Emmanuel CREUSE \n
	     copyright \htmlonly &#169; \endhtmlonly 2003 Denis MERCIER \n
	     copyright \htmlonly &#169; \endhtmlonly 2011, 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	     copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2003-2012
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type convert
#endif

#ifndef __convert_hpp
#define __convert_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __parameters_h
#include "parameters.h"
#endif

#ifndef __colors_hpp
#include "colors.hpp"
#endif


using namespace std;
using namespace mol;

namespace femol
{


//===========
class convert
//===========
{
  public :
    template <class Tf> friend bool lshape_boundary2d (Tf, Tf, Tf, Tf, Tf, Tf);  // return yes if the specified node belongs to the boundary
    template <class Tf> friend bool circle_boundary2d (Tf, Tf, Tf, Tf, Tf);      // return yes if the specified node belongs to the boundary
    template <class Tf> friend bool threeonfourpartofcircle_boundary2d (Tf, Tf, Tf);  // return yes if the specified node belongs to the boundary
    template <class Tf> friend bool square_boundary2d (Tf, Tf, Tf);  // return yes if the specified node belongs to the boundary
    template <class Tf> friend bool halfdisk_boundary2d (Tf, Tf);    // return yes if the specified node belongs to the boundary
    template <class Tf> friend bool fourthdisk_boundary2d (Tf, Tf);  // return yes if the specified node belongs to the boundary
    template <class Tf> friend void make_conversion3dtetra (const char*, const char*);  // to convert 3D tetrahedra meshes from NETGEN
    template <class Tf> friend void make_conversion2dtri (const char*, const char*, const char*);  // to convert 2D triangles meshes from NETGEN
};


//=====Private  methods for convert==========================================


//=====Public  methods for convert===========================================


/*!
  \brief Characterizes a node located on the boundary of an L-Shape
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param xleft x-coordinate of the left side of the L-Shape \n
  \param xright x-coordinate of the right side of the L-Shape \n 
  \param ybottom y-coordinate of the bottom side of the L-Shape \n 
  \param ytop y-coordinate of the top side of the L-Shape \n
  \return rep 1 if it is on the boundary, 0 else.
*/

//---------------------------------------------------------------------------
template <class Tf>
bool lshape_boundary2d (Tf x, Tf y, Tf xleft, Tf xright, Tf ybottom, Tf ytop)
//---------------------------------------------------------------------------
{
  Tf eps = 1e-06;

  bool rep = (  (abs (x - xleft)   < eps)
	     || (abs (x - xright)  < eps)
	     || (abs (y - ytop)    < eps)
	     || (abs (y - ybottom) < eps)
	     || ((abs (x) < eps) && (y < eps))
	     || ((abs (y) < eps) && (x > eps)));

  return rep;
}


/*!
  \brief Characterizes a node located on the boundary of a disk
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param centerx x-coordinate of the circle center  \n
  \param centery y-coordinate of the circle center  \n
  \param radius radius of the circle \n 
  \return rep 1 if it is on the boundary, 0 else.
  \warning Before January, 4 2006, the name of this function was  "unitcircle_boundary2d"
  \warning Pour cette fonction, il faut prendre garde à ce que le fichier de sortie Netgen, généré
	   lui même à partir d'un fichier .in2d,  NE POSSEDE PAS D'AUTRES NOEUDS SUR LA FRONTIERE QUE CEUX
	   MENTIONNES DANS LE FICHIER .in2d LUI MEME. Sinon, elle ne caractérisera pas tous les noeuds de la frontière
	   (cad ceux qui seraient situés du coup sur la corde et non sur l'arc...) Remarque : Tous les maillages
	   fournis dans les répertoire du code simula+ respectent cette propriété !
*/

//--------------------------------------------------------------------
template <class Tf>
bool circle_boundary2d (Tf x, Tf y, Tf centerx, Tf centery, Tf radius)
//--------------------------------------------------------------------
{
  Tf eps = 1e-06;

  bool rep = (abs (sqrt (pow ((x - centerx), 2) + pow ((y - centery), 2)) - radius) < eps);

  return rep;
}


/*!
  \brief Characterizes a node located on the boundary of a 3/4 part of a disk which center is (0,0)
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param radius radius of the 3/4 circle \n 
  \return rep 1 if it is on the boundary, 0 else.
  \warning Before January, 4 2006, the name of this function was  "partofunitcircle_boundary2d"
  \warning Pour cette fonction, il faut prendre garde à ce que le fichier de sortie Netgen, généré
	   lui même à partir d'un fichier .in2d,  NE POSSEDE PAS D'AUTRES NOEUDS SUR LA FRONTIERE QUE CEUX
	   MENTIONNES DANS LE FICHIER .in2d LUI MEME. Sinon, elle ne caractérisera pas tous les noeuds de la frontière
	   (cad ceux qui seraient situés du coup sur la corde et non sur l'arc...) Remarque : Tous les maillages
	   fournis dans les répertoire du code simula+ respectent cette propriété !
*/

//-------------------------------------------------------------
template <class Tf>
bool threeonfourpartofcircle_boundary2d (Tf x, Tf y, Tf radius)
//-------------------------------------------------------------
{
  Tf eps = 1e-06;

  bool rep = (abs (sqrt (pow (x, 2) + pow (y, 2)) - radius) < eps) || ((abs (x) < eps) && (y < 0)) || ((abs (y) < eps) && (x > 0));

  return rep;
}


/*!
  \brief Characterizes a node located on the boundary of a square [-a,a]*[-a,a]
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param a half side of the square \n
  \return rep 1 if it is on the boundary, 0 else.
*/

//---------------------------------------
template <class Tf>
bool square_boundary2d (Tf x, Tf y, Tf a)
//---------------------------------------
{
  Tf eps = 1e-06;

  bool rep = (abs ((abs (x) - a)) < eps) || (abs (abs (y) - a) < eps);

  return rep;
}


//-----------------------------------
template <class Tf>
bool halfdisk_boundary2d (Tf x, Tf y)
//-----------------------------------
{
  Tf eps = 1e-06;

  bool rep = (abs(x) < eps) || (abs (sqrt (pow (x, 2) + pow (y, 2)) - 1) < eps);

  return rep;
}


//-------------------------------------
template <class Tf>
bool fourthdisk_boundary2d (Tf x, Tf y)
//-------------------------------------
{
  Tf eps = 1e-06;

  bool rep = (abs (x) < eps) || (abs (y) <eps) || (abs (sqrt (pow (x, 2) + pow (y, 2)) - 1.0) < eps);

  return rep;
}


/*!
  \brief Convert an output file generated by Netgen (Mesh made of tetrahedra) in an
  input file for simula+
  \n \n
  \param name of the input_file file generated by Netgen
  \param name of the output_file to be read by the constructor in Simula+
  \warning Before January, 4 2006, the name of this function was only "make_conversion"
*/

//---------------------------------------------------------------------------
template <class Tf>
void make_conversion3dtetra (const char* input_file, const char* output_file)
//---------------------------------------------------------------------------
{
  // NETGEN file opening
  ifstream in_file (input_file);
  assert (!in_file.fail());
  // opening of the output file : if it does not exit, it is created. Else, it is overwritten.
  ofstream out_file (output_file, ios::trunc);
  assert (!out_file.fail());

  int number_nodes, number_tetra,number_faces;

  in_file >> number_nodes;
  matrix <Tf> coordinates (number_nodes, 3);
  vector <int> frontier (number_nodes);
  for (int i=1; i<=number_nodes; i++)
    for (int j=1; j<=3; j++)
      in_file >> coordinates (i, j);

  in_file >> number_tetra;
  matrix<int> tetra_nodes (number_tetra, 4);
  int x;
  for (int i=1; i<=number_tetra; i++)
    { in_file >> x;
      for (int j=1; j<=4; j++)
	in_file >> tetra_nodes (i,j);
    }

  in_file >> number_faces;
  for (int i=1; i<= number_faces; i++)
    { in_file >> x;
      for (int j=1; j<=3; j++)
	{ in_file >> x;
	  frontier[x] = 1;
	}
    }

  in_file.close();

  //writing in FORMAT3D
  out_file << 3 << endl;
  out_file << number_nodes << endl;

  out_file.precision (15);
  //out_file.setf(ios::scientific);
  for (int i=1; i<= number_nodes; i++)
    { for (int j=1; j<=3; j++)
	out_file << coordinates (i,j) << "   ";
      out_file << frontier[i] << endl;
    }

  out_file << number_tetra << endl;

  for (int i=1; i<=number_tetra; i++)
    { for (int j=1; j<=4; j++)
	out_file << setw(4) << tetra_nodes (i,j) << "   ";
      out_file << endl;
    }
}


/*!
  \brief Convert an output file generated by Netgen (Mesh made of triangles) in an
  input file for simula+
  \n \n
  \param name of the input_file file generated by Netgen \n
  \param name of the output_file to be read by the constructor in Simula+ \n
  \param domain : The type of domain, to characterize the boundary nodes, which are not given by NETGEN \n \n
  \f$ \textit{domain can bee : "square", "lshape2d", "circle", "threeonfourpartofcircle"} \f$  \n \n
  \warning Pour certaines fonctions permettant de caractériser les noeuds sur une frontière curviligne 
           (unitcircle,partofunitcircle), il faut prendre garde à ce que le fichier de sortie Netgen, généré
	   lui même à partir d'un fichier .in2d,  NE POSSEDE PAS D'AUTRES NOEUDS SUR LA FRONTIERE QUE CEUX
	   MENTIONNES DANS LE FICHIER .in2d LUI MEME. Sinon, ces fonctions ne caractériseront pas les noeuds d'une
	   telle frontière qui seraient situés du coup sur la corde et non sur l'arc... Remarque : Tous les maillages
	   fournis dans les répertoire du code simula+ respectent cette propriété !
	   
	   
*/

//---------------------------------------------------------------------------------------------
template <class Tf>
void make_conversion2dtri (const char* input_file, const char* output_file, const char* domain)
//---------------------------------------------------------------------------------------------
{
  // NETGEN file opening
  ifstream in_file (input_file);
  assert (!in_file.fail());
  ofstream out_file (output_file, ios::trunc);
  assert (!out_file.fail());
  int number_nodes, number_tri;
  int toto;
  Tf x, y;
  Tf titi;

  in_file >> number_nodes;
  matrix <Tf> coordinates (number_nodes, 2);
  vector<int> boundary (number_nodes);
  Tf xleft = -1, xright = 1, ybottom = -1, ytop = 1;
  Tf centerx = 0, centery = 0, radius = 1;

  if (domain == "lshape2d")
    for (int i=1; i<=number_nodes; i++)
      { in_file >> x;
	in_file >> y;
	in_file >> titi;
	coordinates(i,1) = x;
	coordinates(i,2) = y;
	if (lshape_boundary2d (x, y, xleft, xright, ybottom, ytop)) boundary[i] = 1;
      }
  else
    if (domain == "circle")
      for (int i=1; i<=number_nodes; i++)
	{ in_file >> x;
	  in_file >> y;
	  in_file >> titi;
	  coordinates(i,1) = x;
	  coordinates(i,2) = y;
	  if (circle_boundary2d (x, y, centerx, centery, radius)) boundary[i] = 1;
	}
    else
      if (domain == "threeonfourpartofcircle")
	for (int i=1; i<=number_nodes; i++)
	  { in_file >> x;
	    in_file >> y;
	    in_file >> titi;
	    coordinates(i,1) = x;
	    coordinates(i,2) = y;
	    if (threeonfourpartofcircle_boundary2d (x, y, radius)) boundary[i] = 1;
	  }
      else
	if (domain == "square")
	  { Tf halfside = 0.5;
	    for (int i=1; i<=number_nodes; i++)
	      { in_file >> x;
		in_file >> y;
		in_file >> titi;
		coordinates(i,1) = x;
		coordinates(i,2) = y;
		if (square_boundary2d (x, y, halfside)) boundary[i] = 1;
	      }
	  }
	else
	  if (domain == "square1") // to generate the square (0,0)-(1,0)-(1,1)-(0,1)      
	    { Tf halfside = 0.5;
	      for (int i=1; i<=number_nodes; i++)
		{ in_file >> x;
		  in_file >> y;
		  in_file >> titi;
		  coordinates(i,1) = x + 0.5;
		  coordinates(i,2) = y + 0.5;
		  if (square_boundary2d (x, y, halfside)) boundary[i] = 1;
		}
	    }
	  else
	    if (domain == "square2") // to generate the square (1,0)-(2,0)-(2,1)-(1,1)
	      { Tf halfside = 0.5;
		for (int i=1; i<=number_nodes; i++)
		  { in_file >> x;
		    in_file >> y;
		    in_file >> titi;
		    coordinates(i,1) = x + 1.5;
		    coordinates(i,2) = y + 0.5;
		    if (square_boundary2d (x, y, halfside)) boundary[i] = 1;
		  }
	      }
	    else
	      if (domain == "square3") // to generate the square (0,0)-(2,0)-(2,2)-(0,2)
		{ Tf halfside = 0.5;
		  for (int i=1; i<=number_nodes; i++)
		    { in_file >> x;
		      in_file >> y;
		      in_file >> titi;
		      coordinates(i,1) = x + 0.5;
		      coordinates(i,2) = y + 0.5;
		      if (square_boundary2d (x, y, halfside)) boundary[i] = 1;
		      coordinates(i,1) *= 2;
		      coordinates(i,2) *= 2;
		    }
		}
	      else
		if (domain == "square4") // to generate the square (2,0)-(3,0)-(3,1)-(2,1)
		  { Tf halfside = 0.5;
		    for (int i=1; i<=number_nodes; i++)
		      { in_file >> x;
			in_file >> y;
			in_file >> titi;
			coordinates(i,1) = x + 0.5;
			coordinates(i,2) = y + 0.5;
			if (square_boundary2d (x, y, halfside)) boundary[i] = 1;
			coordinates(i,1) += 2;
		      }
		  }
		else
		  if (domain == "halfdisk") // to generate the left unit half disk
		    { for (int i=1; i<=number_nodes; i++)
			{ in_file >> x;
			  in_file >> y;
			  in_file >> titi;
			  coordinates(i,1) = x;
			  coordinates(i,2) = y;
			  if (halfdisk_boundary2d (x, y)) boundary[i] = 1;
			}
		    }
		  else
		    if (domain == "fourthdisk") // to generate the right top unit disk
		      for (int i=1; i<=number_nodes; i++)
			{ in_file >> x;
			  in_file >> y;
			  in_file >> titi;
			  coordinates(i,1) = x;
			  coordinates(i,2) = y;
			  if (fourthdisk_boundary2d (x, y)) boundary[i] = 1;
			}
		    else
		      { cout << red << "ERROR: the domain is up to now not treated in make_conversion2dtri (see convert.h)" << reset;
			exit(1);
		      }
  in_file >> toto;
  in_file >> number_tri;
  matrix<int> tri_nodes (number_tri, 3);
  int m;
  for (int i=1; i<=number_tri; i++)
    { in_file >> m;
      for (int j=1; j<=3; j++)
	in_file >> tri_nodes(i,j);
    }

  in_file.close() ;

  //writing in FORMAT3D
  out_file << 2 << endl;
  out_file << number_nodes << endl;

  out_file.precision (15);
  //out_file.setf(ios::scientific);
  for (int i=1; i<=number_nodes; i++)
    { for (int j=1; j<=2; j++)
	out_file<<coordinates(i,j) << "   ";
      out_file << boundary[i] << endl;
    }

  out_file << number_tri<< endl;

  for (int i=1; i<=number_tri; i++)
    { for (int j=1; j<=3; j++)
	out_file << setw(4) << tri_nodes(i,j) << "   ";
      out_file << endl;
    }
}


}


#endif

