/***************************************************************
 *             Finite Element Method Object Library            *
 *                          tutorial 3                         *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 	3.2.0                  *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2006 CREUSE Emmanuel
 * copyright © 2012 COLLARD Christophe
 * copyright © 2006,2012 Centre National de la Recherche Scientifique
 * copyright © 2006 Arts et Métiers ParisTech
 * copyright © 2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \class tuto3
    \brief tuto3 example \n

    \htmlonly 
    <FONT color="#838383">

    tuto3 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2006, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2006-2012
    \bug none
    \warning none
*/

#ifndef __iostream
#include <iostream>
#endif

#ifndef __string
#include <string>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __time_h
#include <time.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __integration_hpp
#include "MOL++/integration.hpp"
#endif

#ifndef __convert_hpp
#include "FEMOL++/meshes/convert.hpp"
#endif

#ifndef __mesh_tri_2d_hpp
#include "FEMOL++/meshes/mesh_tri_2d.hpp"
#endif

#ifndef __triangle_hpp
#include "FEMOL++/meshes/triangle.hpp"
#endif

#ifndef __p1c2d_hpp
#include "FEMOL++/elements/p1c2d.hpp"
#endif

#ifndef __tetraRT0_hpp
#include "FEMOL++/elements/tetraRT0.hpp"
#endif

#ifndef __tetrap0_hpp
#include "FEMOL++/elements/tetrap0.hpp"
#endif

#ifndef __affiche_hpp
#include "tests/affiche.hpp"
#endif

using namespace std;
using namespace mol;
using namespace femol;


//-------------------------------------------------
// Parameter for reffinement around the singularity
//-------------------------------------------------
long double tau = 0.67;

//------------------------------------
// Nb points for numerical integration
//------------------------------------
int NbIntPts=3;

//-----------------------------------------
// Relaxation Parameter for Uzawa Algorithm
//-----------------------------------------
long double rho = 1e-04;

//---------------------------------
// Stop test for conjugate gradient
//---------------------------------
long double eps1 = 1e-06;

//------------------------------
// Stop test for Uzawa Algorithm
//------------------------------
long double eps2 = 1e-04;

//------------------------------------------------
// Fréquency (in nb iterations) for error checkout
//------------------------------------------------
int nsauv = 50;

//----------------
// Right-Hand-Side
//----------------
long double f (long double x, long double y, long double z)
{ double r = sqrt (powl (x, 2) + powl (y, 2));
  if (r < 1e-12)
    { cout << "Be carefull, f is undefined for x=0 and y=0 !!!" << "\n";
      assert (r > 1e-12);
    }
  else
    { double theta = acos (x / r);
      if (y < 0) {theta = 2*3.14159264-theta;}
          return 2 * powl (r, 2/3.) * sin (2 * theta / 3.) * (0.1 - r) + 7 * (0.1-z) * z * powl (r, -1/3.) * sin (2 * theta / 3) / 3;
    }
}

//---------------
// Exact solution
//---------------
long double u (long double x, long double y, long double z)
{ double r = sqrt (powl (x, 2) + powl (y, 2));
  if (r < 1e-12)
    return 0;
  else
    { double theta = acos (x / r);
      if (y < 0) theta = 2 * 3.14159264 - theta;
      return powl (r, 2/3.) * sin (2 * theta / 3.) * z * (0.1 - z) * (0.1 - r);
    }
}

//---------------------------
// Gradient of exact solution
//---------------------------
vector<long double> p (long double x, long double y, long double z)
{ vector<long double> vec (3);
  double r = sqrt (powl (x, 2) + powl (y, 2));
  if (r < 1e-12)
    { cout << "warning, p is undefined for x=0 and y=0 !!!" << "\n";
      assert (r > 1e-12);
    }
  else
    { double theta = acos (x / r);
      if (y < 0) theta = 2 * 3.14159264 - theta;
      vec[1] = cos (theta) * z * (0.1-z) * sin (2*theta/3.) * (2 * 0.1 * powl (r, -1/3.) / 3. - 5 * powl (r, 2/3.) / 3.)
	                   - 2 * sin (theta) * powl (r, -1/3.) * (0.1-r) * z * (0.1-z) * cos (2 * theta / 3.) / 3.;
      vec[2] = sin (theta) * z * (0.1-z) * sin (2 * theta / 3.) * (2 * 0.1 * powl (r, -1/3.) / 3. - 5 * powl (r, 2/3.)/3.)
	                   + 2 * cos (theta) * powl (r, -1/3.) * (0.1-r) * z * (0.1-z) * cos (2 * theta / 3.) / 3.;
      vec[3] = powl (r, 2/3.) * (0.1-r) * sin (2 * theta / 3.) * (0.1-2*z);
    }
  return vec;
}

//--------------------------------------
// Divergence of exact solution gradient
//--------------------------------------
long double  div_p (long double x, long double y, long double z)
{ // So it is -Laplacian !!!
  return -f (x, y, z);
}



//====
main()
//====
{
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << " BEGINNING tuto3 \n";
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << endl;

  //--------------------------------
  // tetraRT0 finite element loading
  //--------------------------------
  make_conversion3dtetra<long double> ("../../tests/FEMOL++/data/PartOfCyl2/PartOfCylSmall0.dat", "tampon.dat");
  tetraRT0<long double> Th ("tampon.dat");

  //--------------------------------------------
  // Nodes positions modification for refinement
  //--------------------------------------------
  Th.make_anisotropic (tau, 4);

  //-------------------------------
  // tetrap0 finite element loading
  //-------------------------------
  tetrap0<long double> Sh ("tampon.dat");

  //--------------------------------------------
  // Nodes positions modification for refinement
  //--------------------------------------------
  Sh.make_anisotropic (tau, 4);

  //--------------------------------------------------
  // Gauss points and weight for numerical integration
  //--------------------------------------------------
  integration<long double> itg;
  itg.Gauss (NbIntPts, 0, 1);

  int nb_tetra = Th.nbelement ();
  int nb_faces = Th.nbfaces ();
  cout << endl;
  cout << "Nb DoF for ph= " << nb_faces << endl;
  cout << "Nb DoF for uh= " << nb_tetra << endl;

  //-----------------------------------
  // Descriptor copy in local variables
  //-----------------------------------
  matrix<int> face_of_element_local (nb_tetra, 4);
  face_of_element_local = Th.face_of_element ();
  matrix<int> direction_of_normal_local (nb_tetra, 4);
  direction_of_normal_local = Th.direction_of_normal ();

  //-----------------------------------
  // Approximate solutions declarations
  //-----------------------------------
  vector<long double> uh (nb_tetra), uhAnc (nb_tetra);
  vector<long double> ph (nb_faces), phAnc (nb_faces);

  //------------------------------------
  // Constitution de la matrice de masse
  //------------------------------------
  spmatrix<long double> A (Th.UV());

  //-----------------------------------------
  // Computation of the fixed part of the RHS
  //-----------------------------------------
  vector<long double> B;

  int iterations = 0;
  int iter_grad_conj = 0;
  int nb_iterations;
  vector<long double> Int_f (nb_tetra), determinantdiv6 (nb_tetra);

  for (int i=1; i<=nb_tetra; i++)
    { Int_f[i] = Sh.Integrate_f (f, i, itg);
      determinantdiv6[i] = Sh[i].AbsDet() / 6.;
    }

  //--------------------------------------------------------------------------------------------
  // Optional part 
  // uh is initialized by the projection of the exact solution to increase the convergence speed   
  //--------------------------------------------------------------------------------------------
  for (int i=1; i<=nb_tetra; i++)
    uh[i] = Sh.Integrate_f (u, i, itg) / (Sh[i].AbsDet()/6.);

  //-----------------------------------
  // RHS computation and ph computation
  //-----------------------------------
  B = Th.B_PB_MIXTE (uh);
  ph = conjugate_gradient (A, B, phAnc, iter_grad_conj, eps1);

  //----------------
  // Local variables
  //----------------
  clock_t start, end;
  start = clock();
  long double sumalpha, error_L2_u, error_Hdvi_p;

  //---------------
  // Loop beginning
  //---------------
  for (; (uh-uhAnc).norm() / uhAnc.norm() > eps2; iterations++) // precision test
    { nb_iterations = iterations;
      uhAnc = uh;
      phAnc = ph;
      //-----------------------------------
      // RHS computation and ph computation
      //-----------------------------------
      B = Th.B_PB_MIXTE (uh);
      ph = conjugate_gradient (A, B, phAnc, iter_grad_conj, eps1);
      //---------------
      // uh computation
      //---------------
      for (int numele=1; numele<=nb_tetra; numele++)
	{ sumalpha = 0;
	  for (int i=1; i<=4; i++)
	    sumalpha += ph[face_of_element_local(numele,i)] * direction_of_normal_local (numele, i);
	  uh[numele] += rho * (sumalpha + Int_f[numele]) / determinantdiv6[numele];
	}
      //------------------
      // Error computation
      //------------------
      if (iterations%nsauv==0)
	{ cout << "------------------------------------------------------------------\n";
	  cout << "ITERATION= " << iterations;
	  cout << " nb iter conjugate gradient= " << iter_grad_conj ;
	  cout << " stop test : " << (uh-uhAnc).norm() / uhAnc.norm() << "\n";
	  error_L2_u = Sh.error_L2 (u, uh, NbIntPts);
	  error_Hdvi_p = sqrt (pow (Th.error_L2 (p, ph, NbIntPts), 2) + pow (Th.error_div (div_p, ph, NbIntPts), 2));
	  cout << "GLOBAL ERROR = " << error_L2_u+error_Hdvi_p << "\n" ;
	  cout << "\n";
	}
    }
  //------------
  // End of loop
  //------------
  end = clock();
  cout << "------------------------------------------------------------------\n";
  cout << "------------------------------------------------------------------\n";
  cout << "CONVERGENCE OK" << endl;
  cout << "CPU TIME : " << (long double)(end-start)/CLOCKS_PER_SEC << endl;
  cout << "nb iter= " << nb_iterations << "\n";
  cout << "------------------------------------------------------------------\n";
  cout << "------------------------------------------------------------------\n"; 

  //--------------------------------------
  // Error computation when convergence OK
  //--------------------------------------
  cout << "COMPUTATION ERROR...";
  error_L2_u = Sh.error_L2 (u, uh, NbIntPts);
  error_Hdvi_p = sqrt (pow (Th.error_L2 (p, ph, NbIntPts), 2) + pow (Th.error_div (div_p, ph, NbIntPts), 2));
  cout << "END : \n";
  cout << "L2(Omega)-norm for u-uh = " << error_L2_u << "\n";
  cout << "H(div,Omega)-norm for p-ph = " << error_Hdvi_p << "\n" ;
  cout << "GLOBAL ERROR = " << error_L2_u+error_Hdvi_p << "\n" ;
  cout << "\n";

  //-------------------------
  // Destruction des fichiers
  //-------------------------
  remove ("tampon.dat");

  cout << endl;
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << " END tuto3 \n";
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
}
