/*
  Copyright (C) 2008 Ben Asselstine
  Written by Ben Asselstine

  This file is part of fileschanged.

  fileschanged is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  fileschanged is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with fileschanged; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <error.h>
#include <libgen.h>
#include <unistd.h>
#include "filelist.h"
#include <argz.h>
#include "node.h"
#include "list.h"
#include "listdirs.h"


static int 
process_file (void *list, char *filename, int recurse, int deref_symlinks)
{
  int retval;
  struct node_t node;
  retval = node_new (&node, filename, deref_symlinks);
  if (retval == 0)
    {
      if ((S_ISDIR (node.statbuf.st_mode)) && 
	  (recurse))
	{
	  void *dirs = NULL;
	  struct node_t *dir = NULL;
	  unsigned int count;
	  unsigned int i;
	  //add all dirs below and including this one.

	  list_init (&dirs);
	  retval = listdirs (&node, dirs, deref_symlinks);
	  if (retval != 0)
	    return -1;
	  list_count (dirs, &count);
	  for(i = 0; i < count; i++)
	    {
	      list_get_element (dirs, i, &dir);
	      retval = list_add (list, dir);
	    }
	  list_free(dirs);

	}
      else if ((S_ISREG (node.statbuf.st_mode)) || 
	       (S_ISDIR (node.statbuf.st_mode)))
	{
	  //add this file.
	  retval = list_add (list, &node);
	}

      //printf("%s\n",node.filename);

      node_free (&node);
    }

  return 0;
}

static int
remove_duplicates (void *list)
{
  unsigned int i;
  unsigned int count;
  struct node_t *node = NULL;
  struct node_t *lastdir = NULL;
  list_count (list, &count);
  for(i = 0; i < count; i++)
    {
      list_get_element (list, i, &node);
      if (S_ISDIR (node->statbuf.st_mode))
	lastdir = node;
      else if ((S_ISREG (node->statbuf.st_mode)) && (lastdir))
	{
	  char *file;
	  file = strdup (node->filename);
	  if (file)
	    {
	      char *dir;
	      dir = dirname (file);
	      if (strcmp (dir, lastdir->filename) == 0)
		{
		  node_free (node);
		}

	      free (file);
	    }
	  else
	    return -1;
	}
    }

  for(i = 0; i < count; i++)
    {
      list_get_element (list, i, &node);
      if (node_is_empty (node))
	{
	  list_remove_element (list, i);
	}
    }
  return 0;
}

/*
 * filelist.c:
 *   int filelist_populate (void *list_of_files_to_monitor, char *filelist_argz,
 *   char *filelist_argz_len, int recurse);
 *
 * Fill LIST_OF_FILES_TO_MONITOR which is an instantianted list from list.[ch], 
 * with entries that are node_t from node.[ch].  Each entry represents a file
 * that we want to monitor.  The source data for this list is from the
 * argz & argz_len parameters.  If recurse is set, then directories are
 * descended into.
 *
 * Returns 0 upon success, or non-zero upon error.
 *
 */
int 
filelist_populate (void *list_of_files_to_monitor, char *filelist_argz, 
		   size_t filelist_argz_len, int recurse, int deref_symlinks)
{
  int retval = 0;
  char *entry = NULL;
  while ((entry = argz_next (filelist_argz, filelist_argz_len, entry)))
    {
      process_file (list_of_files_to_monitor, entry, recurse, deref_symlinks);
    }
  //okay we have our list now.
  //now remove files for which there are directories.

  list_sort (list_of_files_to_monitor);
  if (remove_duplicates (list_of_files_to_monitor) != 0)
    return -2;

  return retval;
}
