/*
  Copyright (C) 2008 Ben Asselstine
  Written by Ben Asselstine

  This file is part of fileschanged.

  fileschanged is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  fileschanged is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with fileschanged; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301  USA
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/select.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include "fileschanged.h"
#include "node.h"
#include "monitor.h"
#include "list.h"
#include "opts.h"
#include "handlers.h"
#include "fileschanged_priv.h"

/*
 * monitor.c:
 *
 *   While in an open state (eg, we are connected to the FAM server),
 *   we may `begin', `stop' or `pause' monitoring the files.  These 
 *   operations require a LIST of files that we want to begin, stop, or pause 
 *   monitoring of, and the open FAMConnection.  We get the LIST and the
 *   FAM server connection from F.
 *
 *     monitor_begin(struct fileschanged_priv_t *f);
 *     monitor_stop (struct fileschanged_priv_t *f);
 *     monitor_pause_toggle (struct fileschanged_priv_t *f);
 *
 *   After files have begun to be monitored, the FAM server will be notifying
 *   us with events about the files that have somehow changed.  The main loop
 *   of fileschanged is looped around the `handle_events' function. This 
 *   function needs the open FAMConnection, and the LIST of files that we want 
 *   to handle events for.  It also needs a SECS_TO_WAIT_FOR_PENDING parameter.
 *   This is the maximum number of seconds to wait for a message to arrive from
 *   the FAM server.  After that it needs the SECS_TO_HANDLE_PENDING parameter.
 *   This is the maximum number of seconds to process messages from the server
 *   before giving up.  If this value is -1, then it never gives up.  
 *
 *   Note that `begin' also calls `handle_events'.  This is because the server 
 *   can get backlogged trying to send us notifications, when we're not ready
 *   for them.
 *   SECS_TO_HANDLE_PENDING is not -1 when it's called from monitor_begin -- 
 *   the idea being that we don't want to wait for the all of the messages from
 *   the server before beginning to monitor more files.
 *
 *     monitor_handle_events (struct fileschanged_priv_t *f,
 *       int secs_to_wait_for_pending, int secs_to_handle_pending);
 *
 *   This function connects to the handlers via handle_event().
 */


int 
monitor_begin (struct fileschanged_priv_t *f, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  int retval;
  unsigned int i;
  unsigned int count;
  struct node_t *node;
  list_count (f->list_of_files_to_monitor, &count);
  for(i = 0; i < count; i++)
    {
      list_get_element (f->list_of_files_to_monitor, i, &node);
      if (S_ISDIR (node->statbuf.st_mode))
	{
	  //printf ("%04d monitoring directory: '%s'\n", i, node->filename);
	  retval = FAMMonitorDirectory (&f->c, node->filename, &node->request,
					(void *) node);
	  //printf ("FAMMonitorDirectory returns %d (reqnum %d)\n", retval, node->request.reqnum);
	}
      else if (S_ISREG (node->statbuf.st_mode))
	{
	  //printf ("%04d monitoring file: '%s'\n", i, node->filename);
	  retval = FAMMonitorFile (&f->c, node->filename, &node->request,
				   (void *) node);
	  //printf ("FAMMonitorFile returns %d (reqnum %d)\n", retval, node->request.reqnum);
	}
      if (monitor_handle_events (f, 0, 30, hook, handler) > 0)
	return 1;
    }
  return 0;
}

static int 
monitor_do (struct fileschanged_priv_t *f, int (*FAMFunc)(FAMConnection *fc, const FAMRequest *fr))
{
  unsigned int i;
  unsigned int count;
  struct node_t *node;
  list_count (f->list_of_files_to_monitor, &count);
  for(i = 0; i < count; i++)
    {
      list_get_element (f->list_of_files_to_monitor, i, &node);
      FAMFunc (&f->c, &node->request);
      monitor_handle_events (f, 0, 30, NULL, NULL);
    }
  return 0;
}

int 
monitor_stop (struct fileschanged_priv_t *f)
{
  return monitor_do (f, FAMCancelMonitor);
}

int 
monitor_pause_toggle (struct fileschanged_priv_t *f)
{
  int retval;
  int (*FAMFunc)(FAMConnection *fc, const FAMRequest *fr);
  if (f->paused)
    FAMFunc = FAMResumeMonitor;
  else
    FAMFunc = FAMSuspendMonitor;
  retval = monitor_do (f, FAMFunc);
  f->paused = !(f->paused);
  return retval;
}

static int 
monitor_handle_event (struct fileschanged_priv_t *f, int time_limit, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  int retval;
  FAMEvent e;
  time_t ending_time;
  time_t current_time;
  ending_time = time(NULL) + time_limit;
  while ((ending_time >= (current_time = time (NULL))) || (time_limit == -1))
    {
      if (FAMPending (&f->c))
	{
	  retval = FAMNextEvent (&f->c, &e);
	  if (retval < 0)
	    return -1;
	}
      else
	{
	  //second chance for the FAM server.  yay.
	  sleep (0);
	  if (!FAMPending (&f->c))
	    break;
	}
      switch (e.code)
	{
	case FAMExists:
	case FAMEndExist:
	case FAMAcknowledge:
	case FAMMoved:
	  continue;
	case FAMChanged:
	case FAMDeleted:
	case FAMStartExecuting:
	case FAMStopExecuting:
	case FAMCreated:
	    {
	      retval = handle_event (f, &e, current_time, hook, handler);
	      if (retval < 0)
		return -1;
	      break;
	    }
	}
    }
  return 0;
}

int 
monitor_handle_events (struct fileschanged_priv_t *f, int secs_to_wait_for_pending, int secs_to_handle_pending, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  fd_set rfds;
  int numfds;
  struct timeval *tv_ptr = NULL;
  struct timeval tv;
  int retval;
  tv.tv_sec = secs_to_wait_for_pending; //wait for fam events for 2 seconds.
  tv.tv_usec = 500; //always add a bit here.
  tv_ptr = &tv;
  FD_ZERO (&rfds);
  FD_SET (f->c.fd, &rfds);
  numfds = select (FD_SETSIZE, &rfds, NULL, NULL, tv_ptr);
  if (numfds == 1)
    {//if a fam event happened,
      retval = monitor_handle_event (f, secs_to_handle_pending, hook, handler); 
      //get notifications for 30secs tops.
      if (retval < 0)
	{
	  return -1;
	}
    }
  else if (numfds == -1)
    {
      return 1;
    }
  return 0;
}
