/*
  Copyright (C) 2008 Ben Asselstine
  Written by Ben Asselstine

  This file is part of fileschanged.

  fileschanged is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  fileschanged is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with fileschanged; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "handlers.h"
#include "list.h"
#include "node.h"
#include "wl.h"
#include "xvasprintf.h"

/*
 * handlers.c
 *
 * Handle one file notification event from the FAM server.  We need to decide
 * what we're going to do with this event, and we do something different based
 * on what the event is.  We'll call one of the function pointers in 
 * `handlers', and each one of those eventually call `handler'.
 *
 * handle_event needs to know your FAMConnection, and the LIST of files that 
 * we're monitoring, which it gets from the struct fileschanged_priv_t F.  E 
 * is the event from the FAM server, and TIME_OF_EVENT is just a
 * timestamp.  And of course we have to pass around the user's data hook, and 
 * `handler' function pointer.
 *
 * We need to pass down the LIST because we might have to add more files to
 * it, depending on if a new directory was created in a directory we're
 * already monitoring recursively.
 *
 *   handle_event(struct fileschanged_priv_t *f, FAMEvent *e, 
 *     time_t time_of_event);
 *
 * Note: it's important to know how determine_handler() works and why.  If
 * there's a critical function to fileschanged, this is it.
 *
 */
//private prototypes
static int handle_created_file (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook,int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));
static int handle_created_dir (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));
static int handle_changed_file (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));
static int handle_changed_dir (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));
static int handle_deleted_file (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));
static int handle_deleted_dir (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));
static int handle_startexecuting_file (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));
static int handle_stopexecuting_file (struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook));

static struct handler_t handlers[FC_ACTION_MAX]=
{
    { FC_CREATED_FILE,  handle_created_file },
    { FC_CREATED_DIR, handle_created_dir },
    { FC_CHANGED_FILE, handle_changed_file },
    { FC_CHANGED_DIR, handle_changed_dir },
    { FC_DELETED_FILE, handle_deleted_file },
    { FC_DELETED_DIR, handle_deleted_dir },
    { FC_STARTEXECUTING_FILE, handle_startexecuting_file },
    { FC_STOPEXECUTING_FILE, handle_stopexecuting_file },
};

static void 
set_id_for_file (FAMEvent *e, enum fileschanged_action_enum_t *id)
{
  if (e->code == FAMCreated)
    *id = FC_CREATED_FILE;
  else if (e->code == FAMChanged)
    *id = FC_CHANGED_FILE;
  else if (e->code == FAMDeleted)
    *id = FC_DELETED_FILE;
  else if (e->code == FAMStartExecuting)
    *id = FC_STARTEXECUTING_FILE;
  else if (e->code == FAMStopExecuting)
    *id = FC_STOPEXECUTING_FILE;
  else
    *id = FC_ACTION_MAX;
  return;
}

static void 
set_id_for_dir (FAMEvent *e, enum fileschanged_action_enum_t *id)
{
  if (e->code == FAMCreated)
    *id = FC_CREATED_DIR;
  else if (e->code == FAMChanged)
    *id = FC_CHANGED_DIR;
  else if (e->code == FAMDeleted)
    *id = FC_DELETED_DIR;
  else
    *id = FC_ACTION_MAX;
  return;
}

static int 
determine_handler (void *list, FAMEvent *e, enum fileschanged_action_enum_t *id, char **filename, int recurse, int deref_symlinks)
{
  struct node_t *filenode = NULL;
  if ((!list) || (!e) || (!id) || (!filename))
    return -1;
  *id = FC_ACTION_MAX;
  //lookup the node that the event corresponds to.
  filenode = (struct node_t *)e->userdata;
  if (S_ISREG (filenode->statbuf.st_mode))
    {
      //the simple case.
      //okay it's a file we were monitoring to get this event.
      *filename = strdup (filenode->filename);
      if (!*filename)
	return -2;
      set_id_for_file (e, id);
    }
  else if (S_ISDIR (filenode->statbuf.st_mode))
    {
      //okay it's a directory we were monitoring to get this event.
      //it could still be a file in the directory we are monitoring.
      //if e.filename is an absolute path then it must be a directory.
      if (e->filename[0] == '/') //it's a directory
	{
	  *filename = strdup (filenode->filename);
	  if (!*filename)
	    return -3;
	  set_id_for_dir (e, id);
	}
      else
	{
	  //okay it could STILL be a file or directory under one that we're monitoring.
	  //but if we put filenode->filename and e->filename together, and stat it,
	  //deleted files/dirs will fail if we stat them (with node_new).
	  if (e->code != FAMDeleted)
	    {
	      int isdir = 0;
	      char *newfilename = NULL;
	      struct node_t newnode;
	      int retval;
	      if ((newfilename = xasprintf ("%s/%s", filenode->filename, 
					    e->filename)) == NULL)
		return -4;
	      retval = node_new (&newnode, newfilename, deref_symlinks);
	      free (newfilename); newfilename = NULL;
	      if (retval == 0)
		{
		  isdir = S_ISDIR (newnode.statbuf.st_mode);
		  *filename = strdup (newnode.filename);
		  node_free (&newnode);
		}
	      else if (retval < 0)
		return -2;
	      if (isdir)
		set_id_for_dir (e, id);
	      else //regular file
		set_id_for_file (e, id);
	    }
	  else
	    {
	      if (*filename)
		free (*filename);
	      if ((*filename = xasprintf ("%s/%s", filenode->filename, 
					  e->filename)) == NULL)
		return -3;
	      //it's a deleted file or directory.  how can we tell it's type?  it's gone now.
	      //if it were a directory we were monitoring, and we were in recursive mode,
	      //then we would have received a notification for that directory with an absolute name.
	      //but this is a relative name.

	      //here's the deal.  we only show notifications for directories when we're not in recursive mode.
	      if (recurse == 0)
		{
		  //i still don't know if it's a file or directory
		  //just say it's a file.  because we're treating the directory like a directory-file anyway.
		  *id = FC_DELETED_FILE;
		}
	      else
		{
		  //okay we're in recursive mode.  a directory would have it's own notification.
		  //it must be a file
		  *id = FC_DELETED_FILE;
		}
	    }
	}
    }
  if (*id == FC_ACTION_MAX)
    return -2; //unhandled case.
  return 0;
}

int 
handle_event(struct fileschanged_priv_t *f, FAMEvent *e, time_t time_of_event, 
	     void *hook, int (*handler)(char *filename, 
			    		enum fileschanged_action_enum_t action, 
					void *hook))
{
  char *filename = NULL;
  enum fileschanged_action_enum_t id;
  int retval;
  retval = determine_handler (f->list_of_files_to_monitor, e, &id, &filename, f->recurse, f->dereference_symlinks);
  if (retval != 0)
    {
      if (filename)
	free (filename);
      return -1;
    }
  retval = handlers[id].handler (f, id, filename, hook, handler);
  free (filename);
  return retval;
}

static int 
handle_created_file(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  if (handler)
    handler (filename, id, hook);
  return 0;
}

static int 
handle_created_dir(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  //if we're recursing, then we should automatically start monitoring this dir.
  if (f->recurse)
    {
      struct node_t newdir;
      struct node_t *node = NULL;
      unsigned int count;
      int retval;
      retval = node_new (&newdir, filename, f->dereference_symlinks);
      if (retval == 0)
	{
	  //printf("Adding '%s'\n",filename);
	  list_add (f->list_of_files_to_monitor, &newdir);
	  list_count (f->list_of_files_to_monitor, &count);
	  list_get_element (f->list_of_files_to_monitor, count - 1, &node);
	  FAMMonitorDirectory (&f->c, newdir.filename, &newdir.request, 
			       (void*)node);
	  node_free (&newdir);
	}
      else if (retval < 0)
	return -1;
    }

  if (handler)
    handler (filename, id, hook);

  return 0;
}

static int 
handle_changed_file(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
      if (f->timeout > 0)
	{
	  wl_add_file (f->wl, filename);
	}
      else
	{
	  handler (filename, id, hook);
	}
  return 0;
}

static int 
handle_changed_dir(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  if (f->recurse == 0)
    {
      if (handler)
	{
	  handler (filename, id, hook);
	}
    }
  return 0;
}

static int 
handle_deleted_file(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  if (handler)
    {
      handler (filename, id, hook);
    }
  return 0;
}

static int 
handle_deleted_dir(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  if (handler)
    {
      handler (filename, id, hook);
    }
  return 0;
}

static int 
handle_startexecuting_file(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  if (handler)
    {
      handler (filename, id, hook);
    }
  return 0;
}

static int 
handle_stopexecuting_file(struct fileschanged_priv_t *f, enum fileschanged_action_enum_t id, char *filename, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  if (handler)
    {
      handler (filename, id, hook);
    }
  return 0;
}
