// Free Image Manipulator - a program to massive image manipulation.
// Copyright (C) 2006 Kacper Bielecki
// 
// This program is free software; you can redistribute it and/or modify it 
// under the terms of the GNU General Public License as published by the Free 
// Software Foundation; either version 2 of the License, or (at your option) 
// any later version.
// 
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
// more details.
// 
// You should have received a copy of the GNU General Public License along 
// with this program; if not, write to the Free Software Foundation, Inc., 
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
// 
//
// C++ Implementation: mainwindow
//
// Description: 
//
//
// Author: Kacper <kazjote@stud.ics.p.lodz.pl>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "mainwindow.h"
#include "images.h"
#include "treemodel.h"
#include <QFileDialog>
#include <QColorDialog>
#include <QHeaderView>
#include <QButtonGroup>
#include <QApplication>
#include <QProgressDialog>
#include <QMessageBox>
#include <QFileInfo>
#include <QSettings>
#include <QFile>
#include <QDataStream>
#include <QRegExp>
#include <QString>

#include "about.h"

#include <cstdlib>
#include <iostream>

using namespace std;

// Wypadek :) Windowsa rozpatrzony jest w konstruktorze MainWindow
char FONTPATH[256];

const char *SAVE_VERSION = "0.1";
const char *FILE_TYPE = "fim-save;";

extern QApplication *appp;

void MainWindow::showAbout( )
{
	About about;
	about.exec();
}

MainWindow::MainWindow()
 : QMainWindow()
{
	// Na wypadek windowsa
#ifdef WIN32
	strcpy(FONTPATH, getenv("windir"));
	strcat(FONTPATH, "\\FONTS");
#else
	strcpy(FONTPATH, "/usr/share/fonts/TTF");
#endif
	
	mainWindow.setupUi(this);
	mainWindow.previewW->setFixedSize(300, 300);
	images = new Images(mainWindow.centralwidget);
	bgColor = new QColor(0, 0, 0);
	fontColor = new QColor(255, 255, 255);
	/*
	 * Init treemodel
	 */
	treemodel = new TreeModel(0, 4, mainWindow.centralwidget);
	treemodel->setHeaderData(0, Qt::Horizontal, tr("Name"), Qt::DisplayRole);
	treemodel->setHeaderData(1, Qt::Horizontal, tr("Type"), Qt::DisplayRole);
	treemodel->setHeaderData(2, Qt::Horizontal, tr("Width"), Qt::DisplayRole);
	treemodel->setHeaderData(3, Qt::Horizontal, tr("Height"), Qt::DisplayRole);
	
	mainWindow.fileTreeView->setModel(treemodel);
	
	QHeaderView *head = mainWindow.fileTreeView->header();
	head->resizeSection(head->logicalIndex(1), 50);
	head->resizeSection(head->logicalIndex(2), 50);
	head->resizeSection(head->logicalIndex(3), 50);
	head->setResizeMode(head->logicalIndex(0), QHeaderView::Stretch);
	head->setResizeMode(head->logicalIndex(1), QHeaderView::Custom);
	head->setResizeMode(head->logicalIndex(2), QHeaderView::Custom);
	head->setResizeMode(head->logicalIndex(3), QHeaderView::Custom);
	head->setStretchLastSection ( false );
	
	/*
	 * Grouping buttons
	*/
	TBGroup = new QButtonGroup(mainWindow.centralwidget);
	TBGroup->addButton(mainWindow.printTextSelLeft);
	TBGroup->addButton(mainWindow.printTextSelRight);
	LRGroup = new QButtonGroup(mainWindow.centralwidget);
	LRGroup->addButton(mainWindow.printTextSelTop);
	LRGroup->addButton(mainWindow.printTextSelBottom);
	/*
	 * Connecting signals!
	 */
	connect(mainWindow.actionAdd, SIGNAL(activated()), this, SLOT(addFiles()));
	connect(mainWindow.actionDel, SIGNAL(activated()), this,
			  SLOT(delFiles()));
	connect(images, SIGNAL(added(Image_ext* )), 
			  treemodel, SLOT(addRequest(Image_ext* )));
	connect(images, SIGNAL(deleted(int )), 
			  treemodel, SLOT(delRequest(int)));
	connect(images, SIGNAL(added(Image_ext* )), 
			  this, SLOT(makePrevConnection(Image_ext* )));
	connect(mainWindow.fileTreeView, SIGNAL(clicked(const QModelIndex &)), 
			  this, SLOT(imageClicked(const QModelIndex& )));
	connect(mainWindow.selBgColor, SIGNAL(clicked()), this, SLOT(chooseBgColor()));
	connect(mainWindow.selFontColor, SIGNAL(clicked()), this, SLOT(chooseFontColor()));
	connect(mainWindow.clearButton, SIGNAL(clicked()), images,
			  SLOT(clear()));
	connect(mainWindow.procButton, SIGNAL(clicked()), this, SLOT(process()));
	connect(mainWindow.actionAbout, SIGNAL(activated()), this, SLOT(showAbout()));
   connect(mainWindow.actionSave_configuration, SIGNAL(activated()), this, SLOT(saveConfig()));
   connect(mainWindow.actionLoad_configuration, SIGNAL(activated()), this, SLOT(loadConfig()));
   connect(mainWindow.actionSave_configuration_as, SIGNAL(activated()), this, SLOT(saveConfigAs()));
	
	/*
	 * Wyszukanie czcionek.
	 */
	QDir font_dir(FONTPATH);
	QStringList filter("*.ttf");
	QStringList fonts = font_dir.entryList(filter);
	
	for(	QStringList::iterator iter = fonts.begin();
			  iter != fonts.end();
			  iter++
		)
		mainWindow.fontSel->addItem(*iter);
	
	// Ustawienie Validatorów
	sizeValidator = new QIntValidator(0, 8192, mainWindow.centralwidget);
	mainWindow.selWidth->setValidator(sizeValidator);
	mainWindow.selHeight->setValidator(sizeValidator);
	
}

void MainWindow::addFiles()
{
	QStringList files = QFileDialog::getOpenFileNames
			(mainWindow.centralwidget,
			 tr("Choose files to add"),
         ( addDir == "" ) ? QString() : addDir ,
			 tr("Images (*.jpg *.jpeg *.png *.gif)")				
			);
   QString firstFile;
   if ( ( firstFile = files.value(0) ) != "")
   {
      addDir = QFileInfo( firstFile ).dir().absolutePath();
   }
 	QProgressDialog progress("Adding files...", "Abort Copy", 0, files.count(), this);
	
	int i=0;
	for (QStringList::iterator iter = files.begin(); 
			 iter != files.end();
			 iter++)
	{
		progress.setValue(++i);
		appp->processEvents();
		if (progress.wasCanceled())
			break;
		images->addRequest(*iter);
	}
}

MainWindow::~MainWindow()
{
}

void MainWindow::chooseBgColor( )
{
	*bgColor = QColorDialog::getColor(*bgColor);
}

void MainWindow::chooseFontColor( )
{
	*fontColor = QColorDialog::getColor(*fontColor);
}

void MainWindow::makePrevConnection( Image_ext * image )
{
	connect(image, SIGNAL(previewed(Image* )), this, SLOT(drawPreview(Image* )));
}

void MainWindow::drawPreview( Image * image )
{
	mainWindow.previewW->image = image->getImage();
	mainWindow.previewW->repaint(0, 0, -1, -1);
}

void MainWindow::QColorToRgb( const QColor & color, unsigned char rgb[3] )
{
	unsigned int rgb_int = color.rgb();
	unsigned char* t_rgb = reinterpret_cast<unsigned char*>(&rgb_int);
	rgb[0] = t_rgb[2];
	rgb[1] = t_rgb[1];
	rgb[2] = t_rgb[0];
}

void MainWindow::imageClicked( const QModelIndex & index )
{
	transOptions options = getOptions();
	
	(*images)[index.row()]->preview_request(&options);
}

void MainWindow::delFiles( )
{
	QModelIndexList selection = 
			mainWindow.fileTreeView->selectionModel()->selection().indexes();
	while (selection.begin() != selection.end())
	{
		mainWindow.previewW->image=0;
	
		images->delRequest((*selection.begin()).row());
		std::cout << "Koniec!!" << std::endl;
		
		selection = 
				mainWindow.fileTreeView->selectionModel()->selection().indexes();
	}
}

transOptions MainWindow::getOptions( ) const
{
	transOptions options;
	options.text = mainWindow.printTextEdit->text();
	bool ok = true; //false;
	options.w = mainWindow.selWidth->currentText().toInt(&ok, 10);
	if (!ok) options.w = 0;
	options.h = mainWindow.selHeight->currentText().toInt(&ok, 10);
	if (!ok) options.h = 0;
	options.text_size = mainWindow.printTextSize->value();
	options.fontPath = QString(FONTPATH)+"/"+mainWindow.fontSel->currentText();
	options.lr_margin = mainWindow.hSpacing->value();
	options.tb_margin = mainWindow.vSpacing->value();
	if (mainWindow.printTextSelLeft->isChecked())
	{
		options.pos_left = mainWindow.printTextLeft->value();
		options.pos_right = -1;
	} else
	{
		options.pos_right = mainWindow.printTextLeft->value();
		options.pos_left = -1;
	}
	if (mainWindow.printTextSelTop->isChecked())
	{
		options.pos_top = mainWindow.printTextTop->value();
		options.pos_bottom = -1;
	} else
	{
		options.pos_bottom = mainWindow.printTextTop->value();
		options.pos_top = -1;
	}
	options.opacity = mainWindow.printTextOpacity->value();
	options.bgOpacity = mainWindow.bgOpacity->value();
	
	unsigned char rgb[3];
	QColorToRgb(*bgColor, rgb);
	options.bgColor[0] = rgb[0];
	options.bgColor[1] = rgb[1];
	options.bgColor[2] = rgb[2];
	QColorToRgb(*fontColor, rgb);
	options.fontColor[0] = rgb[0];
	options.fontColor[1] = rgb[1];
	options.fontColor[2] = rgb[2];
   options.quality = mainWindow.qualitySpin->value();
	return options;
}

void MainWindow::process( )
{
	// No images
	if (images->count() == 0) return;
	
   // If files weren't processed yet, default dir will be set to addDir
   QString dirPath = QFileDialog::getExistingDirectory(0, tr("Choose directory to save files to."), ( procDir == "" ) ? ( ( addDir == "" ) ? QString() : addDir ) : procDir);
	
	// Dialog cancelled
	if (dirPath == "") return;
   procDir = dirPath;
	
	QProgressDialog progress("Saving files...", "Abort Copy", 0, images->count(), this);
	
	transOptions options = getOptions();
	images->resetIter();
	
	int i=0;
	Image_ext* to_save;
	bool replaceAll = false;
	while ((to_save = images->getIter()))
	{
		progress.setValue(++i);
		appp->processEvents();
		if (progress.wasCanceled())
			break;
		
		// Sprawdzenie czy plik nie istnieje
		QString old_path(QFileInfo(to_save->getPath()).fileName());
		
		if (old_path.lastIndexOf(".") > -1)
			old_path = old_path.mid(0, old_path.lastIndexOf("."));
		QString new_file_name = 
				((mainWindow.addCombo->currentIndex() == 0) ? mainWindow.addEdit->text() : "") +
				old_path +
				((mainWindow.addCombo->currentIndex() == 1) ? mainWindow.addEdit->text() : "");
		
		QString extension;
		ImageFormat format;
		switch(mainWindow.FileTypeCombo->currentIndex())
		{
			case 0:
				format = to_save->getFormat();
				break;
			case 1:
				format = jpeg;
				break;
			case 2:
				format = png;
				break;
			case 3:
				format = gif;
				break;
			default:
				format = bmp;
				break;
		}
		
		switch(format)
		{
			case 0: //jpeg
				extension = ".jpg";
				break;
			case 1: //jpeg
				extension = ".png";
				break;
			case 2: //jpeg
				extension = ".gif";
				break;
			case 3: //jpeg
				extension = ".bmp";
				break;
			default:
				// just to eliminate warning
				break;
		}
#ifdef WIN32
      // Niespojnosc biblioteki??
      QString new_path = dirPath + "\\" + new_file_name + extension;
#else
		QString new_path = dirPath + new_file_name + extension;
#endif
		bool cont = true;
		if (QFileInfo(new_path).exists() && (!replaceAll))
		{
			switch(QMessageBox::question(0, tr("File exists"), 
					 QString(tr("File you are tring to save to already exists (%1). Do you want to replace it?")).arg(new_file_name+extension),
					 QMessageBox::Yes, QMessageBox::YesAll, QMessageBox::No))
			{
				case QMessageBox::YesAll:
					replaceAll = cont = true;
					break;
				
				case QMessageBox::Yes:
					cont = true;
					break;
				
				case QMessageBox::No:
					cont = false;
					break;
			}
		}
		
		if (cont)
		{
			// Plik nie istnieje lub zgoda na jego nadpisanie
			try
			{
				to_save->saveToFile(new_path, mainWindow.qualitySlider->value(),
										  to_save->getFormat(), options);
			}
			catch (...)
			{
				QMessageBox::warning(0, tr("Error"), 
											QString(tr("Error during saving file %1.")).arg(new_file_name+extension), 
											QMessageBox::Ok, 0);
			}
		}
	}
}

void MainWindow::saveOptions( QString filePath )
{
	// Before entering the method u should check if file exists
	QFile file(filePath);
	if (!file.open(QIODevice::WriteOnly))
	{
		QMessageBox::warning(0, QString(tr("Can't write to file.")), 
									QString(tr("Specified file can't be opened for writing. Check if you have right permissions.")),
									QMessageBox::Ok, 0, 0);
		return;
	}
	QDataStream wy( &file );
	transOptions opt = getOptions();
	wy << FILE_TYPE;
	wy << SAVE_VERSION;
	wy << opt.h;
	wy << opt.w;
	wy << opt.bgOpacity;
	wy << opt.lr_margin;
	wy << opt.opacity;
	wy << opt.pos_bottom;
	wy << opt.pos_left;
	wy << opt.pos_right;
	wy << opt.pos_top;
	wy << opt.tb_margin;
	wy << opt.text_size;
	wy << opt.bgColor[0];
	wy << opt.bgColor[1];
	wy << opt.bgColor[2];
	wy << opt.fontColor[0];
	wy << opt.fontColor[1];
	wy << opt.fontColor[2];
	wy << opt.text;
	wy << opt.fontPath;
   wy << opt.quality;
	file.close();
}

void MainWindow::restoreOptions( const transOptions &opt )
{
	mainWindow.printTextEdit->setText(opt.text);
	mainWindow.printTextSize->setValue( opt.text_size );
	mainWindow.printTextOpacity->setValue( opt.opacity );
	mainWindow.bgOpacity->setValue( opt.bgOpacity );
	/**
	 * Wyszukanie czcionki o zadanej nazwie.
	 */
   QString nazwa = QFileInfo(opt.fontPath).fileName(); // Nazwa czcionki
	
	int fontPos = mainWindow.fontSel->findData(nazwa, Qt::DisplayRole);
	if ( fontPos > -1 )
		mainWindow.fontSel->setCurrentIndex( fontPos );
	else
	{
		// 
		QMessageBox::warning(0, QString(tr("Font not found")), 
									QString(tr("Saved font "))+nazwa+QString(tr(" was not found in the system. Choose different font instead.")),
									QMessageBox::Ok, 0, 0);
	}
	
	*bgColor = QColor( opt.bgColor[0], opt.bgColor[1], opt.bgColor[2] );
	*fontColor = QColor( opt.fontColor[0], opt.fontColor[1], opt.fontColor[2] );
	
	if ( opt.pos_left > -1 )
	{
		mainWindow.printTextSelLeft->setDown(true);
		mainWindow.printTextLeft->setValue( opt.pos_left);
	}
	else
	{
		mainWindow.printTextSelRight->setDown(true);
		mainWindow.printTextLeft->setValue( opt.pos_right);
	}
	
	if ( opt.pos_top > -1 )
	{
		mainWindow.printTextSelTop->setDown(true);
		mainWindow.printTextTop->setValue( opt.pos_top);
	}
	else
	{
		mainWindow.printTextSelBottom->setDown(true);
		mainWindow.printTextTop->setValue( opt.pos_bottom );
	}
	
	mainWindow.hSpacing->setValue( opt.lr_margin );
	mainWindow.vSpacing->setValue( opt.tb_margin );
	
	mainWindow.selWidth->setItemText( 0, QString::number( opt.w ) );
	mainWindow.selHeight->setItemText( 0, QString::number( opt.h ) );
   mainWindow.qualitySpin->setValue( opt.quality );
	//mainWindow
}

transOptions MainWindow::loadOptions( QString filePath )
{
   QFile file(filePath);
	if (!file.open(QIODevice::ReadOnly))
	{
		QMessageBox::warning(0, QString(tr("Can't open file.")), 
									QString(tr("Specified file can't be opened for reading. Check if you have right permissions and file really exists.")),
									QMessageBox::Ok, 0, 0);
      return transOptions();
	}
	QDataStream we( &file );
	
	transOptions opt;
	char *file_type;
	char *version;
	
	we >> file_type;
   if (strcmp ( file_type, FILE_TYPE ) != 0 )
	{
		QMessageBox::warning(0, QString(tr("Can't open file.")), 
									QString(tr("This is not properly saved Free Image Manipulator's save file.")),
									QMessageBox::Ok, 0, 0);
		file.close();
		delete [] file_type;
      return transOptions();
	}
	
	we >> version;
   if ( strcmp ( version, SAVE_VERSION ) != 0 )
	{
		QMessageBox::warning(0, QString(tr("Can't open file.")), 
									QString(tr("This save file comes from different Free Image Manipulator's version. I can't open it.")),
									QMessageBox::Ok, 0, 0);
		file.close();
		delete [] file_type;
		delete [] version;
      return transOptions();
	}
	
	we >> opt.h;
	we >> opt.w;
	we >> opt.bgOpacity;
	we >> opt.lr_margin;
	we >> opt.opacity;
	we >> opt.pos_bottom;
	we >> opt.pos_left;
	we >> opt.pos_right;
	we >> opt.pos_top;
	we >> opt.tb_margin;
	we >> opt.text_size;
	we >> opt.bgColor[0];
	we >> opt.bgColor[1];
	we >> opt.bgColor[2];
	we >> opt.fontColor[0];
	we >> opt.fontColor[1];
	we >> opt.fontColor[2];
	we >> opt.text;
	we >> opt.fontPath;
   we >> opt.quality;
	delete [] file_type;
	delete [] version;
	file.close();
   return opt;
}

void MainWindow::saveConfig( )
{
   QString selectedPath;
   if (configFile == "")
      selectedPath = QFileDialog::getSaveFileName(0, QString(tr("Choose configuration file...")), (configDir == "") ? QString() : configDir, QString(tr("FIM (*.fim)")));
   else
      selectedPath = configFile;
   
   if (selectedPath != "")
   {
      // ADD .fim to file if it is not already added
      QRegExp rx(".fim$");
      if (rx.indexIn( selectedPath ) == -1)
         selectedPath += QString(tr(".fim"));
      // The file was selected
      configDir = QFileInfo(selectedPath).dir().absolutePath();
      configFile = selectedPath;
      saveOptions(configFile);
   }
}

void MainWindow::loadConfig( )
{
   QString path = QFileDialog::getOpenFileName(0, QString(tr("Choose configuration file to load")), (configDir == "") ? QString() : configDir, QString(tr("FIM (*.fim)")));
   
   if (path != QString())
   {
      configFile = path;
      transOptions opt = loadOptions( path );
      configDir = QFileInfo( path ).dir( ).absolutePath( );
      restoreOptions(opt);
   }
}

void MainWindow::saveConfigAs( )
{
   configFile = "";
   saveConfig();
}

