// Copyright (C) 2005 Javier Amor Garcia

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#include "Reader.hh"
#include "TextSource.hh"
#include "View.hh"
#include <assert.h>
#include <boost/bind.hpp>


using namespace std;

#define MAX_TV_NSEC_RANGE 999999999

void* timer_callback(void* reader_ptr);



Reader::Reader () : reading(false),  showTextThread(NULL),view(NULL), sourceIndex(0)
{
  period.tv_sec= 0;
  period.tv_nsec=MAX_TV_NSEC_RANGE;
  wpm = 0;
}

Reader::~Reader()
{
  if (showTextThread != NULL)
    delete showTextThread;
  vector<TextSource*>::const_iterator p, end;
  for (p=sources.begin(),end=sources.end(); p != end; p++)
      delete *p;
}


void Reader::setView (View *v)
{
  view = v;
}



void Reader::addSource (TextSource* source)
{
  assert (source != NULL);
  sources.push_back(source);
}





void Reader::switchSource(bool forward)
{
  assert(view);

  if (forward)
    {
      if (sourceIndex <( sources.size()-1))
	sourceIndex++;
      else
	return;
    }
  else
    {
      if (sourceIndex > 0)
	sourceIndex--;
      else
	return;
    }
  view->displayFilename((char*) sources[sourceIndex]->getName().c_str());
      
}


  
void Reader::startRead()
{
  assert(view);
  
  reading = true;
  showTextThread = new thread(boost::bind(&Reader::showText, this));
  
  sourceIndex= 0;
  view->displayFilename((char*) sources[sourceIndex]->getName().c_str());
}

void Reader::stopRead()
{
  reading = false;
}

void Reader::switchReadState()
{
  reading = !reading;
}

bool Reader::getReading()
{
  return reading;
}




void Reader::rewind(int elements)
{
  sources[sourceIndex]->rewind(elements);
}

void Reader::forward(int elements)
{
  sources[sourceIndex]->forward(elements);
}

void Reader::seek(int elements)
{
  sources[sourceIndex]->seek(elements);
}

unsigned int Reader::getSpeed()
{
  return wpm;
}

  
void Reader::setSpeed(unsigned int newWpm)
{
  this->wpm = newWpm;
  setPeriod();
}


void Reader::increaseSpeed(unsigned int dwpm)
{
  wpm +=  dwpm;
  setPeriod();
  if (reading == false)
    switchReadState();
}


void Reader::decreaseSpeed(unsigned int dwpm)
{
  if ( dwpm >= wpm)
    wpm = 0; 
  else
    wpm -= dwpm;
  setPeriod();
 }

void Reader::setPeriod()
{
  assert(view);

  if (wpm == 0)
    {
      stopRead();
      period.tv_nsec= MAX_TV_NSEC_RANGE;
    }
  else
    {
      long  microseconds =   60000000/ (long)  wpm;
      long nanoseconds = microseconds*1000;
      if (nanoseconds < microseconds)  //overflow protection
	nanoseconds = MAX_TV_NSEC_RANGE;
      
      period.tv_nsec= nanoseconds;
    }
     view->displaySpeed(wpm);
}



bool  Reader::readElement()
{
  assert(view);

  const char* text;
  
  if (!reading)
    {
      return false;
    }
  

  text = sources[sourceIndex]->readElement().c_str();
  if (text != NULL)
    {
      view->displayTextData((char*) text);
      view->displayPosition(sources[sourceIndex]->getPosition());
      
      return true;
    }
  else
    {
      if (sourceIndex < sources.size())
	{
	  sourceIndex++;
	  return readElement();
	}
      else
	{
	  return false;
	}
      
    }
  
  
}


const struct timespec* Reader::getPeriod()
{
  return &period;
}


/** 
 * Function that periodically show the next text bit in the view
 * 
 */
void Reader::showText()
{
  while(1)
    {
      while (false == getReading())
	thread::yield();
      if (false == readElement())
	break;
      nanosleep(getPeriod(), NULL);
    }
}





