/*
 * Fisoco, a FInding SOrting and COnverting free software
 * Copyright 2015 Félicien PILLOT <felicien.pillot@member.fsf.org>
 *
 * This file is part of Fisoco.
 *
 * Fisoco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Fisoco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Fisoco.  If not, see <http://www.gnu.org/licenses/>.
 *
 *//////////////////////////////////////
/****************
 *    cli.cc    *
 *    ~~~~~~    *
 ****************/

#include "../include/search.h"
#include "../include/cli.h"
#include "../include/main.h"
#include <regex>

Cli::Cli ()
{
    std::cout << gettext ("Welcome to Fisoco's Command Line Interface !") << std::endl <<
    gettext ("Type 'help' to view a list of commands") << std::endl;    
    std::string command = prompt ();

    while (command != "q" && command != "quit")
    {
        // Check if the command contains exactly two words        
        int number_of_args = check_args (command);
        switch (number_of_args)
        {
            case 1:
                if (command == "e" || command == "erase")
                    erase ();
                else if (command == "h" || command == "help")
                    help ();
                else if (command == "r" || command == "read")
                    read ();
                else if (command == "s" || command == "search")
                    help ("search");
                else if (command == "v" || command == "version")
                    display_version ();
                break;
            case 2:
            {
                std::regex pattern ("^(\\w+)\\s+(\\w+)");
                std::smatch matches;
                regex_search (command, matches, pattern);
//                 for (int i = 0; i < matches.size (); i++)
//                     std::cout << i << ": " << matches[i] << std::endl;
                if (matches[1] == "h" || matches[1] == "help")
                    help (matches[2]);
                if (matches[1] == "s" || matches[1] == "search")
                    search (matches[2]);
                break;
            }
            case 3:
                std::cout << "Command not known, too many arguments, you typed your query whith your eyes closed... I don't know.." << std::endl;
                break;
        }
        command = prompt ();        
    }
}

int
Cli::check_args (std::string command)
{
    std::regex pattern;
    // First check : command isn't empty
    if (command == "")
        return 0;
    else
    {
        // Second check : no more than two words (command + arg)
        pattern = ("^\\w+\\s+\\w+\\s+\\w+");
        if (regex_search (command, pattern))
            return 3;
        else
        {
            // Third check : no less than two words
            pattern = ("^\\w+");
            if (regex_match (command, pattern))
                return 1;
            else
            {
                return 2;
            }
        }
    }
}

void
Cli::display_version ()
{
    std::cout << std::endl
    << "Fisoco v0.6 (alpha) : " <<
    gettext ("a Finding, Sorting and Converting free software.") << std::endl
    << std::endl
    << "Copyright © 2015 Félicien Pillot <felicien.pillot@member.fsf.org>" << std::endl
    << "Fisoco is free software: you can redistribute it and/or modify" << std::endl
    << "under the terms of the GNU General Public License." << std::endl
    << std::endl
    << "Fisoco is distributed in the hope that it will be useful," << std::endl
    << "but WITHOUT ANY WARRANTY; without even the implied warranty of" << std::endl
    << "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the" << std::endl
    << "GNU General Public License for more details." << std::endl
    << std::endl;
}

void
Cli::erase ()
{
    if (Rfiles.size () == 0)
        std::cout << "List already empty" << std::endl;
    else
        Rfiles.erase (Rfiles.begin (), Rfiles.end ());
}

void
Cli::error_dialog (std::string message)
{
    std::cout << gettext ("[ERROR] : ") << message << std::endl;
}

void
Cli::help ()
{
    std::cout << "  \033[1mhelp [COMMAND]\033[00;0m\t\t\t" <<
    gettext ("print general help, or help on COMMAND")
    << std::endl << "  \033[1merase\033[00;0m\t\t\t\t\t" <<
    gettext ("erase all selected elements from the list")
    << std::endl << "  \033[1mread\033[00;0m\t\t\t\t\t" <<
    gettext ("display the list of files actually selected")
    << std::endl << "  \033[1msearch FILE\033[00;0m\t\t\t\t" <<
    gettext ("search FILE on system")
    << std::endl << "  \033[1mquit\033[00;0m\t\t\t\t\t" <<
    gettext ("exit program")
    << std::endl << "  \033[1mversion\033[00;0m\t\t\t\t" <<
    gettext ("version and copyright information") << std::endl;
}

void
Cli::help (std::string argument)
{
    std::cout << "  \033[1m" << argument << " :\033[00;0m" << std::endl;
    if (argument == "help")
        std::cout << "Are you kidding me ?" << std::endl;
    else if (argument == "erase")
        std::cout << "This command requires no arguments, it erases all elements selected (or not) from the list.\nYou can use this when you want to cancel your search and start a new one." << std::endl;
    else if (argument == "read")
        std::cout << "This command requires no arguments, it displays all elements selected (or just all elements founded, if you haven't yet selected some of them)." << std::endl;
    else if (argument == "search")
        std::cout << "This command requires one argument. It takes the FILE or expression given as parameter, and look for it on the entire system (at this point of the development)." << std::endl;
    else if (argument == "quit")
        std::cout << "Are you kidding me ?" << std::endl;
    else if (argument == "version")
        std::cout << "Display all version and copyright informations. Simply." << std::endl;
    else
        std::cout << "Command not known !" << std::endl;
}

void
Cli::notify (std::list<std::string> founded_files)
{
    if (Rfiles.size () == 0)
        Rfiles = founded_files;
    else
        for (std::list<std::string>::iterator founded_file = founded_files.begin ();
             founded_file != founded_files.end (); founded_file++)
             Rfiles.push_back (*founded_file);
}

std::string
Cli::prompt ()
{
    std::string command;
    std::cout << "\033[22;34mfisoco > \033[0;0m";
    getline (std::cin, command);
    return command;
}

void
Cli::read ()
{
    int i = 1;
    if (Rfiles.size () == 0)
        std::cout << "No files to read" << std::endl;
    else
    {
        // Display the list
        for (std::list<std::string>::iterator file = Rfiles.begin (); file != Rfiles.end (); file++, i++)
        {
            std::cout << "\t" << i << " - " << *file << std::endl;
        }
    }
}

void
Cli::search (std::string arg)
{
    start_search (arg);
    std::cout << arg << " has been searched. Type 'read' to see results." << std::endl;
}

int
Cli::start_search (Glib::ustring search)
{
    // Start a new worker thread.
    std::string dir = "/home/fefe";
    // Start a new worker thread.
    cli_WorkerThread =
    Glib::Threads::Thread::
    create (
        sigc::bind
        (
            sigc::bind<Glib::ustring>
            (
                sigc::bind<std::string>
                (
                    sigc::mem_fun
                    (cli_Worker,
                     &Search::do_work_only),
                 dir),
             search),
         this));
    return 1;
}
