/*
 * Fisoco, a FInding SOrting and COnverting free software
 * Copyright 2015 Félicien PILLOT <felicien.pillot@member.fsf.org>
 *
 * This file is part of Fisoco.
 *
 * Fisoco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Fisoco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Fisoco.  If not, see <http://www.gnu.org/licenses/>.
 *
 *//////////////////////////////////////
/****************
 *    gui.cc    *
 *    ~~~~~~    *
 ****************/

#include "../include/gui.h"

Gui::Gui () :
  gui_VBox (Gtk::ORIENTATION_VERTICAL, 5),
  gui_ButtonBox (Gtk::ORIENTATION_HORIZONTAL),
  gui_ButtonQuit (),
  gui_search_button (gettext ("Search")),
  cancel_button_ (gettext ("Cancel")),
  gui_ProgressBar (),
  gui_ScrolledWindow (),
  gui_TextView (),
  gui_Dispatcher (),
  gui_Worker (),
  gui_WorkerThread (0)
{
  set_title (gettext ("Fisoco - Main window"));
  set_border_width (5);
  set_default_size (800, 500);

  add (gui_VBox);

  // Add the MenuBar.
  init_menubar();
  gui_VBox.pack_start (*menubar, Gtk::PACK_SHRINK);

  // Add labels
  gui_VBox.pack_start (gui_text_top, Gtk::PACK_SHRINK);
  gui_text_top.set_markup (gettext ("Please choose your files, then conv"
				      "ert or move them"));
  gui_text_top.set_justify (Gtk::JUSTIFY_CENTER);
  gui_text_top.set_line_wrap ();

  // Entry
  gui_VBox.pack_start (gui_search_entry, Gtk::PACK_SHRINK);
  gui_search_entry.set_max_length (60);
  gui_search_entry.set_placeholder_text (gettext ("Search files..."));
  gui_search_entry.set_icon_from_icon_name ("edit-find");
  gui_search_entry.set_activates_default (true);
  gui_search_entry.set_width_chars (10);
  gui_search_entry.has_focus ();
  
  // Search buttons
  gui_VBox.pack_start (gui_search_button_box, Gtk::PACK_SHRINK);
  gui_set_location_entry.set_max_length (50);
  gui_set_location_entry.
    set_placeholder_text (gettext ("Where to look for..."));
  gui_search_button_box.
    pack_start (gui_set_location_entry, Gtk::PACK_EXPAND_PADDING);
  gui_search_button_box.pack_start (cancel_button_);
  gui_search_button_box.pack_start (gui_search_button);
  gui_search_button_box.set_layout (Gtk::BUTTONBOX_SPREAD);
  gui_search_button_box.set_spacing (20);

  // Add the ProgressBar.
  gui_VBox.pack_start (gui_ProgressBar, Gtk::PACK_SHRINK);
  gui_ProgressBar.set_pulse_step (10);

  // Treeview definition
  gui_refTreeModel = Gtk::TreeStore::create(gui_columns);
  gui_TreeView.set_model(gui_refTreeModel);
  gui_TreeView.append_column(gettext ("Results"), gui_columns.m_col_id);
  gui_TreeView.append_column(gettext ("Name"), gui_columns.m_col_name);
  gui_TreeView.append_column(gettext ("Type"), gui_columns.m_col_type);
  gui_TextView.set_editable (false);

  gui_ScrolledWindow.add (gui_TreeView);
  gui_ScrolledWindow.set_policy (Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  gui_VBox.pack_start (gui_ScrolledWindow);

  // Add the buttons to the ButtonBox.
  gui_VBox.pack_start (gui_ButtonBox, Gtk::PACK_SHRINK);
  gui_ButtonQuit.set_color (Gdk::Color("blue"));
  gui_ButtonQuit.set_label (gettext ("Quit"));
  gui_ButtonBox.pack_start (gui_ButtonQuit, Gtk::PACK_SHRINK);
  gui_ButtonBox.set_border_width (5);
  gui_ButtonBox.set_spacing (5);
  gui_ButtonBox.set_layout (Gtk::BUTTONBOX_END);
  
  // Treeview selection
  Glib::RefPtr<Gtk::TreeSelection> refTreeSelection =
    gui_TreeView.get_selection();
  refTreeSelection->set_mode(Gtk::SELECTION_MULTIPLE);
  // Treeview sort
  for (guint i = 0; i < 3; i++)
    {
      Gtk::TreeView::Column* pCol = gui_TreeView.get_column(i);
      pCol->set_sort_column(gui_columns.m_col_id);
    }

  // Connect the signal handlers to the buttons.
  gui_search_button.signal_clicked ().connect (sigc::mem_fun (*this, &Gui::on_start_button_clicked));
  cancel_button_.signal_clicked ().connect (sigc::mem_fun (*this, &Gui::on_stop_button_clicked));
  gui_ButtonQuit.signal_clicked ().connect (sigc::mem_fun (*this, &Gui::on_quit_button_clicked));
  refActionGroup->add_action
    ("adddevice",sigc::mem_fun (*this,&Gui::menu_add_device));
  refActionGroup->add_action
    ("quit",sigc::mem_fun (*this,&Gui::menu_quit));
  refActionGroup->add_action
    ("manual",sigc::mem_fun (*this,&Gui::menu_manual));
  refActionGroup->add_action
    ("about",sigc::mem_fun (*this,&Gui::menu_about));
  refTreeSelection->selected_foreach_iter
    (sigc::mem_fun (*this, &Gui::selected_row_callback) );
  gui_TreeView.signal_row_activated()
    .connect (sigc::mem_fun (*this, &Gui::on_treeview_row_activated) );
  
  gui_Dispatcher
    .connect (sigc::mem_fun (*this, &Gui::on_notification_from_worker_thread));
  update_start_stop_buttons ();
  show_all_children ();
}

void
Gui::on_treeview_row_activated(const Gtk::TreeModel::Path& path,
			    Gtk::TreeViewColumn* column)
{
  Gtk::TreeModel::iterator iter = gui_refTreeModel->get_iter(path);
  if(iter)
    {
      Gtk::TreeModel::Row row = *iter;
      if (row[gui_columns.m_col_type] == "File")
	std::cout << "Selected file : " <<
	  row[gui_columns.m_col_name] << std::endl;
      else
	std::cout << "Selected directory : " <<
	  row[gui_columns.m_col_name] << std::endl;
    }
}

void
Gui::selected_row_callback (const Gtk::TreeModel::iterator& iter)
{
  Gtk::TreeModel::Row row = *iter;
  Glib::ustring rowName = row[gui_columns.m_col_name];
}

void
Gui::error_dialog (std::string message)
{
  Gtk::MessageDialog dialog (*this, gettext ("Error"));
  dialog.set_secondary_text (message);
  dialog.run ();
}

void
Gui::on_start_button_clicked ()
{
  if (gui_WorkerThread)
    {
      std::cout << gettext ("Can't start a worker thread while another one is running.") << std::endl;
    }
  else
    {
    // Erase previous data
      clean_results ();
      // Start a new worker thread.
      Glib::ustring search = gui_search_entry.get_text ();
      std::string dir = gui_set_location_entry.get_text ();
      gui_WorkerThread =
      Glib::Threads::Thread::
      create (
          sigc::bind
          (
              sigc::bind<Glib::ustring>
              (
                  sigc::bind<std::string>
                  (
                      sigc::mem_fun
                      (gui_Worker,
                       &Search::do_work),
                   dir),
               search),
           this));
    }
  update_start_stop_buttons ();
}

void
Gui::on_stop_button_clicked ()
{
  if (!gui_WorkerThread)
  {
    std::cout << gettext ("Can't stop a worker thread. None is running.") << std::endl;
  }
  else
  {
   // Order the worker thread to stop.
    gui_Worker.stop_work ();
    cancel_button_.set_sensitive (false);
  }
}

void
Gui::update_start_stop_buttons ()
{
  const bool thread_is_running = gui_WorkerThread != 0;
  gui_search_button.set_sensitive (!thread_is_running);
  cancel_button_.set_sensitive (thread_is_running);
}

void
Gui::clean_results ()
{
  //  work to do here

}

void
Gui::update_results ()
{
  if (Rfiles.size () != 0)
    {
      Gtk::TreeModel::Row row = *(gui_refTreeModel->append());
      row[gui_columns.m_col_id] = Rfiles.size ();
      row[gui_columns.m_col_type] = gettext ("Directory");
      row[gui_columns.m_col_name] = Rdir;
      
      for (std::list<std::string>::iterator f = Rfiles.begin ();
           f != Rfiles.end (); ++f)
           {
               Gtk::TreeModel::Row childrow =
               *(gui_refTreeModel->append(row.children()));
               childrow[gui_columns.m_col_type] = gettext ("File");
               childrow[gui_columns.m_col_name] = *f;
           }
    }
}

void
Gui::on_quit_button_clicked ()
{
    if (gui_WorkerThread)
    {
        gui_Worker.stop_work ();
        gui_WorkerThread->join ();
    }
    hide ();
}

// notify () is called from Search::do_work (). It is executed in the worker
// thread. It triggers a call to on_notification_from_worker_thread (), which is
// executed in the GGUI thread.
void
Gui::notify (std::list<std::string> founded_files,
	    std::string founded_dir)
{
  Rfiles = founded_files;
  Rdir = founded_dir;
  gui_Dispatcher.emit ();
}
// It can be called without parameters (when worker has stopped)
void
Gui::notify ()
{
  gui_Dispatcher.emit ();
}

void
Gui::on_notification_from_worker_thread ()
{
    if (gui_WorkerThread && gui_Worker.has_stopped ())
    {
        gui_ProgressBar.set_fraction (100.0);
        gui_WorkerThread->join ();
        gui_WorkerThread = 0;
        update_start_stop_buttons ();
    }
    update_results ();
}

void
Gui::init_menubar ()
{
  refActionGroup = Gio::SimpleActionGroup::create ();
  m_refBuilder = Gtk::Builder::create ();
  try
  {
      m_refBuilder->add_from_file (PACKAGE_UI_DIR "/interface.ui");
  }
  catch (const Glib::Error& ex)
  {
      std::cerr << gettext ("building menus failed: ") <<  ex.what ();
  }
  Glib::RefPtr<Glib::Object> object =
  m_refBuilder->get_object ("menuact");
  gmenu = Glib::RefPtr<Gio::Menu>::cast_dynamic (object);
  if (!gmenu)
    g_warning (gettext ("GMenu not found"));
  menubar = Gtk::manage (new Gtk::MenuBar (gmenu));
}

void
Gui::menu_add_device ()
{
  std::cout << gettext ("The action 'add a device' isn't yet implemented.")
	    << std::endl;
}

void
Gui::menu_manual ()
{
  std::cout << gettext ("The manual isn't yet written.") << std::endl;
}

void
Gui::menu_about ()
{
  std::cout << ("About what ?") << std::endl;
}

void
Gui::menu_quit ()
{
  on_quit_button_clicked ();
}
