/*
 * Fisoco, a FInding SOrting and COnverting free software
 * Copyright 2015-2016 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 *
 * This file is part of Fisoco.
 *
 * Fisoco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Fisoco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Fisoco.  If not, see <http://www.gnu.org/licenses/>.
 *
 *//////////////////////////////////////
 /***************
 *    cli.cc    *
 *    ~~~~~~    *
 ****************/
 
 #include "../include/cli.h"
 #include "../include/global.h"
 #include "../include/search.h"
 #include <regex>
 
 Cli::Cli ()
 {
     debug (gettext ("Debug mode activated, and Fisoco (CLI version) launched !"));
     
     std::cout << gettext ("Welcome to Fisoco's Command Line Interface !") << std::endl <<
     gettext ("Type 'help' to view a list of commands") << std::endl;    
     std::string command = prompt ();
     
     while (command != "q" && command != "quit")
     {
         // Check if the command contains exactly two words        
         int number_of_args = check_args (command);
         switch (number_of_args)
         {
             case 1:
                 if (command == "b" || command == "backup")
                     backup ();
                 else if (command == "c" || command == "clean")
                     clean ();
                 else if (command == "h" || command == "help")
                     help ();
                 else if (command == "i" || command == "info")
                     help ("info");
                 else if (command == "l" || command == "load")
                     load ();
                 else if (command == "r" || command == "read")
                     read ();
                 else if (command == "s" || command == "search")
                     help ("search");
                 else if (command == "v" || command == "version")
                     display_version ();
                 break;
             case 2:
             {
                 std::regex pattern ("^(\\w+)\\s+(\\w+)");
                 std::smatch matches;
                 regex_search (command, matches, pattern);
                 if (matches[1] == "b" || matches[1] == "backup")
                     backup (matches[2]);
                 if (matches[1] == "h" || matches[1] == "help")
                     help (matches[2]);
                 if (matches[1] == "i" || matches[1] == "info")
                     info (std::stoi (matches[2]));
                 if (matches[1] == "l" || matches[1] == "load")
                     load (matches[2]);
                 if (matches[1] == "s" || matches[1] == "search")
                     search (matches[2]);
                 break;
             }
             case 3:
                 std::regex pattern_info ("^info \\d \\d");
                 std::regex pattern_i ("^i \\d \\d");
                 std::smatch matches;
                 int info_present = regex_search (command, matches, pattern_info);
                 int i_present = regex_search (command, matches, pattern_i);
                 if (info_present || i_present)
                 {
                     std::regex pattern ("(\\d+)");
                     while (std::regex_search (command, matches, pattern))
                     {
                         info (std::stoi (matches[1]));
                         command = matches.suffix ().str ();
                     }
                 }
                 else
                     std::cout << gettext ("Command not known, too many arguments, you typed your query whith your eyes closed... I don't know..") << std::endl;
                 break;
         }
         command = prompt ();        
     }
 }
 
 void
 Cli::backup ()
 {
     // Work to do
     std::cout << gettext ("Backup is in hardly development") << std::endl;
 }
 
 void
 Cli::backup (std::string filename)
 {
     // Work to do
     std::cout << "Backup " << filename << gettext (" is in hardly development") << std::endl;
 }
 
 int
 Cli::check_args (std::string command)
 {
     std::regex pattern;
     // First check : command isn't empty
     if (command == "")
         return 0;
     else
     {
         // Second check : no more than two words (command + arg)
         pattern = ("^\\w+\\s+\\w+\\s+\\w+");
         if (regex_search (command, pattern))
             return 3;
         else
         {
             // Third check : no less than two words
             pattern = ("^\\w+");
             if (regex_match (command, pattern))
                 return 1;
             else
             {
                 return 2;
             }
         }
     }
 }
 
 void
 Cli::clean ()
 {
 }
 
 void
 Cli::display_version ()
 {
     std::cout << std::endl
     << "Fisoco v0.6 (alpha) : " <<
     gettext ("a Finding, Sorting and Converting free software.") << std::endl
     << std::endl
     << "Copyright (C) 2015 Felicien Pillot <felicien.pillot@member.fsf.org>" << std::endl
     << "Fisoco is free software: you can redistribute it and/or modify" << std::endl
     << "under the terms of the GNU General Public License." << std::endl
     << std::endl
     << "Fisoco is distributed in the hope that it will be useful," << std::endl
     << "but WITHOUT ANY WARRANTY; without even the implied warranty of" << std::endl
     << "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the" << std::endl
     << "GNU General Public License for more details." << std::endl
     << std::endl;
 }

 void
 Cli::help ()
 {
     std::cout << "  \033[1mhelp [COMMAND]\033[00;0m\t\t\t" <<
     gettext ("print general help, or help on COMMAND")
     << std::endl << "  \033[1mbackup\033[00;0m\t\t\t\t" <<
     gettext ("save the current list of files searched/selected")
     << std::endl << "  \033[1mclean\033[00;0m\t\t\t\t\t" <<
     gettext ("clean all selected elements from the list")
     << std::endl << "  \033[1mload [BACKUP]\033[00;0m\t\t\t\t" <<
     gettext ("load the backup given as parameter")
     << std::endl << "  \033[1minfo N1 [N2 N3 ...]\033[00;0m\t\t\t" <<
     gettext ("display more information about file(s) given as parameter")
     << std::endl << "  \033[1mread\033[00;0m\t\t\t\t\t" <<
     gettext ("display the list of files actually selected")
     << std::endl << "  \033[1msearch FILE\033[00;0m\t\t\t\t" <<
     gettext ("search FILE on system")
     << std::endl << "  \033[1mquit\033[00;0m\t\t\t\t\t" <<
     gettext ("exit program")
     << std::endl << "  \033[1mversion\033[00;0m\t\t\t\t" <<
     gettext ("version and copyright information") << std::endl;
 }
 
 void
 Cli::help (std::string argument)
 {
     std::cout << "  \033[1m" << argument << " :\033[00;0m" << std::endl;
     if (argument == "help")
         std::cout << gettext ("Are you kidding me ?") << std::endl;
     else if (argument == "backup")
         std::cout << gettext ("This command accepts one argument. Without argument, it backups the list of files founded or selected into a text file located at '$HOME/.config/fisoco/list_[number].txt'. You can set a different filename by giving it as parameter.") << std::endl;
     else if (argument == "clean")
         std::cout << gettext ("This command requires no arguments, it cleans all elements selected (or not) from the list.") << std::endl << gettext ("You can use this when you want to cancel your search and start a new one.") << std::endl;
     else if (argument == "info")
         std::cout << gettext ("This command requires one or more argument(s). It displays all informations (path, size, owner, created date, etc.) about file(s) whose number(s) are given as parameter(s).") << std::endl;
     else if (argument == "load")
         std::cout << gettext ("This command requires one argument. It opens and loads the backup given as parameter, or, if no parameters are given, the last saved.") << std::endl;
     else if (argument == "read")
         std::cout << gettext ("This command requires no arguments, it displays all elements selected (or just all elements founded, if you haven't yet selected some of them).") << std::endl;
     else if (argument == "search")
         std::cout << gettext ("This command requires one argument. It takes the FILE or expression given as parameter, and look for it on the entire system (at this point of the development).") << std::endl;
     else if (argument == "quit")
         std::cout << gettext ("Are you kidding me ?") << std::endl;
     else if (argument == "version")
         std::cout << gettext ("Display all version and copyright informations. Simply.") << std::endl;
     else
         std::cout << gettext ("Command not known !") << std::endl;
 }
 
 void
 Cli::info (int number)
 {
     std::vector<std::string> infos = results.get_infos (number);
     if (infos.size () == 0)
         error ("No information / File unknown : check the list by typing 'read'");
     else
     {
         char const *x[] = {"Name", "Path"};
         std::vector<std::string> keys (x, x + sizeof (x) / sizeof (*x));
         // Display
         std::cout << number;
         for (std::vector<std::string>::iterator info = infos.begin (), key = keys.begin (); info != infos.end (); info++, key++)
             std::cout << "\t -> " << *key << " : " << *info << std::endl;
         std::cout << std::endl;
     }
 }
 
 void
 Cli::load ()
 {
     // Work to do
     std::cout << gettext ("load is in hardly development.") << std::endl;
 }
 
 void
 Cli::load (std::string filename)
 {
     // Work to do
     std::cout << "load " << filename << gettext (" is in hardly development.") << std::endl;
 }
 
 void
 Cli::notify (std::list<std::string> files, std::string dir)
 {
     for (std::list<std::string>::iterator file = files.begin (); file != files.end (); file++)
         results.add_file (dir, *file);
 }
 
 std::string
 Cli::prompt ()
 {
     std::string command;
     std::cout << "\033[22;34mfisoco > \033[0;0m";
     getline (std::cin, command);
     return command;
 }
 
 void
 Cli::read ()
 {
     int i = 1;
     std::list<std::string> list_names = results.get_complete_names ();
     if (list_names.size () != 0)
     {
         for (std::list<std::string>::iterator complete_name = list_names.begin ();
              complete_name != list_names.end (); complete_name++, i++)
              {
                  std::cout << i << "\t- " << *complete_name << std::endl;
              }
     }
     else
         std::cout << "List empty : no files searched, founded or selected" << std::endl;
 }
 
 void
 Cli::search (std::string arg)
 {
     // New list
     results = List ();
     // Launch process
     start_search (arg);
     
     std::cout << arg << gettext (" is being searched. Type 'read' to see results (you may wait a while before all results are displayed).") << std::endl;
 }
 
 int
 Cli::start_search (Glib::ustring search)
 {
     // Start a new worker thread.
     std::string dir = "/";
     // Start a new worker thread.
     cli_WorkerThread =
     Glib::Threads::Thread::
     create (
         sigc::bind
         (
             sigc::bind<Glib::ustring>
             (
                 sigc::bind<std::string>
                 (
                     sigc::mem_fun
                     (cli_Worker,
                      &Search::do_work_only),
                  dir),
              search),
          this));
     return 1;
 }
 
