/*
 * Fisoco : FInd, Select, Order and COnvert files
 * Copyright 2015-2016 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 *
 * This file is part of Fisoco.
 *
 * Fisoco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Fisoco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Fisoco.  If not, see <http://www.gnu.org/licenses/>.
 *
 *//////////////////////////////////////
/***************
 *  prompt.cc  *
 *   ~~~~~~    *
 ***************/

#include "../include/cli.h"
#include "../include/global.h"
#include "../include/search.h"
#include <map>
#include <sstream>
#include <stdio.h>

/* Cli ()
 * Welcomes users and runs the main loop.
 */
Cli::Cli ()
{
  debug (gettext ("Debug mode activated, and Fisoco (CLI version) launched !"));
  
  std::cout << gettext ("Welcome to Fisoco's command line interface !") << std::endl <<
    gettext ("Type 'help' to view a list of commands") << std::endl;    
  std::string input = prompt ();
  std::list<std::string> command = parse_input (input);

  while (command.front () != "quit")
    {
      // Debug
      int size = command.size ();
      int i = 1;
      debug ("Input contains " + std::to_string (size) + " items.");
      for (std::list<std::string>::iterator arg = command.begin ();
	   arg != command.end (); arg++)
	{
	  debug (std::to_string (i) + "# argument is " + *arg);
	  i++;
	}
      // Examine the command, then capture another one
      examine (command);
      input = prompt ();        
      command = parse_input (input);
    }
}

/* bind_command ()
 * From a letter or a word, checks if it is a known command, and returns it.
 */
std::string
Cli::bind_command (std::string cinitial, std::string winitial)
{
  std::string command = "";
  // If only one letter was given, associate it with corresponding command
  if (winitial == cinitial)
    {
      winitial = ( (cinitial == "b")  ?  "backup"  :
		   (cinitial == "c")  ?  "clean"   :
		   (cinitial == "h")  ?  "help"    :
		   (cinitial == "l")  ?  "load"    :
		   (cinitial == "i")  ?  "info"    :
		   (cinitial == "q")  ?  "quit"    :
		   (cinitial == "r")  ?  "read"    :
		   (cinitial == "s")  ?  "search"  :
		   (cinitial == "v")  ?  "version" :
		   (cinitial == "")  ?  "" :
		   "error");
    }
  // Check if winitial is really a command
  if (winitial != "backup" && winitial != "clean" &&
      winitial != "help" && winitial != "load" &&
      winitial != "info" && winitial != "quit" &&
      winitial != "read" && winitial != "search" &&
      winitial != "version" && winitial != "")
    error ("Command unknown !");
  else
    command = winitial;
  // Return it as the command to execute
  return command;
}

/* examine ()
 * Examines, deal with the given command and its arguments
 */
void
Cli::examine (std::list<std::string> command)
{
  std::string command_name = command.front ();
  // Examine info args
  if (command_name == "backup")
    {
      if (command.size () < 2)
	backup ();
      else if (command.size () > 2)
	error ("Please supply a maximum of one argument for this command...");
      else
	{
	  std::list<std::string>::iterator arg = command.end ();
	  backup (*arg);
	}
    }
  if (command_name == "info")
    {
      if (command.size () < 2)
	error ("Info about what ? You need to specify some number(s) !");
      std::list<std::string>::iterator arg = command.begin ();
      arg++;
      for (; arg != command.end (); arg++)
	{
	  int number;
	  std::istringstream iss (*arg);
	  iss >> number;
	  info (number);
	}
    }
  else if (command_name == "load")
    {
      if (command.size () < 2)
	load ();
      else if (command.size () > 2)
	error ("Please supply only one argument for this command...");
      else
	{
	  std::list<std::string>::iterator arg = command.begin ();
	  arg++;
	  int number;
	  std::istringstream iss (*arg);
	  iss >> number;
	  load (number);
	}
    }
  else if (command_name == "search")
    {
      if (command.size () != 2)
	error ("Please supply an argument for this command.");
      else
	{
	  std::list<std::string>::iterator arg = command.begin ();
	  arg++;
	  dir_term = "/";
	  search_term = *arg;
	  search ();
	  std::cout << *arg << gettext (" is being searched. Type 'read' to see results (you may wait a while before all results are displayed).") << std::endl;
	}
    }
  else if (command_name == "read")
    {
      if (command.size () == 1)
	interactive_read ();
      else
	error ("This command does not require any argument.");
    }
  else if (command_name == "help")
    {
      if (command.size () < 2)
	help ("");
      else if (command.size () > 2)
	error ("Please supply a maximum of one argument for this command...");
      else
	{
	  std::list<std::string>::iterator arg = command.begin ();
	  arg++;	  
	  help (*arg);
	}
    }
  else if (command_name == "clean")
    {
      if (command.size () == 1)
	clean ();
      else
	error ("This command does not require any argument.");
    }
}

/* display_version ()
 * Displays copyright information and release number.
 */
void
Cli::display_version ()
{
  version ();
}

/* get_help_entries ()
 * Returns all help entries of this program
 */
std::map<std::string,Cli::entry>
Cli::get_help_entries ()
{
  // Declare all entries
  entry help, backup, clean, load, info, read, search, quit, version;
  
  // Initialize them
  help.command = "help";
  help.arguments = "[COMMAND]";
  help.short_desc = gettext ("print general help, or help on COMMAND");
  help.long_desc = "Hahaha.";
  backup.command = "backup";
  backup.arguments = "[NAME]";
  backup.short_desc = gettext ("save the current list of files searched/selected");
  backup.long_desc = gettext ("This command accepts one argument. Without argument, it backups the list of files founded or selected into a text file located at '$HOME/.config/fisoco/list_[number].txt'. You can set a different filename by giving it as parameter.");
  clean.command = "clean";
  clean.arguments = "";
  clean.short_desc = gettext ("clean all selected elements from the list");
  clean.long_desc = gettext ("This command requires no arguments, it cleans all elements selected (or not) from the list.  You can use this when you want to cancel your search and start a new one.");
  load.command = "load";
  load.arguments = "[BACKUP]";
  load.short_desc = gettext ("load the backup given as parameter");
  load.long_desc = gettext ("This command requires one argument. It opens and loads the backup given as parameter, or, if no parameters are given, the last saved.");
  info.command = "info";
  info.arguments = "(N1 [N2 ...] | N1-NX)";
  info.short_desc = gettext ("display information about given file(s)");
  info.long_desc = gettext ("This command requires one or more argument(s). It displays all informations (path, size, owner, created date, etc.) about file(s) whose number(s) are given as parameter(s). Numbers can be discrete or continuous.");
  read.command = "read";
  read.arguments = "";
  read.short_desc = gettext ("display the current list of files");
  read.long_desc = gettext ("This command requires no arguments, it displays all elements selected (or just all elements founded, if you haven't yet selected some of them).");
  search.command = "search";
  search.arguments = "FILE";
  search.short_desc = gettext ("search FILE on system");
  search.long_desc = gettext ("This command requires one argument. It takes the FILE or expression given as parameter, and look for it on the entire system (at this point of the development).");
  quit.command = "quit";
  quit.arguments = "";
  quit.short_desc = gettext ("exit program");
  quit.long_desc = "Haha.";
  version.command = "version";
  version.arguments = "";
  version.short_desc = gettext ("Version and copyright information");
  version.long_desc = gettext ("Display all version and copyright informations. Simply.");
  // Store them in a map
  std::map<std::string,Cli::entry> all_entries;
  all_entries["help"] = help;
  all_entries["backup"] = backup;
  all_entries["clean"] = clean;
  all_entries["load"] = load;
  all_entries["info"] = info;
  all_entries["read"] = read;
  all_entries["search"] = search;
  all_entries["quit"] = quit;
  all_entries["version"] = version;

  // Return them
  return all_entries;
}

/* help ()
 * Displays every help entries, or just the one given in argument
 */
void
Cli::help (std::string argument)
{
  std::map<std::string,Cli::entry> entries = get_help_entries ();
  std::list<std::string> keys = {"help", "backup", "clean", "load", "info",
	    "read", "search", "quit", "version"};
  std::list<std::string>::iterator key;
  if (argument == "")
    // Loop through entries
    for (key = keys.begin (); key != keys.end (); ++key)
      // Display them
      printf (" %-7s   %-21s    %-20s\n",
	      entries[(*key)].command.c_str (),
	      entries[(*key)].arguments.c_str (),
	      entries[(*key)].short_desc.c_str ());
  else
    {
      // Find entry corresponding to the argument
      for (key = keys.begin (); entries[(*key)].command != argument; ++key)
	{
	  if (key == keys.end () && entries[(*key)].command != argument)
	    error ("Cannot give help about unknown command !");
	}
      // Display it
      printf ("\033[0;1m%-7s\033[0m :\n\t%-20s\n",
	      entries[(*key)].command.c_str (),
	      entries[(*key)].long_desc.c_str ());
    }
}

/* parse_input ()
 * Takes user's input (could be anything), extracts the command
 * and eventually the arguments, and returns them in a std::list.
 */
std::list<std::string>
Cli::parse_input (std::string input)
{
  size_t first_space, other_space;
  first_space = input.find (" ");
  std::list<std::string> args;
  
  std::string cinitial = input.substr (0,1);
  std::string winitial = input.substr (0, first_space);
  
  // This checks the first word, and associate it with the correct command
  std::string command = bind_command (cinitial, winitial);
  args.push_back (command);

  if (first_space != std::string::npos)
    {
      other_space = input.find (" ", ++first_space);
      while (other_space != std::string::npos)
	{
	  args.push_back (input.substr (first_space, other_space-first_space));
	  first_space = other_space;
	  other_space = input.find (" ", ++first_space);
	}
      args.push_back (input.substr (first_space));
    }
  return args;
}

std::string
Cli::prompt ()
{
  std::string command;
  std::cout << "\033[22;34mfisoco > \033[0;0m";
  getline (std::cin, command);
  return command;
}
