/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <errno.h>
#include <string.h>
#include <stdio.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <string>
#include <map>

#include "Archiver.h"
#include "flatback.h"
#include "sha1sum.h"
#include "FilteredMemoryCache.h"
#include "FilteredWriteCache.h"

/*
	Constructor
		
	Arguments:
		write_callback - Write callback function
*/
Archiver::Archiver(TArchiverWriteCallback write_callback, TArchiverGetOffset get_offset)
{
  m_index = NULL;
  m_write = write_callback;
  m_get_offset = get_offset;
  m_compression_level = 0;
  m_max_mem_cache = 4*1024*1024;
}

Archiver::~Archiver()
{
	if(m_index)
		delete m_index;
}

/*
	Function:
		CreateArchive
		
	Arguments:
		use_index - Use an index?
    compression_level - Use compression?
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Writes out archive header.
*/
bool Archiver::CreateArchive(bool use_index, unsigned int compression_level)
{
  m_compression_level = compression_level;

	if(m_index)
  {
		delete m_index;
    m_index = NULL;
  }

  if(use_index)
  {
    m_index = new Index(m_write);
    if(!m_index)
      return false;
  }

	archive_header hdr;
	hdr.endian_arch = 255;
	hdr.format_version = FORMAT_VERSION;
	return m_write((const char *)&hdr, sizeof(hdr));
}

/*
	Function:
		CommitArchive
		
	Arguments:
		None
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Finalizes archive and writes the index
*/
bool Archiver::CommitArchive()
{
  bool ret = true;

  /* Create a marker that states the end of archive, beginning of optional index */
  u_int8_t flags = 128; /* Set bit 8 (highest bit) indicating end of archive */
  
  if(!m_write((const char *)&flags, sizeof(flags)))
    ret = false;
  
	if(ret && m_index)
  {
    if(!m_index->CommitIndex())
      ret = false;

		delete m_index;
    m_index = NULL;
  }

	return ret;
}

/*
	Function:
		PrintSettings
		
	Arguments:
		None
	
	Returns:
		Nothing
		
	Remarks:
		Prints the important class variables that can be set externally.
*/
void Archiver::PrintSettings()
{
  fprintf(stderr, "Max Memcache: %llu bytes\n", m_max_mem_cache);
}

/*
	Function:
		StatPack
		
	Arguments:
		filename - file to stat and pack
		file_stat - stat of the file
    use_compression - boolean is compression on or off?
		statbuf - packed metadata about the file
	
	Returns:
		length of file data for a regular file
		or 0 for any other file types.
		DIEs upon error, shouldn't usually happen.
		
	Remarks:
		Packs the file's metadata and handles hardlinks.
*/
off_t Archiver::StatPack(const char *filename, const struct stat &file_stat, bool use_compression, std::string *statbuf)
{
  const char *linked_filename = NULL;
  u_int8_t flags = (use_compression ? 1 : 0);

	if(!S_ISDIR(file_stat.st_mode))
	{
		//Check for hardlinks
		if((file_stat.st_nlink > 1) && (m_hardlink_nodes.find(file_stat.st_ino) == m_hardlink_nodes.end()))
		{
      //This is a hardlink but the first file, so treat it normally and cache the filename and inode
      m_hardlink_nodes[file_stat.st_ino] = filename;
		}
		else if((file_stat.st_nlink > 1) && (m_hardlink_nodes.find(file_stat.st_ino) != m_hardlink_nodes.end()))
		{
			//This is a hardlink
			linked_filename = ((*m_hardlink_nodes.find(file_stat.st_ino)).second).c_str();

			flags |= 2; //Set hardlink bit
		}
	}

  statbuf->append((const char *)&flags, sizeof(u_int8_t));

	statbuf->append(filename, strlen(filename)+1);
	if(flags & 2) //If this is a hardlink
		statbuf->append(linked_filename, strlen(linked_filename)+1);
	
	statbuf->append((const char *)&file_stat.st_mode, sizeof(mode_t));
	statbuf->append((const char *)&file_stat.st_uid, sizeof(uid_t));
	statbuf->append((const char *)&file_stat.st_gid, sizeof(gid_t));
	statbuf->append((const char *)&file_stat.st_rdev, sizeof(dev_t));
	statbuf->append((const char *)&file_stat.st_size, sizeof(off_t));
	statbuf->append((const char *)&file_stat.st_atime, sizeof(time_t));
	statbuf->append((const char *)&file_stat.st_mtime, sizeof(time_t));
	statbuf->append((const char *)&file_stat.st_ctime, sizeof(time_t));
	
	//If this is not a regular file or it is a hardlink return 0
	if((!S_ISREG(file_stat.st_mode)) || (flags & 2))
		return 0;

	return file_stat.st_size;
}

/*
	Function:
		AddToArchive
		
	Arguments:
		filename - Name of file to add to archive
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Adds a file to the archive.
*/
bool Archiver::AddToArchive(const char *filename)
{
	std::string statbuf;
	struct stat file_stat;

	if(!(m_index && m_index->AddEntry(m_get_offset())))
		return false;
	
	/* Read the metadata from the filesystem */
	if(lstat(filename, &file_stat) == -1) {
		fprintf(stderr, "stat failed with error '%s'\n", strerror(errno));
		return false;
	}

  if((!file_stat.st_size) || (!S_ISREG(file_stat.st_mode)) || ((file_stat.st_nlink > 1) && (m_hardlink_nodes.find(file_stat.st_ino) != m_hardlink_nodes.end())))
	{
    /* Zero length, Nonregular file, or hardlink, skip all compression related hassles */
    Sha1Sum meta_sum;
    
    file_stat.st_size = 0; /* FIXME: Is st_size used for non-regular files? The current archive format requires it be set to 0 */
    if(StatPack(filename, file_stat, false, &statbuf)) /* A nonregular file or hardlink should cause StatPack to return 0 */
      return false;

    /* Checksum the metadata */
    meta_sum.AddBytes(statbuf.c_str(), statbuf.length());

    /* Write out the metadata */
    if(!m_write(statbuf.c_str(), statbuf.length())) {
      fprintf(stderr, "Write Callback returned failure\n");
      return false;
    }

    char sumbuf[20];
    /* Compute and store checksum */
    meta_sum.Finish(sumbuf);

    /* Write checksum */
    if(!m_write(sumbuf, 20)) {
      fprintf(stderr, "Write callback returned failure\n");
      return false;
    }
  }
  else
  {
    Sha1Sum file_sum;
    off_t total_bytes;
    FilterOutCacheBase *filtered_cache;
    bool use_compression = (m_compression_level ? true : false);

    if(!FilterCacheFileFromDisk(filename, true, use_compression, &filtered_cache, &file_sum, &total_bytes))
      return false;

    /* Don't use compression if the compressed data is larger or equal to the uncompressed data */
    if(use_compression && (total_bytes <= filtered_cache->TotalBytes()))
    {
      /* Delete the cache */
      delete filtered_cache;
      filtered_cache = NULL;

      use_compression = false;

      if(!FilterCacheFileFromDisk(filename, true, use_compression, &filtered_cache, NULL, NULL))
        return false;
    }

    /* Replace st_size with the filtered size of the file */
    file_stat.st_size = filtered_cache->TotalBytes();
    if(!StatPack(filename, file_stat, use_compression, &statbuf))
    {
      /* A non-zero length regular file should cause StatPack to return non-zero */
      delete filtered_cache;
      return false;
    }

    {
      Sha1Sum meta_sum;

      /* Checksum the metadata */
      meta_sum.AddBytes(statbuf.c_str(), statbuf.length());

      /* Write out the metadata */
      if(!m_write(statbuf.c_str(), statbuf.length()))
      {
        fprintf(stderr, "Write Callback returned failure\n");
        delete filtered_cache;
        return false;
      }

      char sumbuf[20];
      /* Compute and store checksum */
      meta_sum.Finish(sumbuf);

      /* Write checksum */
      if(!m_write(sumbuf, 20))
      {
        fprintf(stderr, "Write callback returned failure\n");
        delete filtered_cache;
        return false;
      }
    }

    if(filtered_cache->DataAvailable())
    {
      /* File is available in the cache */
      /* Instruct the cache to write its data to the write callback */
      if(!filtered_cache->Dump())
      {
        delete filtered_cache;
        return false;
      }
    }
    else
    {
      /* File is not available in the cache */
      /* Delete the cache */
      delete filtered_cache;
      filtered_cache = NULL;

      if(!FilterCacheFileFromDisk(filename, false, use_compression, &filtered_cache, NULL, NULL))
        return false;
      
      if(file_stat.st_size != filtered_cache->TotalBytes())
      {
        fprintf(stderr, "ERR: The filtered file size does not match what is stored in the archive header!\n");
        delete filtered_cache;
        return false;
      }
    }

    delete filtered_cache;

    char sumbuf[20];
    /* Compute and store checksum */
    file_sum.Finish(sumbuf);

    /* Write checksum */
    if(!m_write(sumbuf, 20)) {
      fprintf(stderr, "Write callback returned failure\n");
      return false;
    }
	}

	return true;
}

bool Archiver::FilterCacheFileFromDisk(const char *filename, bool use_mem_cache, bool use_compression, FilterOutCacheBase **filtered_cache, Sha1Sum *file_sum, off_t *total_bytes)
{
  if(!filtered_cache)
    return false;

  FilterOutBase *filter;

  if(use_compression)
  {
    /* Compression is on */
    filter = (FilterOutBase *)new ZLibOutFilter((m_compression_level > 9 ? 9 : m_compression_level));
  }
  else
  {
    /* No compression */
    filter = (FilterOutBase *)new NullOutFilter();
  }
  if(!filter)
    return false;

  /* This will take ownership of the filter object */
  if(use_mem_cache)
  {
    *filtered_cache = (FilterOutCacheBase *)new FilteredMemoryCache(filter, m_max_mem_cache, m_write);
  } else {
    *filtered_cache = (FilterOutCacheBase *)new FilteredWriteCache(filter, m_write);
  }
  if(!(*filtered_cache))
  {
    delete filter;
    return false;
  }

  /* Read the file from disk and put it into the filter */
  if(!FilterFileFromDisk(*filtered_cache, filename, file_sum, total_bytes))
  {
    delete *filtered_cache;
    *filtered_cache = NULL;
    return false;
  }
  
  return true;
}

bool Archiver::FilterFileFromDisk(FilterOutCacheBase *filter, const char *filename, Sha1Sum *file_sum, off_t *ammount_read)
{
  /* Catch null pointers */
  if(!filter)
    return false;

  FILE *file_to_archive = fopen(filename, "rb"); /* Open the file to be backed up for reading */

  if(!file_to_archive) {
    fprintf(stderr, "Failed to open '%s'.\nError was '%s'\n", filename, strerror(errno));
    return false;
  }

  size_t bread;
  char buf[1024];

  if(ammount_read)
    *ammount_read = 0;
  
  /* Read in the file up to datasize
     WARNING: File/Archive corruption possible if file changes while archiving.
              Do not backup files that are in use.
  */
  while(1)
  {
    /* Read the file data to be backed up */
    bread = fread(buf, 1, 1024, file_to_archive);
   
    /* Check for end of file condition if nothing was read */
    if((!bread) && feof(file_to_archive))
      break;

    if(ferror(file_to_archive)) {
      fprintf(stderr, "Error reading from '%s'.\n", filename);
      fclose(file_to_archive);
      return false;
    }
    
    if(ammount_read)
      *ammount_read += bread;

    /* Checksum the raw data from the disk */
    if(file_sum)
      file_sum->AddBytes(buf, bread);
    
    /* Write filedata to cache */
    /* These filters are special, they will take care of the data after filtering.
       Either by sending data to a write callback or storing data in memory, etc. */
    if(!filter->PutData(buf, bread)) {
      fprintf(stderr, "Writing to memory cache failed!\n");
      fclose(file_to_archive);
      return false;
    }

    /* Exit loop at the end of the file */
    if(feof(file_to_archive))
      break;
  }
  
  fclose(file_to_archive);
  
  filter->Flush();
 
  return true;
}
