/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "FilteredWriteCache.h"

#define FLUSH_POINT 8192

FilteredWriteCache::FilteredWriteCache(FilterOutBase *filter, TArchiverWriteCallback write_callback) :
  m_write(write_callback), m_filter(filter), m_total_bytes(0)
{
}

FilteredWriteCache::~FilteredWriteCache()
{
  if(m_filter)
    delete m_filter;
}

bool FilteredWriteCache::PutData(const char *data, size_t len)
{
  if(!m_filter)
    return false;

  WriteFilter(data, len);

	return true;
}

bool FilteredWriteCache::Flush()
{
  if(!m_filter)
    return false;

	if(!m_filter->Flush())
		return false;

	return WriteFilter(NULL, 0);
}

/* Push the data through another filter and then send it to the write callback */
bool FilteredWriteCache::WriteFilter(const char *buffer, unsigned int length)
{
	std::string data;

	/* Process data in filter */
	if(buffer)
		m_filter->PutData(buffer, length);

	/* Write the data to disk if it is past the flush point */
	while(m_filter->NeedFlush(FLUSH_POINT))
	{
		if(!m_filter->GetData(data, FLUSH_POINT))
			return false;
		
		if(!Write(data.c_str(), data.length()))
			return false;
	}
	
	data.clear();
	
	/* Flush the rest if the buffer is NULL */
	if(!buffer)
	{
		while(!m_filter->Empty())
		{
			if(!m_filter->GetData(data, FLUSH_POINT))
				return false;
			
			if(!Write(data.c_str(), data.length()))
				return false;
		}
	}
	
	return true;
}

bool FilteredWriteCache::Write(const char *buffer, unsigned int length)
{
  m_total_bytes += length;
  return m_write(buffer, length);
}

