/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>

#include "Archiver.h"
#include "sha1sum.h"

/*
	Constructor
		
	Arguments:
		archiver - Pointer to Archiver object
	
	Remarks:
		Creates a temporary file used to store the index
*/
Index::Index(TArchiverWriteCallback write_callback)
{
	tmp_index_file = tmpfile();
	m_write = write_callback;
}

/*
	Destructor
		
	Remarks:
		Destroies the temporary file used to store the index if it still exists
*/
Index::~Index()
{
	if(tmp_index_file)
		fclose(tmp_index_file);
}

/*
	Function:
		AddEntry
		
	Arguments:
		index - the position to index
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Adds an entry to the index.
*/
bool Index::AddEntry(off_t index)
{
	if(!tmp_index_file)
		return false;
	return (fwrite(&index, sizeof(off_t), 1, tmp_index_file)==1);
}

/*
	Function:
		CommitIndex
		
	Arguments:
		None
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Writes index to the archive and deletes it
*/
bool Index::CommitIndex()
{
	if(!tmp_index_file)
		return false;
	
	Sha1Sum file_sum;
	off_t index_size = 0;
	size_t bread;
	char buf[1024];
	
	/***********************
	Write out the index data
	***********************/
	rewind(tmp_index_file);
	while((bread = fread(buf, 1, 1024, tmp_index_file)) != 0)
	{
		if(!m_write(buf, bread))
			return false;

		/* Checksum the index */
		file_sum.AddBytes(buf, bread);
		
		index_size += bread;

		if(ferror(tmp_index_file) || feof(tmp_index_file))
			break;
	}
	
	/* Check for file I/O errors */
	if(ferror(tmp_index_file))
		return false;
	
	/*************************
	Close and delete the index
	**************************/
	fclose(tmp_index_file);
	tmp_index_file = NULL;

	/************************
	Write out the IndexFooter
	************************/
	IndexFooter idx_tail;
	
	/* Checksum the index size */
	file_sum.AddBytes(&index_size, sizeof(off_t));
	
	idx_tail.index_size = index_size; 	/* Store the index size */
	file_sum.Finish(idx_tail.sumbuf); /* Compute and store checksum */
	memcpy(idx_tail.magic, "INDEX", strlen("INDEX")); /* Store 'INDEX' */
	
	/* Write the idx_tail */
	if(!m_write((const char *)&idx_tail, sizeof(idx_tail)))
		return false;
	
	return true;
}

