/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <string>
#include <strings.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "Slicer.h"

#define FLUSH_POINT 8192

Slicer::Slicer()
{
  m_size = 0;
  m_slice_offset = 0;
  m_slice_num = 0;
  m_archive_pos = 0;
  m_pause_function = NULL;
  m_slice_file_handle = NULL;
}

Slicer::~Slicer()
{
	if(m_slice_file_handle)
		fclose(m_slice_file_handle);
}

/*
	Function:
		SetSize
		
	Arguments:
		size - size of slice
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Sets the size of a slice
*/
bool Slicer::SetSize(const long long int size)
{
	if(size < 0)
		return false;
	m_size = size;
	return true;
}

/*
	Function:
		SetFullName
		
	Arguments:
		full_name - path and basename for slice
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Sets the path and basename of the slice
*/
bool Slicer::SetFullName(const std::string & full_name)
{
	if(full_name.empty())
		return false;

	int last_sep = full_name.find_last_of('/');
	
	if(last_sep >= 0) {
		if(last_sep > 0)
			m_slice_path = full_name.substr(0, last_sep) + "/";
		else
			m_slice_path = "/";
		m_slice_base_name = full_name.substr(last_sep+1);
	} else {
		m_slice_path = "./";
		m_slice_base_name = full_name;
	}
	
	if(m_slice_base_name.empty())
		return false;

	int ext_sep = m_slice_base_name.find_last_of('.');
	
	if((ext_sep >= 0) && (!strcasecmp(m_slice_base_name.substr(ext_sep).c_str(), ".fbk"))) {
		fprintf(stderr, "Basename looks like an archive. Aborting.\n");
		return false;
	}

	return true;
}

void Slicer::PrintSettings()
{
	fprintf(stderr, "Slice Size: %lli bytes\nPause Between Slices: %s\nSlice Path: %s\nSlice Base Name: %s\n", m_size, (m_pause_function ? "Yes" : "No"), m_slice_path.c_str(), m_slice_base_name.c_str());
}

/*
	Function:
		SwitchSlice
		
	Arguments:
		slice_num - slice number to switch to
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Switches to a different slice
*/
bool Slicer::SwitchSlice(unsigned int slice_num)
{
	if(slice_num > 0)
		if(m_pause_function) /* If there is a pause function */
			if(!m_pause_function(slice_num)) /* And the user doesn't want to switch slices */
				return false; /* Then fail */
	
	if(m_slice_file_handle) {
		fclose(m_slice_file_handle);
		m_slice_file_handle = NULL;
	}
	
	m_slice_offset = 0;
	m_slice_num = slice_num;
	
	return true;
}

/*
	Function:
		CreateNewSlice
		
	Arguments:
		None
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Creates a new slice
*/
bool Slicer::CreateNewSlice()
{
	struct stat st;

	if(!SwitchSlice(m_slice_num+1))
		return false;
	
	/* Kludge to generate slice filename
		FIXME: Have a function for generating the slice filename
	*/
	char buf[11];
	snprintf(buf, 11, "%u", m_slice_num);
	std::string filename = (m_slice_path + m_slice_base_name + std::string(".") + std::string(buf) + std::string(".fbk"));
	
	/* Kludge: Assume that if stat fails with any error, it is safe to try and create the slice file
		FIXME: Do it the right way
	*/
	if(stat(filename.c_str(), &st) == -1)
	{
		m_slice_file_handle = fopen(filename.c_str(), "wb");
		
		if(!m_slice_file_handle)
			return false;
		
		if(ferror(m_slice_file_handle)) {
			fclose(m_slice_file_handle);
			m_slice_file_handle = NULL;
			return false;
		}
		
		return true;
	}
	
	return false;
}

/*
	Function:
		HandleSlices
		
	Arguments:
		None
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Creates a new slice if necessary
*/
bool Slicer::HandleSlices()
{
	if((m_slice_num == 0) || (m_size && (m_slice_offset == m_size)))
		return CreateNewSlice();
	
	return true;
}

/*
	Function:
		WriteAmmount
		
	Arguments:
		None
	
	Returns:
		the ammount of data to write
		
	Remarks:
		Determines the next ammount of data to write so we don't overflow slices.
*/
size_t Slicer::WriteAmmount()
{
	/* Determine if we need to flush data to disk */
	if(m_size) {
    off_t flush_point = MIN(FLUSH_POINT, m_size);
		return (flush_point+m_slice_offset > m_size ? m_size-m_slice_offset : flush_point); /* Only flush up to m_size */
	} else
		return FLUSH_POINT;
}

/*
	Function:
		AppendData
		
	Arguments:
		buffer - buffer of data
		length - length of buffer
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Appends data to the archive
*/
bool Slicer::AppendData(const char *buffer, unsigned int length)
{
	size_t ammt;
	unsigned int bwritten = 0;
	for(ammt = WriteAmmount(); bwritten < length; ammt = WriteAmmount())
	{
		/* Swap out slices if necessary */
		if(!HandleSlices())
			return false;
	
		ammt = length-bwritten < ammt ? length-bwritten : ammt;
		
		if(fwrite(buffer+bwritten, 1, ammt, m_slice_file_handle) != ammt)
			return false;
	
		bwritten += ammt;
		m_slice_offset += ammt;
	}

  m_archive_pos += length;

	return true;
}

