/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <sys/param.h>
#include <stdio.h>
#include <limits.h>

#include "Archiver.h"
#include "sha1sum.h"

/*
	Constructor
		
	Arguments:
	
	Remarks:
		Creates a temporary file used to store the index
*/
Index::Index(TArchiverReadCallback read_callback, TArchiverSeek seek_callback)
{
	tmp_index_file = NULL;
	m_read = read_callback;
  m_seek = seek_callback;
}

/*
	Destructor
		
	Remarks:
		Destroies the temporary file used to store the index if it still exists
*/
Index::~Index()
{
	if(tmp_index_file)
		fclose(tmp_index_file);
}

/*
	Function:
		GetEntryOffsetByNumber
		
	Arguments:
	
	Returns:
    1 - No Index found
    0 - Success
    -1 - Index read error
    -2 - Corrupt index
    -3 - Other error
		
	Remarks:
*/
int Index::GetEntryOffsetByNumber(u_int64_t entry_number, off_t *offset)
{
  Sha1Sum idx_sum;
	unsigned char sumbuf[20];
  IndexFooter idx_tail;
  if(!entry_number)
    return -3;
  /* Create a temporary file to hold the index if necessary */
	if(!tmp_index_file)
  {
    tmp_index_file = tmpfile();
    if(!tmp_index_file)
      return -3;

    off_t padding_size;
    for(padding_size = 0; padding_size < LONG_LONG_MAX; padding_size += 5)
    {
      /* Seek to the end of the archive */
      if(!m_seek(sizeof(idx_tail)+padding_size, SEEK_END))
        return -1;
      /* Read in the index tail */
      if(!m_read((char *)&idx_tail, sizeof(idx_tail)))
        return -1;

      /* Check the index tail for all 0
        Only breakout if there is a X found
      */
      {
        bool breakout = false;
        unsigned int i;
        for(i = 0; i < 5; i++)
        {
          if(idx_tail.magic[4-i] == 'X')
          {
            breakout = true;
            if(i)
            {
              padding_size += i;
              /* Seek to the end of the archive */
              if(!m_seek(sizeof(idx_tail)+padding_size, SEEK_END))
                return -1;
              /* Read in the index tail */
              if(!m_read((char *)&idx_tail, sizeof(idx_tail)))
                return -1;
            }
            break;
          }
        }
        if(breakout)
          break;
      }
    }
    if(padding_size == LONG_LONG_MAX)
      return -3;

    /* Check the index tail for the string 'INDEX' */
    if(memcmp(idx_tail.magic, "INDEX", 5))
      return 1;

    /* Seek back to start of index */
    if(!m_seek(idx_tail.index_size+sizeof(idx_tail)+padding_size, SEEK_END))
      return -1;
    char buf[1024];
    off_t index_pos;
    for(index_pos = 0; index_pos < idx_tail.index_size; index_pos += 1024)
    {
      /* Read in index from the archive */
      if(!m_read(buf, MIN(idx_tail.index_size-index_pos, 1024)))
        return -1;
      idx_sum.AddBytes(buf, MIN(idx_tail.index_size-index_pos, 1024));
      /* Write out the index to the temporary file */
      if(fwrite(buf, MIN(idx_tail.index_size-index_pos, 1024), 1, tmp_index_file) != 1)
        return -3;
    }

    idx_sum.AddBytes((char *)&idx_tail, sizeof(idx_tail)-sizeof(idx_tail.sumbuf)-sizeof(idx_tail.magic));

    /* Finish off the checksum and verify it */
    idx_sum.Finish(sumbuf);
    if(memcmp(sumbuf, idx_tail.sumbuf, 20))
      return -2;
  }

  /* Seek to the index offset in the temporary index file */
  if(fseeko(tmp_index_file, (entry_number-1)*sizeof(off_t), SEEK_SET) == -1)
    return -3;
  /* Read in the entry number */
  if(fread((char *)offset, sizeof(off_t), 1, tmp_index_file) != 1)
    return -1;

  return 0;
}

