/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <zlib.h>
#include "FilterInBase.h"
#include "ZLibInFilter.h"

ZLibInFilter::ZLibInFilter()
{
	m_error = false;
	m_zstream.next_in = Z_NULL;
	m_zstream.avail_in = 0;
	m_zstream.zalloc = Z_NULL;
	m_zstream.zfree = Z_NULL;
	m_zstream.opaque = Z_NULL;
	if(inflateInit(&m_zstream) != Z_OK)
		m_error = true;
}

ZLibInFilter::~ZLibInFilter()
{
	if(!m_error)
		inflateEnd(&m_zstream);
}

/*
	Function:
		PutData
		
	Arguments:
		data - data to filter
		len - length of data
	
	Returns:
		2 - ZStream Error
		1 - Corrupt data
		0 - Success
    -1 - Buffer error
		
	Remarks:
		Decompresses the data and stores it inside of FilterInBase
*/
int ZLibInFilter::PutData(const char *data, size_t len)
{
	unsigned long stream_offset = m_zstream.total_in;
	int retval = 0;
	char buf[1024];
	m_zstream.next_in = (Bytef*)data;
	m_zstream.avail_in = len;
	m_zstream.next_out = (Bytef*)buf;
	m_zstream.avail_out = 1024;

	/* While there is still data to be decompressed */
	while(m_zstream.avail_in || (!m_zstream.avail_out))
	{
		/* Reset output buffer */
		m_zstream.next_out = (Bytef*)buf;
		m_zstream.avail_out = 1024;

		/* Decompress */
		int result = inflate(&m_zstream, Z_SYNC_FLUSH);

		/* Store Decompressed Data */
		if(1024-m_zstream.avail_out)
			if(!StoreData(buf, 1024-m_zstream.avail_out))
				return -1;

		switch(result)
		{
			case Z_STREAM_END:
			case Z_OK:
				break;
			case Z_DATA_ERROR:
				retval = 1;
				switch(inflateSync(&m_zstream))
				{
					case Z_STREAM_ERROR:
						return 2;
					case Z_OK:
						m_zstream.next_in = (Bytef*)(data+(m_zstream.total_in-stream_offset));
						m_zstream.avail_in = len-(m_zstream.total_in-stream_offset);
						break;
					default:
						return retval;
				}
				break;
      case Z_BUF_ERROR:
        /* Output buffer is empty, input buffer is empty. Safe to ignore this error. */
        return 0;
			default:
				return -1;
		}
	}
	
	return retval;
}

