/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*
  TODO: Need to handle errors better. For example don't just die upon the first non-fatal error encountered.
  NOTE: This program like all others are based off of the flatback source
*/
/*
	Flatcheck - Verifies flat backups.
  1st program based off of flatback source
*/

/*
	Includes and global configuration...
*/

#include <string>
#include <iostream>
#include <fstream>

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <limits.h>

#include <getopt.h>

#include "flatcheck.h"
#include "Dicer.h"
#include "Archiver.h"
#include "sha1sum.h"

#define HELPSTRING \
"FlatCheck version " FLATCHECK_VERSION "\n" \
"(C) 2005 Brian Gunlogson\n" \
"./flatcheck [-nszkh] -c slice_basename OR -d device_file\n" \
"	-c basename, --check=basename #Basename of slice\n" \
"	-d device, --device=device_file\n" \
"	-i, --no-index #Do not use index\n" \
"	-n, --no-verify #Do not verify against filesystem. Only check archive integrity.\n" \
"	-p, --no-prompt #Do not prompt for disks assume they are present.\n" \
"	-s size, --slice=size #Size of slice in bytes (required for multi-volume archives)\n" \
"	-k, --ask #Ask for conformation of command line options\n" \
"	-h, --help #Show this help message\n" \

#define FLATBACK_OPTSTRING "s:c:d:inpkh"

static const option long_opts[] = {	{ "size", 1, NULL, 's' },
																		{ "check", 1, NULL, 'c' },
																		{ "device", 1, NULL, 'd' },
																		{ "no-index", 0, NULL, 'i' },
																		{ "no-verify", 0, NULL, 'n' },
																		{ "no-prompt", 0, NULL, 'p' },
																		{ "ask", 0, NULL, 'k' },
																		{ "help", 0, NULL, 'h' },
																		{ NULL, 0, NULL, 0 } };

struct WCDataStruct
{
  Sha1Sum *checksum;
  FILE *filehandle;
  bool goodmatch;
  off_t bread;
};

Dicer *g_dicer = NULL;
Archiver *g_archiver = NULL;

/* Function Prototypes */
void DIE(const char *fmt, ...);
bool dicer_prompt_user(unsigned int slicenum);
void print_useage();
bool dicer_read_callback(char *buffer, unsigned int length);
bool dicer_seek(off_t offset, int method);
bool checksum_write_callback(const char *buffer, unsigned int length, void *ptr);

void DIE(const char *fmt, ...)
{
	va_list ap;
	
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	
	if(g_archiver)
		delete g_archiver;
	if(g_dicer)
		delete g_dicer;

	exit(1);
}

/*
	Function:
		dicer_prompt_user
		
	Arguments:
		message - Message
	
	Returns:
    key pressed as a response to the message
*/
int dicer_prompt_user(const char *message)
{
  fprintf(stderr, message);
  
  char keypress;
  /* FIXME: Replace with something that doesn't require the enter key to be pressed */
  if(fread(&keypress, 1, 1, stdin) != 1)
    DIE("Error in fread(stdin)");
  
  return keypress;
}

/*
	Function:
		print_usage
		
	Arguments:
		None
	
	Returns:
		Never, terminates.
		
	Remarks:
		Prints command line help message.
*/
void print_useage()
{
	DIE(HELPSTRING);
}

bool dicer_read_callback(char *buffer, unsigned int length)
{
  std::string strbuf;

  if(!g_dicer)
    DIE("g_dicer does not exist!\n");

  if(g_dicer->ReadData(strbuf, length))
    return false;

  if(strbuf.length() != length)
    return false;
  
  memcpy(buffer, strbuf.data(), length);
  
  return true;
}

bool dicer_seek(off_t offset, int method)
{
  if(!g_dicer)
    DIE("g_dicer does not exist!\n");

  return g_dicer->Seek(offset, method);
}

bool checksum_write_callback(const char *buffer, unsigned int length, void *ptr)
{
  WCDataStruct *callback_data = (WCDataStruct *)ptr;

  callback_data->checksum->AddBytes(buffer, length);

  if(!callback_data->filehandle)
  {
    callback_data->goodmatch = false;
  }
  else
  {
    char buf[1024];
    size_t c = 0;

    while(c < length)
    {
      size_t bread = fread(buf, 1, MIN(length-c, 1024), callback_data->filehandle);
      callback_data->bread += bread;
      if(bread != MIN(length-c, 1024))
      {
        callback_data->goodmatch = false;
        break;
      }
      if(callback_data->goodmatch && memcmp(buf, buffer+c, bread))
        callback_data->goodmatch = false;
      c += bread;
    }
  }

  return true;
}

/*
	Entry point
*/
int main(int argc, char **argv)
{
	g_dicer = new Dicer(dicer_prompt_user);
	if(!g_dicer)
		DIE("Failed to allocate Dicer object!");
	
	g_archiver = new Archiver(dicer_read_callback, dicer_seek);
	if(!g_archiver)
		DIE("Failed to allocate Archiver object!");

	bool moreopts = true, b_gave_output_base = false, b_gave_output_device = false, confirm_settings = false, quitflag = false, no_index = false, no_verify = false;

	while(moreopts)
	{
		unsigned int i;
		
		switch(getopt_long(argc, argv, FLATBACK_OPTSTRING, long_opts, NULL))
		{
			case 'h':
			case ':':
			case '?':
				print_useage();
			case 's':
				for(i = 0; i < strlen(optarg); i++)
					if(!isdigit(*(optarg+i)))
						DIE("Size parameter is not a base 10 positive integer!\n");

				if(!g_dicer->SetSize(atoll(optarg)))
					DIE("Dicer object rejected size argument\n");
				break;
			case 'd':
        if(b_gave_output_base)
          DIE("Already gave basename cannot give a device file\n");
				if(!g_dicer->SetArchivePath(optarg, true))
					DIE("Dicer object rejected device file argument\n");
				b_gave_output_device = true;
				break;
			case 'c':
        if(b_gave_output_device)
          DIE("Already gave device file cannot give a basename\n");
				if(!g_dicer->SetArchivePath(optarg, false))
					DIE("Dicer object rejected basename argument\n");
				b_gave_output_base = true;
				break;
			case 'k':
				confirm_settings = true;
				break;
      case 'i':
        no_index = true;
        break;
			case 'n':
				no_verify = true;
				break;
			case 'p':
        g_dicer->SetNoPrompt(true);
				break;
			case -1:
				moreopts = false;
				break;
			default:
				DIE("getopt_long returned something unexpected\n");
		}
	}
	
	if(!(b_gave_output_base || b_gave_output_device)) {
		fprintf(stderr, "Basename (-c option) required\n");
		print_useage();
	}
	
	if(argc-optind >= 1)
		DIE("No extra command line options allowed\n");
	
	if(confirm_settings)
	{
		bool ate_keypress = false;
		while(!ate_keypress)
		{
			fprintf(stderr, "\nConfirm Command Line Settings\n\n");
			fprintf(stderr, "=================\n\nArchiver Reports\n=================\n");
			g_archiver->PrintSettings();
			fprintf(stderr, "=================\n\nDicer Reports\n=================\n");
			g_dicer->PrintSettings();
			fprintf(stderr, "=================\n\n");

			fprintf(stderr, "Press 'c' to continue or 'q' to abort.\n");
			
			char keypress;
			/* FIXME: Replace with something that doesn't require the enter key to be pressed */
			if(fread(&keypress, 1, 1, stdin) != 1)
				DIE("Error in fread(stdin)");
			
			switch(keypress)
			{
				case 'q':
					quitflag = true;
				case 'c':
					ate_keypress = true;
					break;
				default:
					fprintf(stderr, "\nPressed key was not a vaild choice.\n");
					sleep(1);
			}
		}
	}
	
	if(!quitflag)
	{
    /* TODO: Maybe make the index optional? The Archiver class already supports it.
       Just change the first argument to OpenArchive to false */
		switch(g_archiver->OpenArchive(!no_index))
    {
      case 2:
  			DIE("ERR: OpenArchive: returned a general error code (probably memory allocation error)\n");
      case 1:
  			DIE("ERR: OpenArchive: reports a filesystem error\n");
      case -1:
  			DIE("ERR: OpenArchive: Unsupported archive version\n");
      case -2:
  			DIE("ERR: OpenArchive: Archive integrity check failed, archive header not found, or archive was written with a different endian CPU.\n");
        break;
      case 0:
        printf("MSG: OpenArchive: succeeded.\n");
        break;
      default:
  			DIE("ERR: OpenArchive: Unknown\n");
    }

    u_int64_t filenum;
    
		for(filenum = 1; filenum < ULONG_LONG_MAX; filenum++)
		{
      printf("FILE: %llu\n", filenum);

      off_t file_offset;
      switch(g_archiver->SeekFileNumber(filenum, false, &file_offset))
      {
        case 1:
          DIE("EOA: %llu\n", filenum);
          break;
        case 0:
          printf("MSG: SeekFileNumber: Seek successful.\n");
          break;
        case -1:
          DIE("ERR: SeekFileNumber: User canceled seek OR filesystem seek error.\n");
          break;
        case -2:
          DIE("ERR: SeekFileNumber: Seek could not continue because of archive corruption.\n");
          break;
        default:
          DIE("ERR: SeekFileNumber: Unknown\n");
          break;
      }

      std::string metadata;
      {
        bool breakout = false;
        switch(g_archiver->FileMetadataByNumber(filenum, &metadata))
        {
          case 1:
            printf("EOA: %llu\n", filenum);
            breakout = true;
            break;
          case 0:
            printf("MSG: FileMetadataByNumber: Getting checksum was successful.\n");
            break;
          case -1:
            DIE("BUG: FileMetadataByNumber: Entry filename too long\n");
            break;
          case -2:
            DIE("ERR: FileMetadataByNumber: Filesystem read error\n");
            break;
          case -10:
            DIE("BUG: FileMetadataByNumber: SeekFileNumber reported failure even though it just reported success\n");
            break;
          default:
            DIE("ERR: FileMetadataByNumber: Unknown\n");
            break;
        }
        if(breakout)
          break;
      }

      Sha1Sum the_checksum;
      the_checksum.AddBytes(metadata.data(), metadata.length());

      unsigned char meta_calcedsum[20], meta_storedsum[20];
      the_checksum.Finish(meta_calcedsum);

      switch(g_archiver->MetaChecksumByNumber(filenum, meta_storedsum))
      {
        case 0:
          printf("MSG: MetaChecksumByNumber: Getting checksum was successful.\n");
          break;
        case -1:
          DIE("BUG: MetaChecksumByNumber: SeekFileNumber reported failure even though it just reported success\n");
          break;
        case -2:
          DIE("ERR: MetaChecksumByNumber: Read error\n");
          break;
        default:
          DIE("ERR: MetaChecksumByNumber: Unknown\n");
          break;
      }

      unsigned int i;
      printf("DEBUG: calculated meta sha1sum ");
      for(i = 0; i < 20; i++)
      {
        printf("%02x", meta_calcedsum[i]);
      }
      printf("\n");
      printf("DEBUG: stored meta sha1sum ");
      for(i = 0; i < 20; i++)
      {
        printf("%02x", meta_storedsum[i]);
      }
      printf("\n");
      
      if(!memcmp(meta_calcedsum, meta_storedsum, 20))
      {
        printf("MSG: Good meta checksum\n");
      } else {
        DIE("ERR: Bad meta checksum!\n");
      }

      bool is_hardlink, is_compressed;
      mode_t file_mode;
      if(!g_archiver->HelperGetMode(metadata, &is_hardlink, &is_compressed, &file_mode))
      {
        printf("META: MODE: %04o\n", file_mode&0xfff);
        printf("META: TYPE: %s ", (is_compressed ? "Compressed" : "Not Compressed"));
        if(is_hardlink) {
          printf("Hardlink\n");
        } else if(S_ISREG(file_mode)) {
          printf("Regular File\n");
        } else if(S_ISDIR(file_mode)) {
          printf("Directory\n");
        } else if(S_ISCHR(file_mode)) {
          printf("Character Device\n");
        } else if(S_ISBLK(file_mode)) {
          printf("Block Device\n");
        } else if(S_ISFIFO(file_mode)) {
          printf("FIFO\n");
        } else if(S_ISLNK(file_mode)) {
          printf("Symbolic Link\n");
        } else if(S_ISSOCK(file_mode)) {
          printf("Socket\n");
        } else {
          printf("Other\n");
        }
      }
      else
      { 
        DIE("ERR: HelperGetMode failed\n");
      }

      std::string entry_file_name;
      if(!g_archiver->HelperGetFileName(metadata, &entry_file_name))
        printf("META: FILENAME: %s\n", entry_file_name.c_str());
      else
        DIE("ERR: HelperGetFileName failed\n");

      if(!no_index)
      {
        off_t entry_offset;
        switch(g_archiver->Index_GetEntry(filenum, &entry_offset))
        {
          case 1:
            DIE("ERR: Index_GetEntry: No Index found\n");
            break;
          case 0:
            printf("MSG: Index_GetEntry: succeeded\n");
            break;
          case -1:
            DIE("ERR: Index_GetEntry: Index read error\n");
            break;
          case -2:
            DIE("ERR: Index_GetEntry: Corrupt index\n");
            break;
          default:
            DIE("ERR: Index_GetEntry: Unknown error\n");
            break;
        }

        if(entry_offset != file_offset)
          DIE("ERR: Index check failed for file #%u\nentry_offset = %llu\nfile_offset = %llu\n", filenum, entry_offset, file_offset);
        else
          printf("MSG: Good index entry\n");
      }
      
      off_t file_size;
      if((!g_archiver->HelperGetFileSize(metadata, &file_size)) && (file_size > 0))
      {
        the_checksum.Reset();
  
        WCDataStruct callback_data;
        callback_data.checksum = &the_checksum;
        if(!no_verify)
        {
          callback_data.filehandle = fopen(entry_file_name.c_str(), "rb");
          if(!callback_data.filehandle)
          {
            DIE("ERR: Failed to open file on filesystem\n");
          }
        }
        else
        {
          callback_data.filehandle = NULL;
        }
        callback_data.goodmatch = true;
        callback_data.bread = 0;

        switch(g_archiver->FileDataByNumber(filenum, checksum_write_callback, &callback_data))
        {
          case 2:
            DIE("BUG: FileDataByNumber: EOA when EOA was already checked for\n");
          case 1:
            DIE("BUG: FileDataByNumber: Write callback returned false\n");
          case 0:
            printf("MSG: FileDataByNumber: Getting data was successful.\n");
            break;
          case -1:
            DIE("BUG: FileDataByNumber: FileMetadataByNumber reported failure even though it just reported success\n");
            break;
          case -2:
            DIE("ERR: FileDataByNumber: Read error\n");
            break;
          case -3:
            DIE("ERR: FileDataByNumber: Corrupt data, decompression error\n");
            break;
          case -4:
            DIE("ERR: FileDataByNumber: Zstream error\n");
            break;
          default:
            DIE("ERR: FileDataByNumber: Unknown\n");
            break;
        }

        if(callback_data.filehandle)
        {
          struct stat st;
          if(fstat(fileno(callback_data.filehandle), &st))
          {
            DIE("ERR: Failed to stat file on filesystem\n");
          }
          
          fclose(callback_data.filehandle);
          
          if(!(callback_data.goodmatch && (st.st_size == callback_data.bread)))
          {
            DIE("ERR: File on filesystem does not match with file in archive!\n");
          }
          else
          {
            printf("MATCH: File matches with file on filesystem!\n");
          }
        }

        unsigned char file_calcedsum[20], file_storedsum[20];
        the_checksum.Finish(file_calcedsum);

        switch(g_archiver->FileChecksumByNumber(filenum, file_storedsum))
        {
          case 0:
            printf("MSG: FileChecksumByNumber: Getting checksum was successful.\n");
            break;
          case -1:
            DIE("BUG: FileChecksumByNumber: SeekFileNumber reported failure even though it just reported success\n");
            break;
          case -2:
            DIE("ERR: FileChecksumByNumber: Read error\n");
            break;
          default:
            DIE("ERR: FileChecksumByNumber: Unknown\n");
            break;
        }

        unsigned int i;
        printf("DEBUG: calculated file sha1sum ");
        for(i = 0; i < 20; i++)
        {
          printf("%02x", file_calcedsum[i]);
        }
        printf("\n");
        printf("DEBUG: stored file sha1sum ");
        for(i = 0; i < 20; i++)
        {
          printf("%02x", file_storedsum[i]);
        }
        printf("\n");

        if(!memcmp(file_calcedsum, file_storedsum, 20))
        {
          printf("MSG: Good file checksum\n");
        } else {
          DIE("ERR: Bad file checksum!\n");
        }
      }

      printf("EFILE: %llu\n", filenum);
		}

		if(!g_archiver->CloseArchive())
			DIE("Failed to close archive!\n");

    printf("Success!\n");
	}

	delete g_archiver;
	delete g_dicer;

	return 0;
}

