/* sha1sum.cpp - Class to compute SHA1 message digest
	 according to the NIST specification FIPS-180-1.
	 
   Copyright (C) 2000, 2001, 2003, 2005 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.  */

/* Written by Scott G. Miller
   Credits:
      Robert Klep <robert@ilse.nl>  			-- Expansion function fix
			Brian Gunlogson <bmg300@yahoo.com>	-- Converted to a stripped down C++ class
*/

#include "sha1sum.h"

#include <sys/types.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define _STRING_ARCH_unaligned 1

#define rol(x,n) ( ((x) << (n)) | ((x) >> (32-(n))) )

/*
  Not-swap is a macro that does an endian swap on architectures that are
  big-endian, as SHA needs some data in a little-endian format
*/

#ifdef WORDS_BIGENDIAN
# define NOTSWAP(n) (n)
# define SWAP(n)							\
    (((n) << 24) | (((n) & 0xff00) << 8) | (((n) >> 8) & 0xff00) | ((n) >> 24))
#else
# define NOTSWAP(n)                                                         \
    (((n) << 24) | (((n) & 0xff00) << 8) | (((n) >> 8) & 0xff00) | ((n) >> 24))
# define SWAP(n) (n)
#endif

#define BLOCKSIZE 4096
/* Ensure that BLOCKSIZE is a multiple of 64.  */
#if BLOCKSIZE % 64 != 0
#error "invalid BLOCKSIZE"
#endif

/* SHA1 round constants */
#define K1 0x5a827999L
#define K2 0x6ed9eba1L
#define K3 0x8f1bbcdcL
#define K4 0xca62c1d6L

/* Round functions.  Note that F2 is the same as F4.  */
#define F1(B,C,D) ( D ^ ( B & ( C ^ D ) ) )
#define F2(B,C,D) (B ^ C ^ D)
#define F3(B,C,D) ( ( B & C ) | ( D & ( B | C ) ) )
#define F4(B,C,D) (B ^ C ^ D)

/* This array contains the bytes used to pad the buffer to the next
 64-byte boundary.  (RFC 1321, 3.1: Step 1)  */
const unsigned char Sha1Sum::m_fillbuf[] = { 0x80, 0 /* , 0, 0, ...  */ };

Sha1Sum::Sha1Sum()
{
	Reset();
}

Sha1Sum::~Sha1Sum()
{
}

void Sha1Sum::Reset()
{
	//Initialize sha context
	m_ctx.A = 0x67452301;
  m_ctx.B = 0xefcdab89;
  m_ctx.C = 0x98badcfe;
  m_ctx.D = 0x10325476;
  m_ctx.E = 0xc3d2e1f0;

  m_ctx.total[0] = m_ctx.total[1] = 0;
  m_ctx.buflen = 0;
}

/* Process LEN bytes of BUFFER, accumulating context into CTX.
	 It is assumed that LEN % 64 == 0.
	 Most of this code comes from GnuPG's cipher/sha1.c.  */
bool Sha1Sum::AddBlock(const void *buffer, size_t len)
{
	if(len % 64 != 0) {
		fprintf(stderr, "sha1sum.cpp: Length is not a multiple of 64\n");
		return false;
	}
	
  const u_int32_t *words = (const u_int32_t *)buffer;
  size_t nwords = len / sizeof (u_int32_t);
  const u_int32_t *endp = words + nwords;
  u_int32_t x[16];
  u_int32_t a = m_ctx.A;
  u_int32_t b = m_ctx.B;
  u_int32_t c = m_ctx.C;
  u_int32_t d = m_ctx.D;
  u_int32_t e = m_ctx.E;

  /* First increment the byte count.  RFC 1321 specifies the possible
     length of the file up to 2^64 bits.  Here we only compute the
     number of bytes.  Do a double word increment.  */
  m_ctx.total[0] += len;
  if (m_ctx.total[0] < len)
    ++m_ctx.total[1];

#define M(I) ( tm =   x[I&0x0f] ^ x[(I-14)&0x0f] \
		^ x[(I-8)&0x0f] ^ x[(I-3)&0x0f] \
		, (x[I&0x0f] = rol(tm, 1)) )

#define R(A,B,C,D,E,F,K,M)  do { E += rol( A, 5 )     \
		+ F( B, C, D )  \
		+ K	      \
		+ M;	      \
		B = rol( B, 30 );    \
		} while(0)

	while (words < endp)
	{
		u_int32_t tm;
		int t;
		/* FIXME: see sha1.c for a better implementation.  */
		for (t = 0; t < 16; t++)
		{
			x[t] = NOTSWAP (*words);
			words++;
		}
		
		R( a, b, c, d, e, F1, K1, x[ 0] );
		R( e, a, b, c, d, F1, K1, x[ 1] );
		R( d, e, a, b, c, F1, K1, x[ 2] );
		R( c, d, e, a, b, F1, K1, x[ 3] );
		R( b, c, d, e, a, F1, K1, x[ 4] );
		R( a, b, c, d, e, F1, K1, x[ 5] );
		R( e, a, b, c, d, F1, K1, x[ 6] );
		R( d, e, a, b, c, F1, K1, x[ 7] );
		R( c, d, e, a, b, F1, K1, x[ 8] );
		R( b, c, d, e, a, F1, K1, x[ 9] );
		R( a, b, c, d, e, F1, K1, x[10] );
		R( e, a, b, c, d, F1, K1, x[11] );
		R( d, e, a, b, c, F1, K1, x[12] );
		R( c, d, e, a, b, F1, K1, x[13] );
		R( b, c, d, e, a, F1, K1, x[14] );
		R( a, b, c, d, e, F1, K1, x[15] );
		R( e, a, b, c, d, F1, K1, M(16) );
		R( d, e, a, b, c, F1, K1, M(17) );
		R( c, d, e, a, b, F1, K1, M(18) );
		R( b, c, d, e, a, F1, K1, M(19) );
		R( a, b, c, d, e, F2, K2, M(20) );
		R( e, a, b, c, d, F2, K2, M(21) );
		R( d, e, a, b, c, F2, K2, M(22) );
		R( c, d, e, a, b, F2, K2, M(23) );
		R( b, c, d, e, a, F2, K2, M(24) );
		R( a, b, c, d, e, F2, K2, M(25) );
		R( e, a, b, c, d, F2, K2, M(26) );
		R( d, e, a, b, c, F2, K2, M(27) );
		R( c, d, e, a, b, F2, K2, M(28) );
		R( b, c, d, e, a, F2, K2, M(29) );
		R( a, b, c, d, e, F2, K2, M(30) );
		R( e, a, b, c, d, F2, K2, M(31) );
		R( d, e, a, b, c, F2, K2, M(32) );
		R( c, d, e, a, b, F2, K2, M(33) );
		R( b, c, d, e, a, F2, K2, M(34) );
		R( a, b, c, d, e, F2, K2, M(35) );
		R( e, a, b, c, d, F2, K2, M(36) );
		R( d, e, a, b, c, F2, K2, M(37) );
		R( c, d, e, a, b, F2, K2, M(38) );
		R( b, c, d, e, a, F2, K2, M(39) );
		R( a, b, c, d, e, F3, K3, M(40) );
		R( e, a, b, c, d, F3, K3, M(41) );
		R( d, e, a, b, c, F3, K3, M(42) );
		R( c, d, e, a, b, F3, K3, M(43) );
		R( b, c, d, e, a, F3, K3, M(44) );
		R( a, b, c, d, e, F3, K3, M(45) );
		R( e, a, b, c, d, F3, K3, M(46) );
		R( d, e, a, b, c, F3, K3, M(47) );
		R( c, d, e, a, b, F3, K3, M(48) );
		R( b, c, d, e, a, F3, K3, M(49) );
		R( a, b, c, d, e, F3, K3, M(50) );
		R( e, a, b, c, d, F3, K3, M(51) );
		R( d, e, a, b, c, F3, K3, M(52) );
		R( c, d, e, a, b, F3, K3, M(53) );
		R( b, c, d, e, a, F3, K3, M(54) );
		R( a, b, c, d, e, F3, K3, M(55) );
		R( e, a, b, c, d, F3, K3, M(56) );
		R( d, e, a, b, c, F3, K3, M(57) );
		R( c, d, e, a, b, F3, K3, M(58) );
		R( b, c, d, e, a, F3, K3, M(59) );
		R( a, b, c, d, e, F4, K4, M(60) );
		R( e, a, b, c, d, F4, K4, M(61) );
		R( d, e, a, b, c, F4, K4, M(62) );
		R( c, d, e, a, b, F4, K4, M(63) );
		R( b, c, d, e, a, F4, K4, M(64) );
		R( a, b, c, d, e, F4, K4, M(65) );
		R( e, a, b, c, d, F4, K4, M(66) );
		R( d, e, a, b, c, F4, K4, M(67) );
		R( c, d, e, a, b, F4, K4, M(68) );
		R( b, c, d, e, a, F4, K4, M(69) );
		R( a, b, c, d, e, F4, K4, M(70) );
		R( e, a, b, c, d, F4, K4, M(71) );
		R( d, e, a, b, c, F4, K4, M(72) );
		R( c, d, e, a, b, F4, K4, M(73) );
		R( b, c, d, e, a, F4, K4, M(74) );
		R( a, b, c, d, e, F4, K4, M(75) );
		R( e, a, b, c, d, F4, K4, M(76) );
		R( d, e, a, b, c, F4, K4, M(77) );
		R( c, d, e, a, b, F4, K4, M(78) );
		R( b, c, d, e, a, F4, K4, M(79) );
		
		a = m_ctx.A += a;
		b = m_ctx.B += b;
		c = m_ctx.C += c;
		d = m_ctx.D += d;
		e = m_ctx.E += e;
	}
	
	return true;
}

void Sha1Sum::AddBytes(const void *buffer, size_t len)
{
	/* When we already have some bits in our internal buffer concatenate
		both inputs first.  */
	if (m_ctx.buflen != 0)
	{
		size_t left_over = m_ctx.buflen;
		size_t add = 128 - left_over > len ? len : 128 - left_over;
		
		memcpy(&m_ctx.buffer[left_over], buffer, add);
		m_ctx.buflen += add;
		
		if (m_ctx.buflen > 64)
		{
			AddBlock(m_ctx.buffer, m_ctx.buflen & ~63);
			
			m_ctx.buflen &= 63;
			/* The regions in the following copy operation cannot overlap.  */
			memcpy(m_ctx.buffer, &m_ctx.buffer[(left_over + add) & ~63], m_ctx.buflen);
		}
		
		buffer = (const char *) buffer + add;
		len -= add;
	}
	
	/* Process available complete blocks.  */
	if (len >= 64)
	{
#if !_STRING_ARCH_unaligned
/* To check alignment gcc has an appropriate operator.  Other
compilers don't.  */
# if __GNUC__ >= 2
#  define UNALIGNED_P(p) (((md5_uintptr) p) % __alignof__ (md5_uint32) != 0)
# else
#  define UNALIGNED_P(p) (((md5_uintptr) p) % sizeof (md5_uint32) != 0)
# endif
		if (UNALIGNED_P(buffer))
			while (len > 64)
			{
				AddBlock(memcpy(m_ctx.buffer, buffer, 64), 64);
				buffer = (const char *) buffer + 64;
				len -= 64;
			}
		else
#endif
		{
			AddBlock(buffer, len & ~63);
			buffer = (const char *) buffer + (len & ~63);
			len &= 63;
		}
	}
	
	/* Move remaining bytes in internal buffer.  */
	if (len > 0)
	{
		size_t left_over = m_ctx.buflen;
		
		memcpy(&m_ctx.buffer[left_over], buffer, len);
		left_over += len;
		if (left_over >= 64)
		{
			AddBlock(m_ctx.buffer, 64);
			left_over -= 64;
			memcpy(m_ctx.buffer, &m_ctx.buffer[64], left_over);
		}
		m_ctx.buflen = left_over;
	}
}

/* Process the remaining bytes in the internal buffer and the usual
   prolog according to the standard and write the result to RESBUF.

   IMPORTANT: On some systems it is required that RESBUF is correctly
   aligned for a 32 bits value.  */
void * Sha1Sum::Finish(void *resbuf)
{
  /* Take yet unprocessed bytes into account.  */
  u_int32_t bytes = m_ctx.buflen;
  size_t pad;

  /* Now count remaining bytes.  */
  m_ctx.total[0] += bytes;
  if (m_ctx.total[0] < bytes)
    ++m_ctx.total[1];

  pad = bytes >= 56 ? 64 + 56 - bytes : 56 - bytes;
  memcpy(&m_ctx.buffer[bytes], m_fillbuf, pad);

  /* Put the 64-bit file length in *bits* at the end of the buffer.  */
  *(u_int32_t *) &m_ctx.buffer[bytes + pad + 4] = NOTSWAP(m_ctx.total[0] << 3);
  *(u_int32_t *) &m_ctx.buffer[bytes + pad] = NOTSWAP((m_ctx.total[1] << 3) |
						    (m_ctx.total[0] >> 29));

  /* Process last bytes.  */
  AddBlock(m_ctx.buffer, bytes + pad + 8);

  return ReadCtx(resbuf);
}


/* Put result from CTX in first 20 bytes following RESBUF.  The result
   must be in little endian byte order.

   IMPORTANT: On some systems it is required that RESBUF is correctly
   aligned for a 32 bits value.  */
void * Sha1Sum::ReadCtx(void *resbuf)
{
  ((u_int32_t *) resbuf)[0] = NOTSWAP (m_ctx.A);
  ((u_int32_t *) resbuf)[1] = NOTSWAP (m_ctx.B);
  ((u_int32_t *) resbuf)[2] = NOTSWAP (m_ctx.C);
  ((u_int32_t *) resbuf)[3] = NOTSWAP (m_ctx.D);
  ((u_int32_t *) resbuf)[4] = NOTSWAP (m_ctx.E);

  return resbuf;
}

