/*
 *  field_types.h - The different types of data fields that can be
 *                  displayed, and their structs.
 *                  This file is part of the FreeLCD package.
 *
 *  $Id: field_types.h,v 1.4 2004/06/20 12:36:56 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file field_types.h
 * Different kinds of data fields that can be defined in a screen, and
 * their internal representation. */

#ifndef _FIELD_TYPES_H
#define _FIELD_TYPES_H

#include "../renderers/grcanvas.h" 

typedef enum
{
  F_NONE = 0,
  F_LABEL,
  F_TEXT,
  F_HEADER,
  F_TIME,
  F_TIMESPAN,
  F_DATE,
  F_SCALAR,
  F_PERCENTAGE,
  F_HISTOGRAM,
  F_PIXMAP,
  F_HSCROLL,
  F_VSCROLL
}
field_type;

/** The time of day, in seconds from 00:00 UTC, optionally with millisecond
 * precision. */
typedef struct
{
  unsigned int    seconds;       /**< Seconds since midnight, 0..86399 */
  unsigned short  milliseconds;  /**< Milliseconds, 0..999 */
  char            ms_defined;    /**< Non-zero if milliseconds are defined */
}
flcd_time;

/** A timespan of up to 136 years, optionally with millisecond resolution. */
typedef struct
{
  unsigned long   seconds;
  unsigned short  milliseconds;
  char            ms_defined;    /**< Non-zero if milliseconds are defined */
}
flcd_timespan;

/** A Gregorian date. */
typedef struct
{
  unsigned char day;
  unsigned char month;
  int           year;
}
flcd_date;

/** An array of values, that will be displayed as a histogram or a graph. */
typedef struct
{
  unsigned int samples;  /**< The number of entries in \a data */
  int          min;      /**< Suggested minimum range */
  int          max;      /**< Suggested maxumum range */
  int          *data;    /**< An array with the actual data */
}
flcd_histogram;

typedef struct
{
  unsigned int size;
  unsigned int stepsize;
}
flcd_scroll;

/** One data field in a screen. */
typedef struct
{
  /** The type of information shown by this field. */
  field_type          type;
  /** A flag, indicating if the client has already supplied some data for
   * this field. */
  char                defined;
  /** A flag, indicating if the provided information is valid. */
  char                valid;
  /** A flag, indicating if this field needs to be redrawn on the canvas. */
  char                changed;

  /** The actual contents of this data field. The \a type variable determines
   * which member of this union should be used. */
  union
    {
      long            scalar;
      double          percentage;
      char            *text;
      flcd_time       time;
      flcd_timespan   timespan;
      flcd_date       date;
      flcd_histogram  histogram;
      flcd_scroll     scroll;
      gr_canvas       pixmap;
    }
  data;
}
field;

#endif /* Multiple include guard */
