/*
 *  xmlt.h : XML data structure.
 *           This file is part of the FreeLCD package.
 *
 *  $Id: xmlt.h,v 1.8 2004/06/20 12:38:35 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file xmlt.h
 * Functions and data structures for parsing and handling XML data.
 */

#ifndef _XMLT_H
#define _XMLT_H 1

#include <glib.h>
#include "dictionary.h"

/** Tags that were not recognized get this value. This value could be
 * replaced later on by xmlt_rescan_document(). */
#define XML_UNKNOWN_TAG -1

/** Attributes that were not recognized get this value. This value could
 * be replaced later on by xmlt_rescan_document(). */
#define XML_UNKNOWN_ATTRIBUTE -1

/** Every tag has a list of attributes, which are key-value pairs as defined
 * in this struct.
 * Whenever an attribute is found inside a tag (e.g. <foo key="value">), the
 * string "key" is looked up in a dictionary of recognized attributes, and
 * is stored here together with "value". */
typedef struct
{
  int attribute; /**< Attribute number, could be #XML_UNKNOWN_ATTRIBUTE. */
  char *value; /**< Value of the attribute. */
  char *unknown_attrib; /**< Unknown attributes are stored here literally. */
}
xml_attribute;

/** Type of XML node, used in the #xml_node struct. */
typedef enum
{
  NODE, CDATA
}
xml_node_t;

/** This struct represents a tag or character data in the XML document.
 * The exact behavior depends on the \a type field. If this is a NODE,
 * it represents a tag. A tag that can be translated with a dictionary
 * is stored by its respective integer value. If the tag was not found,
 * \a type is #XML_UNKNOWN_TAG, and the literal tag is stored in
 * \a cdata. If \a type is CDATA, all fields except for \a cdata and
 * \a parent lose their meaning.
 */
typedef struct _xml_node
{
  xml_node_t type; /**< NODE or CDATA */
  int tag; /**< Translated value of an XML tag. */
  /** Pointer to the parent xml_node, or NULL if this is the root tag. */
  struct _xml_node *parent;
  GSList *attributes; /**< Tag attributes as a list of #xml_attribute. */
  GSList *children; /**< Children of this tag. */
  char *cdata; /**< Either stores an unidentified tag, or CDATA. */
}
xml_node;

/** Create a context for parsing an XML stream.  
 * Whenever a complete document was received, \a callback is invoked with
 * the provided \a cb_data, and the root tag of the document. The
 * dictionaries for tags and attributes are used for looking up tag names
 * and attribute keys. (These are stored as an integer, not as the complete
 * string, for speed and memory efficiency.)
 * \param callback This function will be called when a complete document
 * was parsed. A document is considered to be complete whenever the root tag
 * is closed.
 * \param cb_data An extra parameter that will be passed to \a callback as
 * well.
 * \param tags A dictionary with all recognized tags and their respective
 * index values.
 * \param attribs A dictionary with all recognized attributes and their
 * respective index values.  */
void *
xmlt_create_context (void (*callback) (void *, xml_node *),
                     void *cb_data,
                     dictionary *tags, dictionary *attribs);

/** Reset a parser context to the initial state.
 *  This function is usually called when garbage was detected in the stream.
 *  \param context Parser context to reset.
 *  \return 1 if the context was succesfully reset, 0 if not. */
int
xmlt_reset_context (void *context);

/** Parse a chunk of XML data
 * \param context Pointer to a parser context, as returned by 
 * xmlt_create_context() .
 * \param data Pointer to some XML data.
 * \param len Size of the XML data.
 * \return 1 if the data could be successfully parsed, 0 if not. */
int 
xmlt_parse (void *context, const char *data, size_t len);

/** Free all memory associated with a particular context.
 * \param context Pointer to a parser context, as returned by
 * xmlt_create_context() . */
void 
xmlt_free_context (void *context);

/** Free all memory associated with a particular document.
 * \param doc Pointer to a document. */
void 
xmlt_free_document (xml_node *doc);

/** Try to identify the unknown tags and attributes using a different
 * set of dictionaries.
 * \param doc Document to scan.
 * \param tags Dictionary to use for tags.
 * \param attribs Dictionary to use for attribs. */
void
xmlt_rescan_document (xml_node *doc, dictionary *tags, dictionary *attribs);

/** Get the next node in the document, depth first.
 * \param iter Pointer to the current node. 
 * \param doc Document to traverse.
 * \return Pointer to the next node, or NULL if \a iter was the last node. */
xml_node *
xmlt_get_next (xml_node *iter, xml_node *doc);

/** Get the next node in the document, without descending.
 * \param iter Pointer to the current node. 
 * \return Pointer to the next node, or NULL if \a iter was the last node
 * in this level. */
xml_node *
xmlt_get_next_shallow (xml_node *iter);

/** Find the first occurrence of a certain tag, starting right after the
 * given iterator, or at the first child of the document node if this
 * iterator is NULL. Searching is not done in depth.
 * \param doc The searched document.
 * \param iter Document iterator that marks the starting point of the
 * search. If NULL, the document is searched from the start.
 * \param tag The tag to look for.
 * \return Pointer to the first xml node that matches TAG, or NULL if no
 * such node was found. */
xml_node *
xmlt_find (xml_node *doc, xml_node *iter, int tag);

/** Same as xmlt_find(), but the search criterium is determined by a 
 * custom function.
 * \param doc The searched document.
 * \param iter Document iterator that marks the starting point of the
 * search. If NULL, the document is searched from the start.
 * \param compare Pointer to a function that determines if an xml node
 * satisfies the search condition. This function should return a non-zero
 * value if a match is found.
 * \return Pointer to the first xml node that matches TAG, or NULL if no
 * such node was found. */
xml_node *
xmlt_find_if (xml_node *doc, xml_node *iter, int(*compare)(xml_node*));

/** Get the contents of a node's attribute. 
 * \param node The node to get an attribute from.
 * \param attribute The attribute to look for.
 * \return The contents of the attribute, or NULL if \a node does not have
 * an attribute with the given key.
 */
const char *
xmlt_get_attrib (xml_node *node, int attribute);

/** Get the first character data beneath a given node.
 * \param node The node to get the character data from.
 * \return The first character data beneath \a node, or NULL if nothing
 * was found.
 */
const char *
xmlt_get_first_cdata (xml_node *node);

/** For every node in the document, this routine calls a function with some
 * user-supplied data and the node as its parameters.
 * \param doc For every node in this document, \a func will be called.
 * \param user_data Opaque pointer to some data, that will also be passed
 * to \a func.
 * \param func The function that will be called.
 */
void
xmlt_for_each (xml_node *doc, void *user_data,
               void (*func) (void *, xml_node *));

#endif /* Multiple include guard */
