/*
 *  charcanvas.c - Canvas for character-oriented devices.
 *                 This file is part of the FreeLCD package.
 *  
 *  $Id: charcanvas.c,v 1.6 2004/06/20 14:36:48 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(FOO)
#endif

#include <stdlib.h>
#include <glib.h>

#include "charcanvas.h"
#include "common/debug.h"

/*---------------------------------------------------- cc_create_canvas --*/
cc_canvas *
cc_create_canvas (unsigned int width, unsigned int height)
{
  size_t i;
  size_t size = width * height;
  cc_elem *element;
  cc_canvas *new = g_new (cc_canvas, 1);
   
  new->width = width;
  new->height = height;
  new->elements = g_new (cc_elem, size);
  element = new->elements;
  
  for (i = 0; i < size; ++i, ++element)
    {
      element->c = L' ';
      element->damaged_flag = 0;
      element->bitmap = 0;
    }
  
  return new;
}

/*---------------------------------------------------- cc_delete_canvas --*/
void
cc_delete_canvas (cc_canvas *canvas)
{
  int i;
  int size;
  cc_elem *element;
  
  if (canvas == NULL)
    return;

  size = canvas->width * canvas->height;
  element = canvas->elements;

  for (i = 0; i < size; ++i, ++element)
    free (element->bitmap);

  free (canvas->elements);
  free (canvas);
}

/*--------------------------------------------------------- cc_put_char --*/
void
cc_put_char (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos, 
             wchar_t character)
{
  cc_elem *element = cc_get_element (canvas, x_pos, y_pos);

  if (element != NULL && element->c != character)
    {
      element->c = character;
      element->damaged_flag = 1;
    }
}

/*------------------------------------------------------- cc_write_text --*/
void
cc_write_text (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos,
               char *utf8)
{
  wchar_t curr;
  unsigned char u, u2, u3, u4, u5, u6;
  
  while ((u = *utf8) != '\0')
    {
      curr = 0;

      if (u < 0x80)
        {
          /* Range 0 - 7F */
          curr = u;
        }
      else if (u <= 0xc0 || ((u2 = *(utf8 + 1)) & 0xc0) != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xe0) == 0xc0)
        {
          /* Range 80 - 7FF */
          curr = u & 0x1f;
          curr <<= 6;
          curr |= u2 & 0x3f;
          ++utf8;
        }
      else if (u <= 0xe0 || ((u3 = (*utf8 + 2)) & 0xc0) != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xf0) == 0xe0)
        {
          /* Range 800 - FFFF */
          curr = u & 0x0f;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          utf8 += 2;
        }
      else if (u <= 0xf0 || ((u4 = (*utf8 + 3)) & 0xc0) != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xf8) == 0xf0)
        {
          /* Range 10000 - 1FFFFF */
          curr = u & 0x07;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          curr <<= 6;
          curr |= u4 & 0x3f;
          utf8 += 3;
        }
      else if (u <= 0xf8 || ((u5 = (*utf8 + 4)) & 0xc0) != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xfc) == 0xf8)
        {
          /* Range 200000 - 3FFFFFF */
          curr = u & 0x03;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          curr <<= 6;
          curr |= u4 & 0x3f;
          curr <<= 6;
          curr |= u5 & 0x3f;
          utf8 += 4;
        }
      else if (u <= 0xfc || ((u6 = (*utf8 + 4)) & 0xc0) != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xfe) == 0xfc)
        {
          /* Range 4000000 - 7FFFFFFF */
          curr = u & 0x01;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          curr <<= 6;
          curr |= u4 & 0x3f;
          curr <<= 6;
          curr |= u5 & 0x3f;
          curr <<= 6;
          curr |= u6 & 0x3f;
          utf8 += 5;
        }
      else
        {
          /* Invalid character */
          return;
        }

      cc_put_char (canvas, x_pos++, y_pos, curr);
      ++utf8;
    }
}

/*------------------------------------------------------ cc_get_element --*/
cc_elem*
cc_get_element (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos)
{
  if (x_pos >= canvas->width || y_pos >= canvas->height)
    return NULL;

  return canvas->elements + x_pos + (y_pos * canvas->width);
}

/*-------------------------------------------- cc_clear_damaged_regions --*/
void
cc_clear_damaged_regions (cc_canvas *canvas)
{
  size_t i;
  size_t size = canvas->width * canvas->height;
  cc_elem *element = canvas->elements;
  
  for (i = 0; i < size; ++i, ++element)
    element->damaged_flag = 0;
}

/*----------------------------------------------------- cc_clear_canvas --*/
void
cc_clear_canvas (cc_canvas *dest)
{
  size_t i;
  size_t size = dest->width * dest->height;
  cc_elem *element = dest->elements;
  
  for (i = 0; i < size; ++i, ++element)
    {
      if (element->c != ' ')
        {
          element->c = ' ';
          element->damaged_flag = 1;
        }
    }
}

/*------------------------------------------------------ cc_copy_region --*/
void
cc_copy_region (cc_canvas *src, unsigned int sx, unsigned int sy,
                unsigned int width, unsigned int height, 
                cc_canvas *dest, unsigned int dx, unsigned int dy)
{
  unsigned int w, h;
  unsigned int x, y;
  cc_elem *src_elem;

  assert (src != NULL);
  assert (dest != NULL);

  /* Do not copy if it is not going to be visible anyway. */
  if (dx >= dest->width || dy >= dest->height) 
    return;

  /* Determine the actual number of characters that are going to be
   * copied, the width and height of this rectangle are stored in w and h. */
  if (dx + width  > dest->width ) w = dest->width  - dx; else w = width;
  if (dy + height > dest->height) h = dest->height - dy; else h = height;

  for (y = 0; y < h; ++y)
    {
      for (x = 0; x < w; ++x)
        {
          src_elem = cc_get_element (src, sx + x, sy + y);

          cc_put_char (dest, dx + x, dy + y, 
                       src_elem == NULL ? ' ' : src_elem->c);
        }
    }
}


#ifdef UNIT_TEST_CHARCANVAS_C

/* UNIT_CFLAGS -I/usr/include/glib-2.0 -I/usr/lib/glib-2.0/include -lglib-2.0 */

int main (int argc, char **argv)
{
  cc_canvas *src  = cc_create_canvas (10, 2);
  cc_canvas *dest = cc_create_canvas (20, 6);
  
  cc_put_char (src, 0, 0, 'a');
  cc_put_char (src, 0, 1, 'b');
  cc_put_char (src, 9, 0, 'c');
  cc_put_char (src, 9, 1, 'd');
  
  /* Copy one line to dest. */
  cc_copy_region (src, 0, 0, 10, 1, dest, 0, 1);
  if (cc_get_element (dest, 0, 1)->c != 'a')
    {
      printf ("(1) Missing 'a' in position (0,1)\n");
      return 1;
    }

  if (cc_get_element (dest, 9, 1)->c != 'c')
    {
      printf ("(2) Missing 'c' in position (9,1)\n");
      return 1;
    }

  if (cc_get_element (dest, 0, 2)->c != ' ')
    {
      printf ("(3) Position (0, 2) is not empty\n");
      return 1;
    }

  /* Copy another line to dest. */
  cc_copy_region (src, 0, 1, 10, 1, dest, 0, 1);
  if (cc_get_element (dest, 0, 1)->c != 'b')
    {
      printf ("(4) Missing 'b' in position (0,1)\n");
      return 1;
    }

  if (cc_get_element (dest, 0, 2)->c != ' ')
    {
      printf ("(5) Position (0, 2) is not empty\n");
      return 1;
    }

  /* Copy both lines to dest. */
  cc_copy_region (src, 0, 0, 10, 2, dest, 0, 1);
  if (cc_get_element (dest, 0, 1)->c != 'a')
    {
      printf ("(6) Missing 'a' in position (0,1)\n");
      return 1;
    }

  if (cc_get_element (dest, 0, 2)->c != 'b')
    {
      printf ("(7) Missing 'b' in position (0,2)\n");
      return 1;
    }

  return 0;
}

#endif

