/*
 *  charcanvas.h - Canvas for character-oriented devices.
 *                 This file is part of the FreeLCD package.
 *
 *  $Id: charcanvas.h,v 1.5 2004/06/20 14:36:48 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file charcanvas.h
 * The canvas is an abstraction between the layout and data formatting
 * algorithms on one side, and the device drivers on the other. All drivers
 * take a canvas as their input, and try to display the bitmap data and
 * unicode text as good as possible within the constraints of the hardware. */

#ifndef _FLCD_CHARCANVAS_H
#define _FLCD_CHARCANVAS_H 1


#if HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_WCHAR_H
# include <wchar.h>
#else
# define wchar_t unsigned int
#endif

/** One cc_elem represents one character element on the device. */
typedef struct
{
  wchar_t c; /**< The character at this position. */
  char damaged_flag; /**< Damaged elements need to be redrawn on the device. */
  unsigned char *bitmap; /**< If not NULL, this element is part of a bitmap. */
}
cc_elem;

/** The actual canvas. */
typedef struct
{
  unsigned int width; /**< Width in characters. */
  unsigned int height; /**< Height in characters. */
  cc_elem *elements; /**< Array of (\a width * \a height) elements. */
}
cc_canvas;


/** Return a new canvas with the given dimensions.
 * All elements are set to a space character initially.
 * \param width Width in characters.
 * \param height Height in characters.
 * \return A pointer to the newly created canvas. */
cc_canvas *
cc_create_canvas (unsigned int width, unsigned int height);

/** Free all memory of a canvas.
 * \param canvas Pointer to the canvas to clean up. */
void
cc_delete_canvas (cc_canvas *canvas);

/** Put a UTF-8 character at the given position.
 * \param canvas Pointer to the canvas to draw on.
 * \param x_pos X position, must be smaller than \a canvas->width.
 * \param y_pos Y position, must be smaller than \a canvas->height.
 * \param character Unicode character. */
void
cc_put_char (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos, 
             wchar_t character);

/** Write a UTF-8 string at the given position.
 * \param canvas Pointer to the canvas to draw on.
 * \param x_pos X position, must be smaller than (\a canvas->width - length
 * of \a utf8_string).
 * \param y_pos Y position, must be smaller than \a canvas->height.
 * \param utf8_string Pointer to a UTF-8 encoded string. */
void
cc_write_text (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos,
               char *utf8_string);

/** Return the element at the given position.
 * \param canvas Pointer to the canvas to query.
 * \param x_pos X position, must be smaller than \a canvas->width.
 * \param y_pos Y position, must be smaller than \a canvas->height.
 * \return The pointer to the element at the position (x_pos, y_pos). */
cc_elem*
cc_get_element (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos);

/** Reset all damaged region flags (normally done after the canvas was
 * displayed on a device.
 * \param canvas Pointer to the canvas to reset. */
void
cc_clear_damaged_regions (cc_canvas *canvas);

/** Clear a canvas completely. */
void
cc_clear_canvas (cc_canvas *canvas);

/** Copy a region from one canvas to the other.
 * \param src Source canvas.
 * \param sx Source X position.
 * \param sy Source Y position.
 * \param width Width of the region.
 * \param height Height of the region.
 * \param dest Destination canvas.
 * \param dx Destination X position.
 * \param dy Destination Y position.
 */
void
cc_copy_region (cc_canvas *src, unsigned int sx, unsigned int sy,
                unsigned int width, unsigned int height, 
                cc_canvas *dest, unsigned int dx, unsigned int dy);

#endif /* Multiple include guard */
