/*
 *  charlayout.c - Character-based layout algorithm.
 *                 This file is part of the FreeLCD package.
 *  
 *  $Id: charlayout.c,v 1.7 2004/06/20 14:39:57 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file charlayout.c 
 * Layout algorithm, optimized for character displays.
 */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_STRING_H
# include <string.h>
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(FOO)
#endif

#include <glib.h>

#include "common/field_types.h"
#include "common/debug.h"
#include "common/layout_hint.h"
#include "server/guile.h"
#include "charrenderer.h"
#include "layout_tags.h"
#include "charlayout.h"

/* The dictionary for field types is defined in common/field_types.c */
extern dictionary field_type_dict;

/*------------------------------------------------------------------------
 * Layout data structure type definitions
 *------------------------------------------------------------------------
 */

/** Horizontal widget alignment. */
typedef enum
{
  ALIGN_LEFT, ALIGN_RIGHT, ALIGN_CENTERED
}
align_hor_t;

/** Vertical widget alignment. */
typedef enum
{
  ALIGN_TOP, ALIGN_BOTTOM, ALIGN_MIDDLE
}
align_ver_t;

/** Scroll types. */
typedef enum
{
  SCROLL_HOR, SCROLL_VER, SCROLL_BOUNCE
}
scroll_t;

/*------------------------------------------------------------------------
 * Widget data structure                                                  
 *------------------------------------------------------------------------
 */

/** Widget types, stored in #cl_widget. */
typedef enum
{
  CROW,     /**< Row layout widget */
  CCOL,     /**< Column layout widget */
  CFIELD,   /**< Data field */
  CLABEL,   /**< Text label */
  CHSCROLL, /**< Horizontal scroller */
  CVSCROLL  /**< Vertical scroller */
} 
widget_class_t;

/** Size hints for the layout algorithm.
 * The width and height correspond to a penalty, or ugliness. When
 * allocating space to widgets, the layout algorithm attempts to
 * keep the total penalty at a minimum. Or, if it doesn't fit, even
 * with the maximum penalty, the layout is split into smaller parts
 * and the procedure starts all over again. In this case, one layout
 * takes up several screen cycles on the display. 
 */
typedef struct _widget_size_hint
{
  int width;  /**< Width in characters. */
  int height; /**< Height in characters. */
  int penalty; /**< Penalty for these dimensions. */
  
  /** Child layout cache.
   * The \a children_layout field is used to cache the optimum layout
   * of the children for a particular dimension. */
  struct _widget_size_hint **children_layout;
}
widget_size_hint;

/** A widget represents one element of a screen, before it is rendered
 * to a canvas. They can be broadly
 * grouped into layout and content widgets. Layout widgets are rows and
 * columns: they can contain other widgets, and will take care of their
 * layout. Content widges are actually visible on the screen. Their
 * size and position are detemined by the hierarchy of rows and columns
 * above them. 
 */
typedef struct
{
  widget_class_t type; /**< Widget type. */
  field_type subtype;  /**< Widget subtype, only defined for data fields. */
  GSList *children;      /**< Child widgets, contains #cl_widget pointers. */
  GSList *layout_hints;  /**< Contains \ref widget_size_hint pointers. */
    
  /** Horizontal expansion factor.
   * After fitting all widgets with their suggested size,
   * the leftover space is distributed along the widgets according to their
   * expansion factor. Zero means that the widget will always be allocated
   * its minimal size. */
  int expand_x;
  int expand_y; /**< Vertical expansion factor. */

  /** Horizontal alignment.
   * If there is still some space left for widgets to move around after
   * expanding them, they will be packed inside the available space at
   * a position determined by \ref align_hor and \ref align_ver. */
  align_hor_t align_hor;
  align_ver_t align_ver;  /**< Vertical alignment. */

  const char *format;
  const char *name;
  /** Generic data field, the exact use depends on the #type field. */
  void *data;
}
cl_widget;

typedef struct
{
  short      timezone_hrs;
  short      timezone_min;
}
cl_timeformat;

/** An iterator and the list it belongs to. */
typedef struct
{
  GSList **list;
  GSList *iter;
}
list_iter_pair;

static
void _widget_calc_hints (void *widget_, void *dummy);

/*-------------------------------------------------------- _widget_init --*/
/** Initialise a widget with zero sizes and empty lists.
 * \param widget Pointer to the widget to initialise.
 * \param type Type of widget. */
static void
_widget_init (cl_widget *widget, widget_class_t type)
{
  widget->type = type;
  widget->children = NULL;
  widget->layout_hints = NULL;
  widget->expand_x = 1;
  widget->expand_y = 1;
  widget->align_hor = ALIGN_LEFT;
  widget->align_ver = ALIGN_TOP;
  widget->name = NULL;
  widget->data = NULL;
  widget->format = NULL;
}

/*------------------------------------------------------- _generic_free --*/
static void
_generic_free (gpointer object, gpointer user_data)
{
  (void)user_data;
  g_free (object);
}

/*-------------------------------------------------------- _widget_free --*/
static void
_widget_free (gpointer _widget, gpointer user_data)
{
  cl_widget *widget = (cl_widget *)_widget;
  (void)user_data;

  g_slist_foreach (widget->children, _widget_free, NULL);
  g_slist_free (widget->children);
  g_slist_foreach (widget->layout_hints, _generic_free, NULL);
  g_slist_free (widget->layout_hints);
  g_free (widget->data);
  g_free (widget);
}

/*--------------------------------------------------- _layout_elem_free --*/
static void
_layout_elem_free (gpointer _elem, gpointer user_data)
{
  cl_layout_elem *elem = (cl_layout_elem *)_elem;
  (void)user_data;

  g_free (elem->utf8_data);
  cl_free_layout (elem->scroll_region);
}

/*------------------------------------------------- _widget_is_scroller --*/
/** Find out if a widget is a scrolling area.
 * \return Nonzero if \a widget is a scroller, zero if it is not. */
static int
_widget_is_scroller (const cl_widget *widget)
{
  return widget->type == HSCROLL || widget->type == VSCROLL;
}

/*------------------------------------------------ _widget_is_container --*/
/** Find out if a widget is a container for other widgets.
 * \return Nonzero if \a widget is a container, zero if it is not. */
static int
_widget_is_container (const cl_widget *widget)
{
  return    widget->type == CROW || widget->type == CCOL
         || _widget_is_scroller (widget);
}

/*------------------------------------------------------- _hint_compare --*/
/** Compare two layout hints to see if they specify the same width and height.
 * \param _data Pointer to one widget.
 * \param _compare Pointer to the other widget.
 * \return Zero if \a _data and \a _compare are not equal, nonzero if equal. */
static gint
_compare_hints (gconstpointer _data, gconstpointer _compare)
{
  widget_size_hint *hint    = (widget_size_hint*)_data;
  widget_size_hint *compare = (widget_size_hint*)_compare;

  return (hint->width != compare->width || hint->height != compare->height);
}

/*---------------------------------------------------------- _hint_init --*/
/** Initialise a layout hint.
 * \param hint Pointer to the hint to initialise. */
static void
_hint_init (widget_size_hint *hint)
{
  hint->width = 0;
  hint->height = 0;
  hint->penalty = 0;
  hint->children_layout = 0;
}

/*----------------------------------------------------------- _hint_dup --*/
/** Make a copy of a layout hint.
 * \param hint Pointer to the hint to copy.
 * \return Pointer to allocated memory that holds a copy of \ref hint. */
static widget_size_hint *
_hint_dup (widget_size_hint *hint)
{
  widget_size_hint *new_hint = g_malloc (sizeof (widget_size_hint));

  new_hint->width = hint->width;
  new_hint->height = hint->height;
  new_hint->penalty = hint->penalty;
  new_hint->children_layout = 0;
  
  return new_hint;
}

/*------------------------------------------------ _layout_strategy_row --*/
static void
_layout_strategy_row (widget_size_hint *hint, widget_size_hint *new_hint)
{
  assert (hint != NULL);
  assert (new_hint != NULL);

  if (hint->height > new_hint->height) 
    new_hint->height = hint->height;
  
  new_hint->width += hint->width;
}

/*------------------------------------------------ _layout_strategy_col --*/
static void
_layout_strategy_col (widget_size_hint *hint, widget_size_hint *new_hint)
{
  assert (hint != NULL);
  assert (new_hint != NULL);

  if (hint->width > new_hint->width) 
    new_hint->width = hint->width;
  
  new_hint->height += hint->height;
}

/*---------------------------------------------- _widget_calc_penalties --*/
static void
_widget_calc_penalties (cl_widget *widget, 
                        void(*strategy)(widget_size_hint*,
                                        widget_size_hint*))
{
  GSList         *iter;
  list_iter_pair *pairs;
  int            index_; /* "index" was already taken by string.h */
  int            array_size;
  
  assert (widget != NULL);
  assert (strategy != NULL);

  /* All children are instructed to gather their own layout hints. */
  g_slist_foreach (widget->children, _widget_calc_hints, 0);

  /* Reserve enough memory for storing the iterators to the children's
   * lists of layout hints, and store them in the pairs array. */
  array_size = g_slist_length (widget->children);
  pairs = g_malloc (sizeof (list_iter_pair) * array_size);
  
  iter = widget->children;
  index_ = 0;
  while (iter != NULL)
    {
      cl_widget *child = (cl_widget*)iter->data;
      assert (child->layout_hints != NULL);
      pairs[index_].list = &child->layout_hints;
      pairs[index_].iter = child->layout_hints;
      iter = g_slist_next (iter);
      ++index_;
    }

  assert (index_ == array_size);

  /* Loop until we have exhausted all layout combinations. */
  for (;;) 
    {
      widget_size_hint new_hint;
      GSList *found;

      _hint_init (&new_hint);
      
      /* Try the current combination and see what turns up. */
      for (index_ = 0; index_ < array_size; ++index_)
        {
          widget_size_hint *hint;

          hint = (widget_size_hint*)pairs[index_].iter->data;

          if (hint != NULL)
            {
              strategy (hint, &new_hint);
              new_hint.penalty += hint->penalty;
            }
        }

      /* See if the final dimensions were already in the list. */
      found = g_slist_find_custom (widget->layout_hints, 
                                   &new_hint, _compare_hints);

      if (found == NULL)
        {
          /* Not in the list yet. Create a new layout hint and add it to
           * the widget. */

          widget_size_hint *copy = _hint_dup (&new_hint);
          widget_size_hint **array;

          array = g_malloc (sizeof (widget_size_hint*) * array_size);
          for (index_ = 0; index_ < array_size; ++index_)
            array[index_] = pairs[index_].iter->data;

          copy->children_layout = array;
          widget->layout_hints = g_slist_append (widget->layout_hints, copy);
        }
      else
        {
          /* It's already there. If the penalty of the new hint is less than
           * this one, replace it. */

          widget_size_hint *found_hint = found->data;

          if (found_hint->penalty > new_hint.penalty)
            {
              widget_size_hint **array = found_hint->children_layout;

              found_hint->penalty = new_hint.penalty;
              for (index_ = 0; index_ < array_size; ++index_)
                array[index_] = pairs[index_].iter->data;
            }
        }

      /* Find the next combination of children layouts. */
      for (index_ = 0; index_ < array_size; ++index_)
        {
          pairs[index_].iter = g_slist_next (pairs[index_].iter);
          
          if (pairs[index_].iter == NULL)
              pairs[index_].iter = *pairs[index_].list;
          else
              break;
        }

      /* Exit the loop if the possibilities are exhausted. */
      if (index_ == array_size)
        break;

    } /* for (;;) */
}

/*---------------------------------------------------- _add_layout_hint --*/
static void
_add_layout_hint (GSList **list, int width, int height, int penalty)
{
  widget_size_hint *new_hint = g_malloc (sizeof (widget_size_hint));

  new_hint->width   = width;
  new_hint->height  = height;
  new_hint->penalty = penalty;
  *list = g_slist_append (*list, new_hint);
}

/*-------------------------------------------------- _widget_calc_hints --*/
static void
_widget_calc_hints (void *widget_, void *dummy)
{
  cl_widget *widget = (cl_widget*)widget_;
  GSList    **hintlist = &widget->layout_hints;
  GSList    *scriptlist;
  (void)dummy;
  
  switch (widget->type)
    {
    case CLABEL:
      _add_layout_hint (hintlist, strlen ((char*)widget->data), 1, 0);
      break;

    case CHSCROLL:
      widget->subtype = F_HSCROLL;
      /* fallthrough */
    case CROW:
      _widget_calc_penalties (widget, _layout_strategy_row);
      break;

    case CVSCROLL:
      widget->subtype = F_VSCROLL;
      /* fallthrough */
    case CCOL:
      _widget_calc_penalties (widget, _layout_strategy_col);
      break;

    case CFIELD:
      switch (widget->subtype)
        {
          case F_TEXT:
              _add_layout_hint (hintlist, 20, 4, 0);
              _add_layout_hint (hintlist, 20, 2, 1);
              _add_layout_hint (hintlist, 20, 1, 2);
              _add_layout_hint (hintlist, 10, 2, 3);
              _add_layout_hint (hintlist, 10, 1, 4);
              _add_layout_hint (hintlist, 5, 1, 8);
              _add_layout_hint (hintlist, 1, 1, 16);
              break;

          case F_TIME:
              /* Check if this format is handled by Scheme scripts. */
              if (   widget->format != NULL 
                  && (scriptlist = guile_get_time_layout_hints (widget->format))
                      != NULL)
                {
                  GSList *iter = scriptlist;

                  debug ("Found hint list for custom time format");

                  while (iter != NULL)
                    {
                      layout_hint *hint = iter->data;
                      iter = g_slist_next (iter);
                      _add_layout_hint (hintlist, hint->width, hint->height,
                                        hint->penalty);
                      debug_4 ("  new hint %d x %d  for %d", hint->width, hint->height,
                               hint->penalty);
                    }
                }
              else
                {
                  /* Format: 00:00:00 */
                  _add_layout_hint (hintlist, 8, 1, 1);
                  /* Format: 00:00 */
                  _add_layout_hint (hintlist, 5, 1, 4);
                  /* Format: 00 */
                  _add_layout_hint (hintlist, 2, 1, 10);
                }
              break;

          case F_DATE:
              /* Format: 01 jan 2004 */
              _add_layout_hint (hintlist, 11, 1, 0);
              /* Format: 01 jan\2004 */
              _add_layout_hint (hintlist, 6, 2, 2);
              /* Format: 01-01-2004 */
              _add_layout_hint (hintlist, 10, 1, 1);
              /* Format: 01-01-04 */
              _add_layout_hint (hintlist, 8, 1, 5);
              /* Format: 01 jan */
              _add_layout_hint (hintlist, 6, 1, 6);
              /* Format: 01-01 */
              _add_layout_hint (hintlist, 5, 1, 7);
              /* Format: 01 */
              _add_layout_hint (hintlist, 2, 1, 10);
              break;
              
          case F_SCALAR:
              _add_layout_hint (hintlist, 10, 1, 0);
              _add_layout_hint (hintlist, 8, 1, 2);
              _add_layout_hint (hintlist, 4, 1, 3);
              break;
              
          default:
              debug_2 ("   illegal field type %i", widget->subtype);
              assert (0);
        }
      break;

    default:
      debug_2 ("unknown widget type %i", widget->type);
      assert (0);
    }
}

static void
_widget_calc_slack (cl_widget *widget, widget_size_hint *best_hint, 
                    int w, int h, int *slack_w, int *slack_h, int *factor)
{
  unsigned int index_ = 0;
  int tot_height = 0, tot_width = 0;
  GSList *iter = widget->children;
  widget_size_hint *child_hint;

  *factor = 0;
  
  while (iter != NULL)
    {
      cl_widget* child = iter->data;
      iter = g_slist_next (iter);
      child_hint = best_hint->children_layout[index_++];
      assert (child_hint != NULL);

      if (widget->type == CROW)
        {
          debug_3 ("  slack for row : width = %i, expand_x = %i", 
                   child_hint->width, child->expand_x);
          
          tot_width += child_hint->width;
          *factor += child->expand_x;
          if (tot_height > child_hint->height) 
            child_hint->height = tot_height;
        }
      else if (widget->type == CCOL)
        {
          debug_3 ("  slack for col : height = %i, expand_y = %i", 
                   child_hint->height, child->expand_y);
          
          tot_height += child_hint->height;
          *factor += child->expand_y;
          if (tot_width > child_hint->width)
            child_hint->width = tot_width;
        }
    }

  *slack_w = w - tot_width;
  *slack_h = h - tot_height;

  debug_4 ("x slack : %i  (%i - %i)", *slack_w, w, tot_width);
  debug_2 ("x slack : %i", *slack_h);
}

/*---------------------------------------------- _widget_allocate_space --*/
static int
_widget_allocate_space (cl_widget *widget, int w, int h, int x, int y, 
                        struct cl_layout_s* layout)
{
  widget_size_hint *best_hint = NULL;
  cl_widget        *child;
  unsigned int     index_;
  int              slack_w = 0, slack_h = 0, factor = 0, cx = 0, cy = 0, 
                   ex = 0, ey = 0, add;
  float            division = 0.0f, carry = 0.0f;
  cl_layout_elem*  elem = NULL;
  
  /* Check all layout hints, and find the one that has the lowest penalty
   * value, and still fits inside the allocated width and height. */
  GSList *iter = widget->layout_hints;
  debug_3 ("look for hints... I am %d x %d", w, h);
  while (iter != NULL)
    {
      widget_size_hint *hint = iter->data;
      iter = g_slist_next (iter);
      debug_4 ("  considering hint... %d x %d (%d)", hint->width, hint->height
               ,hint->penalty);
      
      /* Scrolling areas are a special case. While their virtual area is
       * allowed to be larger than the allocated area, either the width or
       * the height must fit inside the allocated area. */
      if (widget->type == CHSCROLL)
        {
          if (hint->height > h)
            continue;
        }
      else if (widget->type == CVSCROLL)
        {
          if (hint->width > h)
            continue;
        }
      else if (hint->width > w || hint->height > h)
        {
          continue;
        }
      
      if (best_hint == NULL || best_hint->penalty > hint->penalty)
        best_hint = hint;
    }

  /* If none of the layout hints fits, give up. */
  if (best_hint == NULL)
    return 0;
  
  debug_4 ("best hint: %d x %d (%d)", best_hint->width, best_hint->height, best_hint->penalty);

  /* Determine how much horizontal and vertical slack we have with this
   * set of child layouts. */
  if (_widget_is_container (widget))
    _widget_calc_slack (widget, best_hint, w, h, &slack_w, &slack_h, &factor);

  if (widget->type == CROW || widget->type == CHSCROLL)
    {
      division = (factor == 0) ? 0.0f : slack_w / (float) factor;
      debug_4 ("  hor division (%f) = %i / %i", division, slack_w, factor);
    }
  else if (widget->type == CCOL || widget->type == CVSCROLL)
    {
      division = (factor == 0) ? 0.0f : slack_h / (float) factor;
      debug_4 ("  ver division (%f) = %i / %i", division, slack_h, factor);
    }

  if (widget->type == CLABEL || widget->type == CFIELD)
    {
      elem = layout->layout_data + layout->layout_data_size++;
      elem->x = x;
      elem->y = y;
      elem->width = w;
      elem->height = h;
      elem->is_label = (widget->subtype == F_LABEL);
      elem->format = widget->format;
      elem->utf8_data = g_strdup (widget->subtype == F_LABEL ?
                                  widget->data : widget->name);
      elem->ref_field = NULL;
      elem->scroll_region = NULL;

      return 1;
    }
    
  if (widget->type == CHSCROLL || widget->type == CVSCROLL)
    {
      cl_layout *sub_layout = g_malloc (sizeof (cl_layout)); 

      elem = layout->layout_data + layout->layout_data_size++;
      elem->x = x;
      elem->y = y;
      elem->width = w;
      elem->height = h;
      elem->is_label = (widget->subtype == F_LABEL);
      elem->format = widget->format;
      elem->utf8_data = NULL;
      elem->ref_field = NULL;

      _widget_allocate_space (widget, 
                              best_hint->width, best_hint->height, 
                              0, 0, sub_layout);
      
      elem->scroll_region = sub_layout;
      sub_layout->layout_canvas = cc_create_canvas (elem->width, elem->height);

      return 1;
    }
    
  index_ = 0;
  iter = widget->children;
  while (iter != NULL)
    {
      widget_size_hint *best_layout = best_hint->children_layout[index_];
      child = iter->data;
      iter = g_slist_next (iter);

      if (widget->type == CCOL)
        {
          add = (division * child->expand_y) + carry + 0.999999f;
          carry = division * child->expand_y - add;

          ex = (child->expand_x > 0) ? w : best_layout->width;
          ey = best_layout->height + add;
        }
      else
        {
          add = (division * child->expand_x) + carry + 0.999999f;
          carry = division * child->expand_x - add;

          ex = best_layout->width + add;
          ey = (child->expand_y > 0) ? h : best_layout->height;
        }
        
      switch (child->align_hor)
        {
          case ALIGN_LEFT:     cx = x; break;
          case ALIGN_CENTERED: cx = x + (ex - best_layout->width) / 2; break;
          case ALIGN_RIGHT:    cx = x + ex - best_layout->width; break;
        }

      switch (child->align_ver)
        {
          case ALIGN_TOP:      cy = y; break;
          case ALIGN_MIDDLE:   cy = y + (ey - best_layout->height) / 2; break;
          case ALIGN_BOTTOM:   cy = y + ey - best_layout->height; break;
        }

      _widget_allocate_space (child, ex, ey, cx, cy, layout);

      switch (widget->type)
        {
        case CROW:
          x += ex; /* best_layout->width;  */
          break;
          
        case CCOL:
          y += ey; /* best_layout->height;  */
          break;
          
        default:
          assert (0);
        }
      
      ++index_;
    }

  return 1;
}
 
/*------------------------------------------------- _is_layout_or_field --*/
static int
_is_layout_or_field (xml_node *node)
{
  int tag = node->tag;
  return tag == ROW || tag == COL || tag == FIELD || tag == LABEL;
}

/*------------------------------------------------------- _parse_widget --*/
static cl_widget *
_parse_widget (xml_node *node)
{
  cl_widget   *result = g_malloc (sizeof (cl_widget));
  xml_node    *iter   = NULL;
  const char  *name   = NULL;
  const char  *type   = NULL;
  const char  *halign = NULL;
  const char  *valign = NULL;
  const char  *format = NULL;
  void        *lookup;
  field_type  ftype;

  switch (node->tag)
    {
    case ROW:
      _widget_init (result, CROW); 
      break;
      
    case COL:
      _widget_init (result, CCOL); 
      break;
      
    case HSCROLL:
      _widget_init (result, CHSCROLL); 
      break;
      
    case VSCROLL:
      _widget_init (result, CVSCROLL); 
      break;
      
    case LABEL:
      name = xmlt_get_attrib (node, NAME);
      halign = xmlt_get_attrib (node, HALIGN);
      valign = xmlt_get_attrib (node, VALIGN);
      ftype = F_LABEL;

      _widget_init (result, CLABEL); 
      result->data = g_strdup (xmlt_get_first_cdata (node));
      result->name = g_strdup (name);
      result->subtype = ftype;

      debug_3 ("new label '%s': '%s'", result->name, result->data);

      if (halign != NULL)
        {
          debug_2 ("label halign %s", halign);
          if (!strcmp (halign, "center"))
            result->align_hor = ALIGN_CENTERED;
          else if (!strcmp (halign, "right"))
            result->align_hor = ALIGN_RIGHT;
        }
      
      if (valign != NULL)
        {
          debug_2 ("label valign %s", valign);
          if (!strcmp (valign, "middle"))
            result->align_ver = ALIGN_MIDDLE;
          else if (!strcmp (valign, "bottom"))
            result->align_ver = ALIGN_BOTTOM;
        }

      break;
      
    case FIELD:
      name = xmlt_get_attrib (node, NAME);
      type = xmlt_get_attrib (node, TYPE);
      halign = xmlt_get_attrib (node, HALIGN);
      valign = xmlt_get_attrib (node, VALIGN);
      format = xmlt_get_attrib (node, FORMAT);

      debug_3 ("Field: %s %s", name?name:"NULL",type?type:"NULL");

      if (name == NULL || type == NULL)
        {
          g_free (result);
          return NULL;
        }

      lookup = dict_lookup (&field_type_dict, type);
      if (lookup == NULL)
        {
          g_free (result);
          return NULL;
        }
      
      ftype = *((field_type*) lookup);
      _widget_init (result, CFIELD); 

      if (halign != NULL)
        {
          debug_2 ("halign %s", halign);
          if (!strcmp (halign, "center"))
            result->align_hor = ALIGN_CENTERED;
          else if (!strcmp (halign, "right"))
            result->align_hor = ALIGN_RIGHT;
        }
      
      if (valign != NULL)
        {
          debug_2 ("valign %s", valign);
          if (!strcmp (valign, "middle"))
            result->align_ver = ALIGN_MIDDLE;
          else if (!strcmp (valign, "bottom"))
            result->align_ver = ALIGN_BOTTOM;
        }

      if (format != NULL)
        {
          debug_2 ("format %s", format);
          result->format = g_strdup (format);
        }

      result->name = g_strdup (name);
      result->subtype = ftype;

      switch (ftype)
        {
        case F_LABEL:
        case F_TEXT:
          result->data = g_strdup (xmlt_get_first_cdata (node));
          break;

        default:
          result->data = NULL;
        }
      break;

    default:
      debug_2 ("Unknown layout element '%s'", node->cdata);
      g_free (result);
      return NULL;
    }

  if (   node->tag == ROW || node->tag == COL 
      || node->tag == HSCROLL || node->tag == VSCROLL)
    {
      while ((iter = xmlt_find_if (node, iter, _is_layout_or_field)) != NULL)
        {
          void *child = (void *)_parse_widget (iter);

          if (child == NULL)
            {
              g_free (result);
              return NULL;
            }

          result->children = g_slist_append (result->children, child);
        }
    }

  return result; 
}

/*------------------------------------------------------ cl_make_layout --*/
cl_layout *
cl_make_layout (xml_node *layout, int width, int height)
{
  cl_widget *widgets = 0;
  cl_layout *result = g_malloc (sizeof (cl_layout)); 
  xml_node  *node = 0;
  int       blocksize = sizeof (cl_layout_elem) * width * height;
  
  assert (layout);
  assert (width);
  assert (height);
  
  debug_3 ("Create new layout, size %i x %i", width, height);

  result->layout_data_size = 0;
  result->layout_data = g_malloc (blocksize);
  result->layout_canvas = NULL;

  /* Make a preliminary layout */
  node = xmlt_find_if (layout, node, _is_layout_or_field);
  if (node == NULL)
    return 0;

  widgets = _parse_widget (node);
  _widget_calc_hints (widgets, 0);
  if (!_widget_allocate_space (widgets, width, height, 0, 0, result))
    {
      /** \todo Handle the case where a layout cannot be fitted
       * inside the dimensions of its canvas, and it needs to be split
       * horizontally or vertically. */
    }
  
  _widget_free ((gpointer) widgets, NULL);

  return result;
}

/*------------------------------------------------------ cl_free_layout --*/
void
cl_free_layout (struct cl_layout_s *layout)
{
  size_t i;

  if (layout == NULL)
    return;
  
  for (i = 0; i < layout->layout_data_size; ++i)
    _layout_elem_free (layout->layout_data + i, NULL);
    
  if (layout->layout_canvas != NULL)
    cc_delete_canvas (layout->layout_canvas);

  g_free (layout->layout_data);
  g_free (layout);
}
