/*
 *  charlayout.h - Create a layout for a character-based renderer.
 *                 This file is part of the FreeLCD package.
 *
 *  $Id: charlayout.h,v 1.5 2004/06/20 14:39:57 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file charlayout.h
 * Routines for creating screen layouts for character based devices.
 */


#ifndef _CHARLAYOUT_H
#define _CHARLAYOUT_H

#include <common/xmlt.h>
#include <common/field_types.h>
#include <renderers/charcanvas.h>


/* Forward definition */
struct cl_layout_s;

/** A visible element in the layout of the screen. They are less abstract
 * than widgets: they have a position on a canvas, and a size. The result
 * of fitting a hierarchy of widgets into a canvas, is a set of these
 * structs. 
 */
typedef struct cl_layout_elem_s
{
  int x;            /**< X position on the canvas. */
  int y;            /**< Y position on the canvas. */
  int width;        /**< Width of the element in charaters. */
  int height;       /**< Height of the element in characters. */
  char is_label;    /**< A flag, indicating if it is a static label, or a 
                         dynamic data field. */
  
  const char    *format;         /**< Name of a Guile script to convert this
                                      field to a different format. */
  char          *utf8_data;      /**< Textual content of this element. */
  field         *ref_field;      /**< The corresponding data field. */
  struct cl_layout_s *scroll_region;  /**< Nested layout for scrolling sections. */
}
cl_layout_elem;

/** Physical layout of a screen.
 * After fitting a tree of widgets to a canvas with a given size, this
 * struct contains the resulting visible elements. 
 */
typedef struct cl_layout_s
{
  size_t         layout_data_size; /**< Number of elements in this layout. */
  cl_layout_elem *layout_data;     /**< An array of elements. */

  /** If this layout is used for scrolling a region inside another layout, 
   *  this member variable points to another canvas. Whenever an update is
   *  required, the cr_render routines clip a rectangle from this subcanvas,
   *  and paste it inside the parent. */
  cc_canvas      *layout_canvas;
}
cl_layout;


/** Create a screen layout from XML data. 
 * \param layout The XML layout, as specified by a client.
 * \param width The width of the canvas to fit the layout into.
 * \param height The height of the canvas to fit the layout into.
 */
cl_layout *
cl_make_layout (xml_node *layout, int width, int height);

/** Free all resources used by a layout that was created earlier by
 * cl_make_layout(). 
 * \param layout The layout struct to clean up.
 */
void
cl_free_layout (cl_layout *layout);

#endif /* Multiple include guard */
