/*
 *  charrenderer.c - Simple module for rendering a character-based layout
 *                   to a canvas.
 *                   This file is part of the FreeLCD package.
 *  
 *  $Id: charrenderer.c,v 1.7 2004/06/21 16:37:56 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <assert.h>

#if HAVE_STRING_H
# include <string.h>
#elif HAVE_STRINGS_H
# include <strings.h>
#endif

#include "charrenderer.h"
#include "common/debug.h"
#include "server/guile.h"

CRR_RESULT
cr_render_elem (cl_layout_elem *elem, dictionary *data, cc_canvas *canvas)
{
  assert (elem != NULL);
  assert (data != NULL);
  assert (canvas != NULL);

  debug_4 ("   render element '%s' at (%i,%i)", elem->utf8_data, elem->x, elem->y);

  if (elem->is_label)
    {
      cc_write_text (canvas, elem->x, elem->y, elem->utf8_data);
    }
  else
    {
      char temp[256];
      char *buf;
      
      field *fld = (field *)dict_lookup (data, elem->utf8_data);

      if (fld == NULL)
        return CRR_ERR_INVALID_FIELD;

      switch (fld->type)
        {
          case F_SCALAR:
            sprintf (temp, "%li", fld->data.scalar);
            cc_write_text (canvas, elem->x, elem->y, temp);
            break;

          case F_PERCENTAGE:
            sprintf (temp, "%i%%", (int)(fld->data.percentage * 100));
            cc_write_text (canvas, elem->x, elem->y, temp);
            break;

          case F_TEXT:
            if (fld->data.text != NULL)
              {
                /* \todo : add text support */
              }

            break;
              
          case F_TIME:
            /* Check if there's a Guile script to handle this. */
            if (elem->format != NULL)
              {
                buf = guile_format_time (elem->format, elem->width, 
                                         elem->height,
                                         fld->data.time.seconds,
                                         (fld->data.time.seconds + 3600) % 86400);
                if (buf != NULL)
                  strcpy (temp, buf);
                else
                  return CRR_ERR_GUILE;
              }
            else
              {
                if (elem->width >= 8)
                    sprintf (temp, "%02i:%02i:%02i",
                             fld->data.time.seconds / 3600, 
                             fld->data.time.seconds / 60 % 60,
                             fld->data.time.seconds % 60);

                else if (elem->width >= 5)
                    sprintf (temp, "%02i:%02i",
                             fld->data.time.seconds / 3600, 
                             fld->data.time.seconds / 60 % 60);

                else if (elem->width >= 2)
                    sprintf (temp, "%02i:%02i", fld->data.time.seconds / 3600,
                             fld->data.time.seconds / 60 % 60);
              }

            cc_write_text (canvas, elem->x, elem->y, temp);
            break;

          default:
            debug_2 ("unknown element %i", fld->type);
            snprintf (temp, 256, "%s", "---------------------------------");
            temp[elem->width] = 0;
            cc_write_text (canvas, elem->x, elem->y, temp);
        }
    }

  return CRR_NOTHING;
}

/*----------------------------------------------------------- cr_render --*/
void
cr_render (cl_layout *layout, dictionary *data, cc_canvas *canvas)
{
  cl_layout_elem *elem = layout->layout_data;
  size_t i;

  for (i = 0; i < layout->layout_data_size; ++i, ++elem)
    cr_render_elem (elem, data, canvas);
}
