/*
 *  grcanvas.c - Canvas for graphical devices.
 *               This file is part of the FreeLCD package.
 *
 *  $Id: grcanvas.c,v 1.3 2004/06/21 17:46:07 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(foo)
#endif

#if HAVE_STRING_H
# include <string.h>
#else
# if HAVE_MEMORY_H
#  include <memory.h>
# endif
#endif

#include <glib.h>
#include "grcanvas.h"

#define TRANSPARENT 0
#define OPAQUE      255

#define ALPHABLEND(alpha,val1,val2) \
        ((alpha) * ((int)(val1) - (((int)(val2)) >> 8) + (val2)))
	 
#define ALPHACOMPOSE(alpha1,alpha2) \
	((alpha1) + (((256 - (alpha1)) * (alpha2)) >> 8))

gr_canvas*
gr_create_canvas (int width, int height, color background)
{
  size_t i;
  int size = width * height;
  gr_canvas *new = g_malloc (sizeof (gr_canvas));

  assert (width > 0);
  assert (height > 0);
  
  new->buf = g_malloc (size * sizeof (color));

  for (i = 0; i < size; ++i)
    new->buf[i] = background;

  new->width = width;
  new->height = height;
  new->nr_of_regions = 0;
  new->nr_of_reserved = 0;
  new->damaged_regions = NULL;

  return new;
}

gr_canvas *
gr_copy_canvas (gr_canvas *src)
{
  int size = src->width * src->height;
  gr_canvas *new = g_malloc (sizeof (gr_canvas));

  assert (size > 0);
  assert (src != NULL);
  
  new->buf = g_malloc (size * sizeof (color));
  memcpy (new->buf, src->buf, size * sizeof (color));

  return new;
}

gr_canvas *
gr_copy_rect_from_canvas (gr_canvas *src, int x, int y, int width, int height)
{
  gr_canvas *result = g_malloc (sizeof (gr_canvas));
  color *srcbuf;
  color *destbuf;
  int i;
  
  assert (src != NULL);
  assert (x >= 0);
  assert (x + width <= src->width);
  assert (y >= 0);
  assert (y + width <= src->height);

  result->width = width;
  result->height = height;
  result->buf = g_malloc (width * height * sizeof (color));
  result->nr_of_regions = 0;
  result->nr_of_reserved = 0;
  result->damaged_regions = NULL;

  srcbuf = result->buf;
  destbuf = src->buf + (src->width * y) + x;

  for (i = 0; i < height; ++i)
    {
      memcpy (destbuf, srcbuf, width * sizeof (color));
      destbuf += width * sizeof (color);
      srcbuf += src->width * sizeof (color);
    }

  return result;
}

void
gr_draw_hline (gr_canvas *dest, int x, int y, int width, color col)
{
  color *pos;
  
  if (col.alpha == TRANSPARENT || width == 0 || y < 0 || y >= dest->height)
    return;

  if (width < 0)
    {
      x += width;
      width = -width;
    }
	  
  if (x < 0)
    {
      width += x;
      x = 0;
    }

  if (x >= dest->width)
    return;
  
  if (x + width > dest->width)
    width = dest->width - x;

  pos = dest->buf + (y * dest->width) + x;
  if (col.alpha == OPAQUE)
    {
      for (; width > 0; --width, ++pos)
        {
          *pos = col;
        }
    }
  else
    {
      for (; width > 0; --width, ++pos)
        {
          pos->red = ALPHABLEND (pos->red, col.red, col.alpha);
          pos->green = ALPHABLEND (pos->green, col.green, col.alpha);
          pos->blue = ALPHABLEND (pos->blue, col.blue, col.alpha);
          pos->alpha = ALPHACOMPOSE (pos->alpha, col.alpha);
        }
    }
}

void
gr_draw_vline (gr_canvas *dest, int x, int y, int height, color col)
{
  color *pos;
  
  if (col.alpha == TRANSPARENT || height == 0 || x < 0 || x >= dest->width)
    return;

  if (height < 0)
    {
      y += height;
      height = -height;
    }
	  
  if (y < 0)
    {
      height += y;
      y = 0;
    }

  if (y >= dest->height)
    return;
  
  if (y + height > dest->height)
    height = dest->height - y;

  pos = dest->buf + (y * dest->height) + x;
  if (col.alpha == OPAQUE)
    {
      for (; height > 0; --height, pos += dest->height)
        *pos = col;
    }
  else
    {
      for (; height > 0; --height, pos += dest->height)
        {
          pos->red = ALPHABLEND (pos->red, col.red, col.alpha);
          pos->green = ALPHABLEND (pos->green, col.green, col.alpha);
          pos->blue = ALPHABLEND (pos->blue, col.blue, col.alpha);
          pos->alpha = ALPHACOMPOSE (pos->alpha, col.alpha);
        }
    }
}

void
gr_draw_rect (gr_canvas *dest, int x, int y, int width, int height, color col)
{
  gr_draw_hline (dest, x, y, width, col);
  gr_draw_hline (dest, x, y + height - 1, width, col);
  gr_draw_vline (dest, x, y + 1, height - 2, col);
  gr_draw_vline (dest, x + width - 1, y + 1, height - 2, col);
}

void
gr_fill_rect (gr_canvas *dest, int x, int y, int width, int height, color col)
{
  for (; height > 0; --height, ++y)
    gr_draw_hline (dest, x, y, width, col);
}

