/*
 *  grcanvas.h - Canvas for graphical devices.
 *               This file is part of the FreeLCD package.
 *
 *  $Id: grcanvas.h,v 1.1 2004/01/16 23:42:10 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file grcanvas.h
 * A canvas for graphical devices. This canvas uses true color and alpha
 * channels; it is up to the device drivers to translate this to whatever
 * the device is able to display. The canvas also keeps track of damaged
 * regions to optimize the update process.
 */

#ifndef _FLCD_GRCANVAS_H
#define _FLCD_GRCANVAS_H 1

/** A color, consisting of a red, green, blue, and an alpha component. */
typedef struct
{
  unsigned char  red;
  unsigned char  green;
  unsigned char  blue;
  unsigned char  alpha;
}
color;

/** A damaged region on a canvas. */
typedef struct
{
  int x;      /**< X offset */
  int y;      /**< Y offset */
  int width;  /**< Width of the damaged region in pixels */
  int height; /**< Height of the damaged region in pixels */
}
gr_damaged;

/** A graphical canvas, which is used to hide driver implementation
 *  details from the screen drawing routines. */
typedef struct
{
  int    width;  /**< Width of the canvas in pixels */
  int    height; /**< Height of the canvas in pixels */
  color  *buf;   /**< An array of pixels, where every pixel is represented
		      by a color */

  unsigned int nr_of_regions;      /**< The number of damaged regions */
  unsigned int nr_of_reserved;     /**< The amount of reserved heap space */
  gr_damaged   *damaged_regions;   /**< An array of damaged regions */
}
gr_canvas;


gr_canvas * 
gr_create_canvas (int width, int height, color background);

gr_canvas * 
gr_copy_canvas (gr_canvas *src);

gr_canvas *
gr_copy_rect_from_canvas (gr_canvas *src, int x, int y, int width, int height);

gr_canvas *
gr_create_from_png (void *png_data);

gr_canvas *
gr_delete_gr_canvas (gr_canvas *bmap);

void
gr_put_pixel (gr_canvas *dest, int x, int y, color pixel);

color
gr_get_pixel (gr_canvas *dest, int x, int y);

void
gr_draw_line (gr_canvas *dest, int x1, int y1, int x2, int y2, color col);

void
gr_draw_hline (gr_canvas *dest, int x, int y, int width, color col);

void
gr_draw_hline (gr_canvas *dest, int x, int y, int height, color col);

void
gr_draw_rect (gr_canvas *dest, int x, int y, int width, int height, color col);

void
gr_fill_rect (gr_canvas *dest, int x, int y, int width, int height, color col);

void
gr_draw_gr_canvas (gr_canvas *dest, int x, int y, gr_canvas *src);

void
gr_resize_canvas (gr_canvas *dest, int new_width, int new_height);

void
gr_optimize_damaged_regions (gr_canvas *dest);

#endif

