/*
 *  drivers.c - Loading and managing the driver plugins.
 *              This file is part of the FreeLCD package.
 *
 *  $Id: drivers.c,v 1.8 2004/06/20 14:14:09 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */


#include <stdlib.h>

#if HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_STRING_H
# include <string.h>
#endif

#if HAVE_SYSLOG_H
# include <syslog.h>
#else
# define syslog(FOO,BAR,BAZ)
#endif

#include <glib.h>

#include "../common/xmlt.h"
#include "../common/backend.h"
#include "../common/debug.h"

#include "ltdl.h"
#include "drivers.h"

#ifndef FLCDD_MODULE_PATH_ENV
# define FLCDD_MODULE_PATH_ENV "FLCDD_MODULE_PATH"
#endif

typedef enum { DRIVER = 100, BACKEND = 101 } tag_idx;
typedef enum { NAME = 100 } attr_idx;

static tag_idx tag_array[] = { DRIVER, BACKEND };
static attr_idx attr_array[] = { NAME };

static dict_pair tag_translate[] = {
    { "backend", tag_array + 1 },
    { "driver", tag_array + 0 }
};

static dict_pair attr_translate[] = {
    { "name", attr_array + 0 }
};

static dictionary tag_dict = { tag_translate, 
                               sizeof (tag_translate) / sizeof (dict_pair) };
static dictionary attr_dict = { attr_translate,
                               sizeof (attr_translate) / sizeof (dict_pair) };

static GSList *driver_list;

/** Driver information is stored here. */
typedef struct
{
  /** The libtool module that we are using. */
  lt_dlhandle    module;
  /** A driver handle, returned by drv_create_handle(). */
  void           *drv_handle;
  /** A backend handle, returned by backend_create_handle(). */
  void           *backend_handle;
  /** This function must be called to release a driver handle. It is the
   * drv_close_handle() of a loaded driver module. */
  void           (*destroy_handle) (void *handle);
  const char *   (*get_info) (const char *field);
  void           (*process_canvas) (void *handle, void *canvas);
  drv_dimensions (*get_dimensions) (void *handle);
  drv_type       type;
}
driver_info;


static int
_ltdl_error ()
{
  debug_2 ("libtool init failed: %s\n", lt_dlerror ());
  syslog (LOG_ERR, "libtool init failed: %s\n", lt_dlerror ());

  return -1;
}

static void *
_module_error (const char *name)
{
  debug_3 ("loading plug-in '%s' failed: %s\n", name, lt_dlerror ());
  syslog (LOG_ERR, "loading plug-in '%s' failed: %s\n", name, lt_dlerror ());

  return 0;
}

static void *
_create_failed (void *backend)
{
  backend_free_handle (backend);
  return 0;
}

static void
_free_driver_info (gpointer _info, gpointer user_data)
{
  driver_info *info = (driver_info *)_info;

  debug_2 ("  cleaning up driver %s", info->get_info ("name"));
  info->destroy_handle (info->drv_handle);
  lt_dlclose (info->module);
  g_free (info);
}

int
drivers_init ()
{
  static int initialized = 0;
  const char *path = 0;

  if (initialized)
    return -1;

  /* lt_dlmalloc = (lt_ptr(*)(size_t)) g_malloc; */

  /* Preload anything, if necessary */
  /* LTDL_SET_PRELOADED_SYMBOLS(); */
  
  if (lt_dlinit () != 0)
    return _ltdl_error ();

  path = getenv (FLCDD_MODULE_PATH_ENV);
  if (path != NULL && lt_dlsetsearchpath (path) != 0)
    return _ltdl_error ();

  lt_dladdsearchdir (MODULE_PATH);
  initialized = 1;
  driver_list = 0;

  return 0;
}

void *
drivers_create_output (xml_node *config)
{
  typedef void*(ch_func)(xml_node *);
  typedef void(bb_func)(void *, void *);
  typedef void(dh_func)(void *);
  typedef const char *(gi_func)(const char *);
  typedef drv_dimensions(dm_func)(void *);
  typedef drv_type(gettype_func)(void);

  driver_info  *info;
  ch_func      *create_handle;
  bb_func      *bind_backend;
  gettype_func *get_type; 
  
  void       *backend;
  void       *driver;
  const char *backend_name;
  const char *driver_name;
  xml_node   *found;
  lt_dlhandle module;

  xmlt_rescan_document (config, &tag_dict, &attr_dict);
  
  found = xmlt_find (config, 0, BACKEND);
  if (found == NULL)
    return 0;
  
  backend_name = xmlt_get_attrib (found, NAME);
  backend = backend_create_handle (found, backend_name);
  if (backend == NULL)
    return 0;

  found = xmlt_find (config, 0, DRIVER);
  if (found == NULL)
    return _create_failed (backend);

  driver_name = xmlt_get_attrib (found, NAME);
  if (driver_name == NULL)
    return _create_failed (backend);

  module = lt_dlopenext (driver_name);
  if (module == NULL)
    return _module_error (driver_name);
  
  create_handle = (ch_func*)lt_dlsym (module, "drv_create_handle");
  if (create_handle == NULL)
    return _module_error(driver_name);

  bind_backend   = (bb_func*)lt_dlsym (module, "drv_bind_backend");
  if (bind_backend == NULL)
    return _module_error(driver_name);
  
  get_type       = (gettype_func*)lt_dlsym (module, "drv_get_type");
  if (get_type == NULL)
    return _module_error(driver_name);
  
  driver = create_handle (found);
  if (driver == NULL)
    return _create_failed (backend);
  
  info = g_new (driver_info, 1);
  info->module         = module;
  info->drv_handle     = driver;
  info->backend_handle = backend;
  info->destroy_handle = (dh_func*)lt_dlsym (module, "drv_destroy_handle");
  info->get_info       = (gi_func*)lt_dlsym (module, "drv_get_info");
  info->get_dimensions = (dm_func*)lt_dlsym (module, "drv_get_dimensions");
  info->process_canvas = (bb_func*)lt_dlsym (module, "drv_process_canvas");
  info->type           = get_type();
  
  bind_backend (driver, backend);
  debug_2 ("Adding driver %s to list", info->get_info ("name"));
  driver_list = g_slist_append (driver_list, info);
  
  return info;  
}

void
drivers_process_canvas (void *_info, cc_canvas *canvas)
{
  driver_info *info = (driver_info *)_info;

  info->process_canvas (info->drv_handle, canvas);
}

drv_type
drivers_get_type (void *_info)
{
  driver_info *info = (driver_info *)_info;

  return info->type;
}

drv_dimensions
drivers_get_dimensions (void *_info)
{
  driver_info *info = (driver_info *)_info;

  return info->get_dimensions (info->drv_handle);
}

void
drivers_cleanup ()
{
  g_slist_foreach (driver_list, _free_driver_info, NULL);
  g_slist_free (driver_list);
  lt_dlexit ();
}

