/*
 *  guile.c - Routines for setting up libguile.
 *            This file is part of the FreeLCD package.
 *  
 *  $Id: guile.c,v 1.4 2004/06/20 14:16:19 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */


#if HAVE_CONFIG_H
# include "config.h"
#endif

#include "guile.h"

#if HAVE_STRING_H
# include <string.h>
#endif

#if HAVE_STRINGS_H
# include <strings.h>
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
#else
# if HAVE_SYS_DIRENT_H
#  include <dirent.h>
# endif
#endif

#if HAVE_TYPES_H
# include <types.h>
#else
# if HAVE_SYS_TYPES_H
#  include <sys/types.h>
# endif
#endif

#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#include <libguile.h>
#include <guile/gh.h>

#include "../common/layout_hint.h"
#include "../common/debug.h"
extern int keep_going;

typedef struct
{
  const char *name;
  GSList     *layout_hints;
} time_format;

static GSList *time_formats;
static GSList *date_formats;
static SCM guile_hooks[10];


static void
_free_generic (gpointer _ptr, gpointer user_data)
{
  g_free (_ptr);
}

static void
_free_time_format (gpointer _ptr, gpointer user_data)
{
  time_format *fmt = (time_format *)_ptr;

  debug_3 ("Free Guile scripted time format \"%s\", %p", fmt->name, (void*)fmt->name);
  g_slist_foreach (fmt->layout_hints, _free_generic, NULL);
  g_slist_free (fmt->layout_hints);
  g_free ((char *)fmt->name);
  g_free (fmt);
}

static void
_free_date_format (gpointer _ptr, gpointer user_data)
{
/*  date_format *fmt = (date_format *)_ptr; */
}

static gint
_guile_compare_timefmt (gconstpointer _fmt, gconstpointer _name)
{
  const time_format *fmt = (const time_format *)_fmt;
  const char *name = (const char *)_name;

  return strcmp (fmt->name, name);
}

static SCM
_flcd_time_format (SCM name, SCM layout_list)
{
  const char  *name_str;
  time_format *new_format;
  layout_hint *new_hint;
  size_t      len;

  if (!SCM_STRINGP (name))
    return SCM_UNSPECIFIED;
  
  name_str = SCM_STRING_CHARS (name);
  len = scm_ilength (layout_list);
  debug_3 ("New time format : %s  (%d hints)", name_str, len);

  new_format = g_new (time_format, 1);
  new_format->layout_hints = NULL;
  new_format->name = g_strdup (name_str);
  debug_3 ("   name is \"%s\", %p", name_str, (void*)name_str);

  for (; len && !SCM_NULLP (layout_list); --len, layout_list = SCM_CDR (layout_list))
    {
      new_hint = g_new (layout_hint, 1);

      new_hint->width = SCM_INUM (SCM_CAAR (layout_list));
      new_hint->height = SCM_INUM (SCM_CADAR (layout_list));
      new_hint->penalty = SCM_INUM (SCM_CADDAR (layout_list));

      debug_4 ("   hint: %d x %d  for %d", new_hint->width,new_hint->height,new_hint->penalty);
      
      new_format->layout_hints = 
        g_slist_append (new_format->layout_hints, new_hint);
    }
  
  debug ("   done");
  time_formats = g_slist_append (time_formats, new_format);

  return SCM_UNSPECIFIED;
}

static SCM
_flcd_server_version (void)
{
  return scm_makfrom0str (PACKAGE_VERSION);
}

static SCM
_flcd_server_quit (void)
{
  keep_going = 0;

  return SCM_UNSPECIFIED;
}

static void
_traverse_directory (const char *location)
{
  size_t        location_len;
  DIR           *directory;
  struct dirent *entry;
  char          *file_name;
  struct stat   filestat;
  int           fd;
  mode_t        mode;
  
  directory = opendir (location);
  if (directory == NULL)
    {
      debug_2 ("Cannot open %s", location);
      return;
    }
  
  debug_2 ("Reading Guile scripts from %s", location);

  location_len = strlen (location);
  file_name = g_malloc (location_len + 256);
  memcpy (file_name, location, location_len);

  /* Make sure there's a separator between the directory and the file name. */
  if (file_name[location_len - 1] != '/')
    {
      file_name[location_len] = '/';
      ++location_len;
    }
  
  while ((entry = readdir (directory)) != NULL)
    {
      size_t entry_length;

      if (entry->d_name == NULL)
        break;
      
      entry_length = strlen (entry->d_name);
      if (entry_length >= 255) /* Too long to be trusted */
        continue;
      
      memcpy (file_name + location_len, entry->d_name, entry_length + 1);
      if ((fd = open (file_name, O_RDONLY)) < 0)
        continue;

      if (fstat (fd, &filestat) < 0)
        {
          close (fd);
          continue;
        }
      
      close (fd);
      mode = filestat.st_mode;
      
      if (S_ISLNK (mode)) /* We do not follow symbolic links. */
        continue;
      
      if (S_ISDIR (mode) && entry->d_name[0] != '.') /* Recursive traversal. */
        {
          _traverse_directory (file_name);
        }
      else if (   S_ISREG (mode) 
               && !strcasecmp (entry->d_name + entry_length - 4, ".scm"))
        {
          debug_2 ("  interpreting %s ...", file_name);
          scm_primitive_load (scm_makfrom0str (file_name));
        }
    }
  
  g_free (file_name);
  closedir (directory);
}

void
guile_call_hook (guile_hook id, void *data, long param1, long param2)
{
  switch (id)
    {
    case HOOK_NEXT_SCREEN:
      break;
      
    case HOOK_DATA_UPDATE:
      break;
      
    case HOOK_OPEN_CONNECTION:
      break;

    case HOOK_CLOSE_CONNECTION:
      break;
    }
}

GSList *
guile_get_time_layout_hints (const char *name)
{
  GSList *found = g_slist_find_custom (time_formats, name, 
                                       _guile_compare_timefmt);

  if (found == NULL)
    return NULL;
      
  return ((time_format*)found->data)->layout_hints;
}

char*
guile_format_time (const char *name, int width, int height, 
                   int utct, int localt)
{
  size_t namelen = strlen (name);
  char   *result;
  char   *buf;
  GSList *found = g_slist_find_custom (time_formats, name, 
                                       _guile_compare_timefmt);
  debug_2 ("format time for %s", name);

  if (found == NULL)
    {
      debug ("nothing found");
    return NULL;
    }
  
  buf = g_malloc (namelen + 64);
  sprintf (buf, "(time-%s %d %d %d %d)", name, utct, localt, width, height);
  result = SCM_STRING_CHARS (scm_eval_string (scm_makfrom0str (buf)));
  g_free (buf);

/*  return g_strdup (result); */
  return result;
}

void
guile_init ()
{
  const char *location = getenv ("FLCDD_SCRIPT_PATH");
  
  time_formats = NULL;
  date_formats = NULL;
  
  debug ("adding guile functions");
  scm_c_define_gsubr ("flcd:server-version", 0, 0, 0, _flcd_server_version);  
  scm_c_define_gsubr ("flcd:server-quit"   , 0, 0, 0, _flcd_server_quit);  
  scm_c_define_gsubr ("flcd:time-format"   , 2, 0, 0, _flcd_time_format);  

  debug ("adding guile hooks");
  guile_hooks[0] = scm_create_hook ("flcd:hook:next-screen", 1);
  guile_hooks[1] = scm_create_hook ("flcd:hook:data-update", 2);
  guile_hooks[2] = scm_create_hook ("flcd:hook:open-connection", 2);
  guile_hooks[3] = scm_create_hook ("flcd:hook:close-connection", 1);

  if (location == NULL)
    location = "../scripts/";

  _traverse_directory (location);
}

void
guile_cleanup ()
{
  debug ("guile_cleanup()");

  g_slist_foreach (time_formats, _free_time_format, NULL);
  g_slist_free (time_formats);
  g_slist_foreach (date_formats, _free_date_format, NULL);
  g_slist_free (date_formats);
}
