/*
 *  main.c - main() of the server (flcdd)
 *           This file is part of the FreeLCD package.
 *
 *  $Id: main.c,v 1.10 2004/06/20 14:17:34 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#define GNU_PREFIX ""   /* "GNU " */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#ifndef APP_DEFAULT_USER
# define APP_DEFAULT_USER "freelcd"
#endif

#if defined (MSDOS) || defined (WIN32) || defined (_WIN64)
# include <io.h>  /* For close() */
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#else
# if HAVE_SYS_UNISTD_H
#  include <sys/unistd.h>
# endif
#endif

#if HAVE_LOCALE_H
# include <locale.h>
#endif

#if HAVE_SYSLOG_H
# include <syslog.h>
#else
# define syslog(FOO,BAR,BAZ)
#endif

#if HAVE_TYPES_H
# include <types.h>
#else 
# if HAVE_SYS_TYPES_H
#  include <sys/types.h>
# endif
#endif

#if HAVE_STRING_H
# include <string.h>
#else
# if HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif

#include <assert.h>
#include <libguile.h>
#include <glib.h>
#include <gmodule.h>

#if STDC_HEADERS
# include <stdlib.h>
#endif

#include <pwd.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <sys/stat.h>

#if HAVE_GETOPT_LONG
# include <getopt.h>
#endif

#if HAVE_SOCKET_H
# include <socket.h>
#else
# if HAVE_SYS_SOCKET_H
#  include <sys/socket.h>
# endif
#endif

#include "renderers/charlayout.h"
#include "renderers/charrenderer.h"
#include "renderers/charcanvas.h"

#include "common/socket.h"
#include "common/xmlt.h"
# include "common/debug.h"

#include "connections.h"
#include "drivers.h"
#include "xmlconfig.h"
#include "scheduler.h"
#include "guile.h"

/* gettext helper macro */
#if ENABLE_NLS
# include <libintl.h>
# define _(foo) gettext (foo)
#else
# define bindtextdomain(X,Y)
# define textdomain(Domain)
# define _(foo) foo
#endif /* enable_nls */

/* Options definition */
#if HAVE_GETOPT_LONG
static struct option longopts[] = {
  {"port", required_argument, 0, 'p'},
# if HAVE_FORK
   {"foreground", no_argument, 0, 'f'},
# endif
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, 'V'},
  {0, 0, 0, 0}
};
#endif /* have getopt_long */

void inner_main (void *, int, char **);

static GMainLoop *main_loop = NULL; /**< GLib main event loop */
static guint     timer_obj  = 0;    /**< Timer object for calling sch_tick() */
static socket_t  sock;              /**< The server socket */
int              keep_going;



/*------------------------------------------------------- exit_program --*/
static void
exit_program ()
{
  syslog (LOG_INFO, _("exiting"));

  conn_close_all ();
  close (sock);
  sch_cleanup ();
  guile_cleanup ();
  drivers_cleanup ();
  if (timer_obj > 0)
    g_source_remove (timer_obj);

  if (main_loop != NULL)
    g_main_loop_unref (main_loop);

  closelog ();
  close_debug_output ();
  exit (EXIT_FAILURE);
}

/*----------------------------------------------------- sighup_handler --*/
static void
sighup_handler ()
{
  syslog (LOG_INFO, _("received SIGHUP"));
  /** \todo: reload configuration */
}

/*---------------------------------------------------- sigterm_handler --*/
static void
sigterm_handler ()
{
  syslog (LOG_INFO, _("received SIGTERM, exiting"));
  exit_program ();
}

/*---------------------------------------------------- sigterm_handler --*/
static void
sigint_handler ()
{
  syslog (LOG_INFO, _("received SIGINT, exiting"));
  exit_program ();
}

/*------------------------------------------------------------ sys_err --*/
static void
sys_err (const char *errstr)
{
  syslog (LOG_ERR, errstr);
  exit_program ();
}

/*---------------------------------------------------------- show_help --*/
static void
show_help (const gchar *app_name)
{
  printf (_("Usage: %s [options] ...\nOptions:\n"), app_name);
#ifdef HAVE_GETOPT_LONG
  printf (_(" \
--port, -p <portnr> Let the server listen on another port number\n"));
# if HAVE_FORK
  printf (_(" --foreground, -f    Run the server in the foreground\n"));
# endif
  printf (_(" --help, -h          Program usage\n"));
  printf (_(" --version, -V       Show program version and exit\n"));
#else /* ! HAVE_FORK */
  printf (_(" -p <portnr>  Let the server listen on another port number\n"));
# if HAVE_FORK
  printf (_(" -f           Run the server in the foreground\n"));
# endif
  printf (_(" -h           Program usage\n"));
  printf (_(" -V           Show program version and exit\n"));
#endif /* HAVE_FORK */
  printf (_("\nReport bugs to <%s>\n"), PACKAGE_BUGREPORT);
}

/*------------------------------------------------------- show_version --*/
static void
show_version (const gchar *app_name)
{
  printf ("%s - %s%s\n", app_name, GNU_PREFIX, PACKAGE_STRING);
  printf (_("Copyright (C) %s %s\n"), "2002, 2003", "Jeroen van den Berg");
  printf (_("\
This program is free software; you may redistribute it under the terms of\n\
the GNU General Public License.  This program comes with absolutely NO\n\
WARRANTY. For more information about these matters, see the file named\n\
COPYING.\n"));
}

/*------------------------------------------------ drop_root_privilege --*/
static void
drop_root_privilege (const char *user)
{
  struct passwd *pw_ent;

  /*  If we have root privs, try to set our uid to something safer.
   *  First try a specified username, then try "nobody" (which is a
   *  reasonable fallback on most systems).
   */
  if (getuid () == 0 || geteuid () == 0)
    {
      debug_2 ("running as root, try to become user '%s'", user);

      pw_ent = getpwnam (user);
      if (pw_ent == NULL)
        {
          debug_2 ("WARNING: %s is not a valid user, fallback to 'nobody'", 
                   user);

          syslog (LOG_WARNING,
                  _("user '%s' is not a valid user, fallback to 'nobody'\n"),
                  user);

          pw_ent = getpwnam ("nobody");
          if (pw_ent == NULL)
            {
              sys_err (_("cannot drop root privileges, aborting"));
              exit (EXIT_FAILURE);
            }
        }
      else
        {
          debug_2 ("dropping root privileges, switching to %s", user);
        }

      if (setuid (pw_ent->pw_uid) < 0)
        sys_err ("setuid");
    }
}


/*--------------------------------------------------------------- main --*/
int
main (int argc, char **argv)
{
  scm_boot_guile (argc, argv, inner_main, 0);
  return EXIT_SUCCESS;
}
        
void 
inner_main (void *closure, int argc, char **argv)
{
  char         c;
  int          run_in_foreground = 0; /* (--foreground) */
  unsigned int port = 8783; /* (--port) */
  const gchar  *app_name;
  pid_t        child; 
  long         tmp;
  fd_set       fdset;
  xml_node     *xconf;
  xml_node     *output_config;
  void         *driver;
  GIOChannel   *giochan;
  char         *bind_addr = 0;
  
#if HAVE_SETLOCALE
  /* Set up gettext internationalisation */
  if (setlocale (LC_ALL, "") == NULL)
    {
      perror (_("warning: setting locale to LC_ALL failed"));
    }
#endif

#if ENABLE_NLS
  bindtextdomain (PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  textdomain (PACKAGE);
#endif

  /* Check if we can dynamically load modules. */
  if (!g_module_supported())
    {
      perror (_("error: dynamic module loading is not supported on "
                "this platform, cannot load drivers"));
      exit (EXIT_FAILURE);
    }

  /* Find out name of executable */
  app_name = *argv + strlen (*argv) - 1;
  while (app_name > *argv && *(app_name - 1) != G_DIR_SEPARATOR)
    --app_name;

  /* Parse command line options */
#ifdef HAVE_GETOPT_LONG
  while ((c = getopt_long (argc, argv, "p:fbhV", longopts, 0)) > 0)
#else
  while ((c = getopt (argc, argv, "p:fbhV")) > 0)
#endif
    {
      switch (c)
        {
        case 'p':                 /* --port, -p */
          tmp = strtol (optarg, 0, 10);
          if (errno == ERANGE || errno == EINVAL || tmp < 1 || tmp > 65536)
            {
              fprintf (stderr,
                _("%s: Port number must be between 1 and 65536\n"), app_name);
              exit (EXIT_FAILURE);
            }
          port = (unsigned int)tmp;
          break;

        case 'f':                /* --foreground, -f */
          run_in_foreground = 1;
          break;
              
        case 'h':                /* --help, -h */
          show_help (app_name);
          exit (EXIT_SUCCESS);

        case 'V':                /* --version, -V */
          show_version (app_name);
          exit (EXIT_SUCCESS);

        case '?':               /* Invalid option */
          show_help (app_name);
          exit (EXIT_FAILURE);

        default:
          /* In an ideal world, this shouldn't happen. */
          exit (EXIT_FAILURE);
        }
    }

  if (optind < argc)
    {
      perror (_("too many arguments"));
      exit (EXIT_FAILURE);
    }

  /* Fork into the background */
#ifdef HAVE_FORK
  if (!run_in_foreground)
    {
      child = fork();
      if (child < 0)
        sys_err ("fork");
            
      if (child != 0)
        exit (EXIT_SUCCESS);

      /* Create a new session */
      if (setsid() < 0 )
        sys_err ("setsid");
    }
#endif

  /* Configure system logging */
  openlog (app_name, LOG_CONS | LOG_PID | LOG_PERROR, LOG_DAEMON);

  /* Take some precautions */
  /* FIXME: uncomment this part when configuration files are handled properly.
  if (chdir( "/" ) < 0)
    sys_err ("chdir");
  */
  umask (0);
  
  if (run_in_foreground)
    {
      set_debug_output (fopen ("log.txt", "w"));
    }
  else
    {
      close (STDOUT_FILENO);
      close (STDERR_FILENO);
    }
  close (STDIN_FILENO);

  /* Parse the configuration file. For now, we look in the current directory.
   * Somewhere in the future this should become /etc or ~ . 
   */
  xconf = xc_parse_config_file ("flcdd.xml");
  if (xconf == NULL)
    {
      sys_err (_("Failed to parse inifile"));
      exit (EXIT_FAILURE);
    }

  /* Do we need to bind the server socket to another address? (By default
   * it is bound to 127.0.0.1) */
  /* \todo Implement this. */
  
  /* Open server socket */
  sock = socket_create_server (port, &fdset, bind_addr);
  if (sock < 0)
    {
      switch (sock)
        {
        case -1:
          sys_err ("socket() failed, no socket could be created");
          break;
        case -2:
          sys_err ("setsockopt() failed, socket options could not be set");
          break;
        case -3:
          sys_err ("inet_aton() failed, network address could not be translated");
          break;
        case -4:
          sys_err ("bind() failed, socket could not be bound to address");
          break;
        default:
          sys_err (_("cannot open server socket"));
          break;
        }
      exit (EXIT_FAILURE);
    }

  /* Initialise screen scheduling. */
  sch_init ();

  /* Load driver modules */
  if (drivers_init () != 0)
    {
      sys_err (_("Failed to initialise drivers"));
      exit (EXIT_FAILURE);
    }
  
  output_config = 0;
  while ((output_config = xmlt_find (xconf, output_config, OUTPUT)) != 0)
    {
      const char *name = xmlt_get_attrib (output_config, NAME);

      if (name == NULL)
        {
          puts ("No output name given.");
          continue;
        }
      
      driver = drivers_create_output (output_config);

      if (driver == NULL)
        {
          debug_2 ("drivers_create_output failed for %s", name);
          syslog (LOG_ERR, _("Failed to create output for '%s'"), name);
        }
      else
        {
          if (!sch_register_output (driver, name))
            {
              debug_2 ("failed to register output %s", name);
              syslog (LOG_ERR, _("Failed to register output for '%s'"), name);
            }
        }
    }

  /* We do not need root privileges anymore, drop them before
   * messing with potential dangers like plug-ins. 
   */
  drop_root_privilege (APP_DEFAULT_USER);

  /* Add the socket events to the main loop. */
  main_loop = g_main_new (1);

  giochan = g_io_channel_unix_new (sock);
  assert (giochan != NULL);
  g_io_add_watch (giochan, G_IO_IN, conn_open, NULL);
  
  xmlt_free_document (xconf);
  guile_init ();
  
  if (signal (SIGHUP, sighup_handler) == SIG_ERR)
    {
      sys_err (_("Cannot catch SIGHUP"));
      exit (EXIT_FAILURE);
    }
  
  if (signal (SIGTERM, sigterm_handler) == SIG_ERR)
    {
      sys_err (_("Cannot catch SIGHUP"));
      exit (EXIT_FAILURE);
    }

  if (signal (SIGINT, sigint_handler) == SIG_ERR)
    {
      sys_err (_("Cannot catch SIGINT"));
      exit (EXIT_FAILURE);
    }

  /* Enter event loop */
  debug ("entering main loop");
  if (!run_in_foreground)
    syslog (LOG_INFO, _("running"));
  
  timer_obj = g_timeout_add (50, sch_tick, NULL);
  g_main_run (main_loop);
  
  keep_going = 1;
/*
  while (keep_going != 0)
    {
       *  Poll the sockets from our file descriptor set, and call
       *  one of the conn_* functions whenever something interesting
       *  happens.
       *
      socket_poll (sock, &fdset, conn_open, conn_data, conn_close);

       *  Try to keep the scheduler ticks in a 20 Hz pace. *
      sch_tick (NULL);
    }
*/

  exit_program ();
}
