/*
 *  scheduler.c : Routines for managing screens and scheduling them for
 *                being displayed.
 *                This file is part of the FreeLCD package.
 *
 *  $Id: scheduler.c,v 1.6 2004/06/20 14:18:52 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file scheduler.c
 * Routines for managing screens and scheduling them for being displayed.
 */

#include <stdio.h>
#include <errno.h>

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_STRING_H
# if !STDC_HEADERS && HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#endif

#if !HAVE_SNPRINTF
# define snprintf(A,B,C,D)  printf(A,C,D)
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(FOO)
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#elif HAVE_SYS_UNISTD_H
# include <sys/unistd.h>
#endif

#include <glib.h>

#include "scheduler.h"

#include "server/connections.h"
#include "server/drivers.h"
#include "common/field_types.h"
#include "common/xmlt.h"
#include "common/debug.h"
#include "renderers/charlayout.h"
#include "renderers/charrenderer.h"
#include "renderers/layout_tags.h"

typedef enum
{
  ERR_NONE = 0, 
  ERR_DATA_INCOMPLETE = 1, 
  ERR_DUPLICATE_NAMES = 2, 
  ERR_UNKNOWN_SCREEN_ID = 3
}
error_code;

static const char* error_description[] =
{
  "",
  "not all data fields have a name and type",
  "duplicate data field names",
  "no screen with that ID has been registered yet"
};

/*------------------------------------------------------------------------*/
/* XML dictionaries */

typedef enum
{
  VALUE, HRS, MIN, SEC, MSEC, DAY, MON, YEAR 
} data_tag_t;	

static data_tag_t data_tags[] =
{ 
  VALUE, HRS, MIN, SEC, MSEC, DAY, MON, YEAR 
};

static tag_t cl_tagindex_[] =
{
  ROW, COL, FIELD, LABEL
};

static dict_pair data_tag_pairs[] = 
{
  { "col",   &cl_tagindex_[1] },
  { "day",   &data_tags[5]    },
  { "field", &cl_tagindex_[2] },
  { "hrs",   &data_tags[1]    },
  { "label", &cl_tagindex_[3] },
  { "min",   &data_tags[2]    },
  { "mon",   &data_tags[6]    },
  { "msec",  &data_tags[4]    },
  { "row",   &cl_tagindex_[0] },
  { "sec",   &data_tags[3]    },
  { "value", &data_tags[0]    },
  { "year",  &data_tags[7]    }
};  


static attr_t cl_attrindex_[] =
{
  TYPE, ID, NAME, VALIGN, HALIGN, EXPAND, FORMAT, TIMEZONE, SCROLL
};	  
	  
static dict_pair data_attr_pairs[] =
{
  { "expand",   &cl_attrindex_[5]  },
  { "format",   &cl_attrindex_[6]  },
  { "halign",   &cl_attrindex_[4]  },
  { "id",       &cl_attrindex_[1]  },
  { "name",     &cl_attrindex_[2]  },
  { "scroll",   &cl_attrindex_[8]  },
  { "timezone", &cl_attrindex_[7]  },
  { "type",     &cl_attrindex_[0]  },
  { "valign",   &cl_attrindex_[3]  }
};

static dictionary data_tag_dict =
    { data_tag_pairs, sizeof (data_tag_pairs) / sizeof (dict_pair) };

static dictionary data_attr_dict =
    { data_attr_pairs, sizeof (data_attr_pairs) / sizeof (dict_pair) };

/* The dictionary for field types is defined in common/field_types.c */
extern dictionary field_type_dict;


/** All currently registered screens. */
static GSList  *screenlist;
/** All currently registered devices. */
static GSList  *outputlist;

/*-------------------------------------------------- _screen_id_compare --*/
static gint
_screen_id_compare (gconstpointer _scrn, gconstpointer _id)
{
  const sch_screen *scrn = (const sch_screen *)_scrn;
  const screen_id *id    = (const screen_id *)_id;

  assert (scrn != NULL);
  assert (id != NULL);

  return    scrn->id.originator != id->originator 
         || strcmp (scrn->id.name, id->name);
}

/*------------------------------------------------ _driver_name_compare --*/
static gint
_driver_name_compare (gconstpointer _drv, gconstpointer _name)
{
  const sch_output *drv  = (const sch_output *)_drv;
  const char       *name = (const char *)_name;

  assert (drv != NULL);
  assert (drv->device_name != NULL);
  assert (name != NULL);

  return strcmp (drv->device_name, name);
}

/*-------------------------------------------------------- _free_screen --*/
/** Free all resources of a #sch_screen.
 * \param _scr Opaque pointer to a #sch_screen.
 */
static void
_free_screen (gpointer _scr, gpointer user_data)
{
  sch_screen *scr = (sch_screen *)_scr;
  (void) user_data;

  /* Remove it from all the devices that have an instance of this screen. */

  free ((char*) scr->id.name);
  dict_free (&scr->field_dict);
  
  free (_scr);
}

/*----------------------------------------------- _free_screen_instance --*/
/** Free all resources of a #sch_screen_instance.
 * \param _scr Opaque pointer to a #sch_screen_instance.
 */
static void
_free_screen_instance (gpointer _inst, gpointer userdata)
{
  sch_screen_instance *inst = (sch_screen_instance *)_inst;
  (void)userdata;

  if (inst->layout.c_layout != NULL)
    cl_free_layout (inst->layout.c_layout);

  /* \todo Add cleanup of inst->g_layout */
  free (_inst);
}

/*-------------------------------------------------------- _free_driver --*/
/** Free all resources of a #sch_output.
 * \param _scr Opaque pointer to a #sch_output.
 */
static void
_free_driver (gpointer _drv, gpointer userdata)
{
  sch_output *drv = (sch_output*)_drv;
  (void)userdata;

  g_slist_foreach (drv->screens, _free_screen_instance, NULL);
  g_slist_free (drv->screens);
  cc_delete_canvas (drv->c_canvas);
  g_free ((char*) drv->device_name);
  g_free (_drv);
}

/*----------------------------------------------------- _refresh_output --*/
/** 
 */
static void
_refresh_output (sch_output *output)
{
  sch_screen_instance *instance;

  instance = (sch_screen_instance *)output->on_display->data;
  assert (instance != NULL);
  
  debug_2 ("_refresh_output for %s", output->device_name);

  if (drivers_get_type (output->driver_handle) == DRV_CHARACTER)
    {
      cr_render (instance->layout.c_layout, &instance->scr->field_dict, 
                 output->c_canvas);

      drivers_process_canvas (output->driver_handle, output->c_canvas);
      cc_clear_damaged_regions (output->c_canvas);
    }
  else
    { 
      /* \todo Render graphical canvas */
    }
}

/*------------------------------------------- _assign_screen_to_outputs --*/
/** Assign a screen to one or more outputs.  This function will create a
 * number of screen instances, and add these to the screens list of one
 * or more #sch_output structs.  The distribution is round-robin by default,
 * but alerts are shown on every screen.  This can be overridden by a
 * Scheme script.
 * \param screen One of more instances of this #sch_screen are created.
 */
static void
_assign_screen_to_outputs (sch_screen *screen, xml_node *layout)
{
  static GSList *last_assigned = NULL;
  sch_output *output;
  sch_screen_instance *new_instance;
  
  debug ("_assign_screen_to_outputs");
  if (outputlist == NULL)
    {
      debug ("   output list is empty, do nothing");
      return;
    }

  /* First, check if a Scheme function is available. */
  /* \todo */

  /* If not, pick one ourselves. */
  if (last_assigned != NULL)
    last_assigned = g_slist_next (last_assigned);

  if (last_assigned == NULL)
    last_assigned = outputlist;

  output = (sch_output *)last_assigned->data;
  assert (output != NULL);
      
  new_instance = g_new (sch_screen_instance, 1);
  new_instance->scr = screen;
  new_instance->times_shown = 0;

  if (drivers_get_type (output->driver_handle) == DRV_CHARACTER)
    {
      drv_dimensions size = drivers_get_dimensions (output->driver_handle);
      
      new_instance->layout.c_layout = 
          cl_make_layout (layout, size.width, size.height);

      /* new_instance->g_layout = NULL; */
    }
  else
    {
      /* \todo : Implement for graphical devices. */
      new_instance->layout.c_layout = NULL;
    }

  if (output->screens == NULL)
    {
      /* No screens were assigned yet. */
      output->screens = g_slist_append (output->screens, new_instance);
      output->on_display = output->screens;
      _refresh_output (output);
    }
  else
    {
      output->screens = g_slist_append (output->screens, new_instance);
    }
}

/*------------------------------------------------------- _count_fields --*/
/** Increase a counter if a field tag was found in a data definition.
 * \param _data Pointer to a counter.
 * \param node The XML node that will be tested.
 */
static void
_count_fields (void *_data, xml_node *node)
{
  size_t *data = (size_t*)_data;
  
  if (node->tag == FIELD)
    ++*data;
}

/*-------------------------------------------------------- _fill_fields --*/
/** Copy a field's type and name to a dictionary.
 * \param _data Opaque pointer to a pointer to a dictionary entry.
 * \param node XML definition of the data field.
 */
static void
_fill_fields (void *_data, xml_node *node)
{
  dict_pair **data = (dict_pair**)_data;
  
  if (node->tag == FIELD)
    {
      const char *value = xmlt_get_attrib (node, TYPE);
      const char *key   = xmlt_get_attrib (node, NAME);
      void       *lookup; 
      field      *new_field;

      if (value == NULL)
        return;
      
      new_field = g_new0 (field, 1);
      
      /* Unknown data types are silently ignored; hopefully we can remain
       * reasonably forwards and backwards compatible. */
      lookup = dict_lookup (&field_type_dict, value);
      if (lookup == NULL)
        new_field->type = F_NONE;
      else
        new_field->type = *(field_type*)lookup;
      
      (*data)->key = key ? g_strdup (key) : 0;
      (*data)->value = new_field;
      ++*data;
    }
}

static void _show (void *userdata, xml_node *node)
{
  debug_2 ("  node %i", node->tag);
}

/*------------------------------------------------- _value_from_subnode --*/
static int
_value_from_subnode (xml_node *node, int subnode_tag)
{
  const char *text;
  int temp;
  xml_node *subnode = xmlt_find (node, NULL, subnode_tag);
  
  if (subnode == NULL)
    {
      debug_2 ("  cannot get value from subnode: no subnode %i found",
               subnode_tag);
      xmlt_for_each (node, 0, _show);
      return 0;
    }
  
  text = xmlt_get_first_cdata (subnode);
  if (text == NULL)
    {
      debug ("  cannot get value from subnode: subnode empty");
      return 0;
    }
  
  temp = strtol (text, 0, 10);
  if (errno == ERANGE || errno == EINVAL)
    {
      debug ("  cannot get value from subnode: subnode content out of range");
      return 0;
    }

  return temp;
}

/*--------------------------------------------------------- _parse_data --*/
static void
_parse_data (void *_data, xml_node *node)
{
  dictionary *dict = (dictionary*)_data;

  if (node->tag == VALUE)
    {
      const char *id = xmlt_get_attrib (node, NAME);
      field *lookup;
      
      if (id == NULL)
        return;
      
      lookup= (field *)dict_lookup (dict, id);
      
      if (lookup == NULL)
        return;
          
      lookup->defined = 1;
      lookup->changed = 0;
      lookup->valid = 1;
      
      switch (lookup->type)
        {
        case F_NONE:
          debug_2 ("  %s -> illegal", id);
          lookup->valid = 0;
          return;

        case F_TEXT:
        case F_LABEL:
        case F_HEADER:
          {
            const char *text = xmlt_get_first_cdata (node);

            if (text == NULL)
              {
                free (lookup->data.text);
                lookup->data.text = NULL;
                lookup->valid = 0;
              }
            else
              {
                debug_3 ("  %s -> %s", id, text);
                if (   lookup->data.text == NULL 
                    || strcmp (lookup->data.text, text))
                  {
                    debug ("  (change)");
                    lookup->changed = 1;
                    free (lookup->data.text);
                    lookup->data.text = g_strdup (text);
                  }
              }
          }
          break;

        case F_TIME:
          lookup->data.time.seconds = 
              3600 * _value_from_subnode (node, HRS)
            + 60   * _value_from_subnode (node, MIN)
            +        _value_from_subnode (node, SEC);
          
          lookup->data.time.milliseconds = _value_from_subnode (node, MSEC);
          lookup->data.time.ms_defined = 
                    (xmlt_find (node, NULL, MSEC) == NULL) ? 0 : 1;
          break;
          
        case F_TIMESPAN:
          lookup->data.timespan.seconds = _value_from_subnode (node, SEC);
          lookup->data.timespan.milliseconds = _value_from_subnode (node, MSEC);
          lookup->data.timespan.ms_defined = 
                    (xmlt_find (node, NULL, MSEC) == NULL) ? 0 : 1;

          debug_4 ("  %s -> %i.%i", id, lookup->data.timespan.seconds,
                 lookup->data.timespan.milliseconds);
          break;
          
        case F_DATE:
          lookup->data.date.day = _value_from_subnode (node, DAY);
          lookup->data.date.month = _value_from_subnode (node, MON);
          lookup->data.date.year = _value_from_subnode (node, YEAR);
          debug_4 ("  %s -> %i-%i", id, lookup->data.date.day,
                 lookup->data.date.month);
          break;
          
        case F_SCALAR:
          {
            char *endptr;
            const char *text = xmlt_get_first_cdata (node);
            long value;
              
            lookup->valid = 0;
              
            if (text != NULL && *text != '\0')
              {
                debug_3 ("  %s -> %s", id, text);
                value = strtol (xmlt_get_first_cdata (node), &endptr, 10);

                if (endptr != NULL && *endptr == '\0')
                  {
                    if (lookup->data.scalar != value)
                      {
                        debug ("  (changed)");
                        lookup->changed = 1;
                        lookup->data.scalar = value;
                      }
                  }
              }
          }
        break;
        
        case F_PERCENTAGE:
          {
            char *endptr;
            const char *text = xmlt_get_first_cdata (node);
            double value;
            
            lookup->valid = 0;
            
            if (text != NULL && *text != '\0')
              {
                value = strtod (xmlt_get_first_cdata (node), &endptr);

                if (endptr != NULL && *endptr == '\0')
                  {
                    if (lookup->data.percentage != value)
                      {
                        lookup->changed = 1;
                        lookup->data.percentage = value;
                      }
                  }
              }
          }

          break;

        case F_HISTOGRAM:
          /* FIXME: implement */
          break;

        case F_PIXMAP:
          /* FIXME: implement */
          break;

        default:
          assert (0);
            
        }
    }    
}

/*-------------------------------------------------------- _reply_error --*/
static void
_reply_error (void *destination, error_code code)
{
  char   temp[256];
  size_t len;

/*
 * FIXME: snprintf only available in ISO-C99? Uses trusted input anyway,
 *        worth the trouble?
 
  len = snprintf (temp, 256, "ERROR %i : %s\n", 
                  (int)code, error_description[code]);
*/
  
  len = sprintf (temp, "ERROR %i : %s\n", 
                 (int)code, error_description[code]);

  if (len > 0 && len <= 256)
    conn_reply (destination, temp, len);
}

/*-------------------------------------------------------- _next_screen --*/
static void
_next_screen (sch_output *output)
{
  sch_screen_instance *instance;

  debug ("_next_screen");

  output->on_display = g_slist_next (output->on_display);
  if (output->on_display == NULL)
    output->on_display = output->screens;

  instance = (sch_screen_instance *)output->on_display->data;
  assert (instance != NULL);

  ++instance->times_shown;
  cc_clear_canvas (output->c_canvas);
  _refresh_output (output);
}

/*--------------------------------------------------- _check_field_dict --*/
static int
_check_field_dict (dictionary *dict)
{
  int       i;
  dict_pair *current_pair;
  size_t    dictsize;

  if (dict->size == 0)
    return ERR_NONE;

  /* Extra check before sorting: all fields must be provided, and
   * should not be empty. */
  current_pair = dict->dict;
  for (i = 0; i < dict->size; ++i, ++current_pair)
    {
      if (   current_pair->key == NULL || *(current_pair->key) == '\0'
          || current_pair->value == NULL )
        {
          return ERR_DATA_INCOMPLETE;
        }
    }

  qsort (dict->dict, dict->size, sizeof (dict_pair),
         dict_pair_compare);

  /* Last check before it's accepted: duplicate names. */
  current_pair = dict->dict;
  dictsize = dict->size - 1;
  
  for (i = 0; i < dictsize; ++i, ++current_pair)
    {
      if (!strcmp (current_pair->key, (current_pair + 1)->key))
        return ERR_DUPLICATE_NAMES;
    }

  return ERR_NONE;
}

/*------------------------------------------------------------ sch_init --*/
void
sch_init (void)
{
  outputlist = NULL;
  screenlist = NULL;
}
          
/*------------------------------------------------------ sch_add_screen --*/
void
sch_add_screen (void *originator, const char *name, sch_importance importance,
                int repetition, xml_node *layout)
{
  screen_id      id;
  GSList         *found;
  dict_pair      *current_pair;
  sch_output     *output;
  drv_dimensions size;
  sch_screen     *scr;
  int            is_new_screen = 0, error_code = 0;
  
  assert (outputlist != NULL);

  output = outputlist->data;
  assert (output != NULL);
  size = drivers_get_dimensions (output->driver_handle);

  debug_2 ("add screen %s", name);

  id.originator = originator;
  id.name = name;

  found = g_slist_find_custom (screenlist, &id, _screen_id_compare);

  if (found == NULL)
    {
      /* This is a new screen. */
      is_new_screen = 1;
      scr = g_new (sch_screen, 1);

      scr->id.originator   = originator;
      scr->id.name         = g_strdup (name);
    }
  else
    {
      /* This screen was already available in the screen list. */
      scr = (sch_screen *)found->data;
      dict_free (&scr->field_dict);
    }

  scr->importance      = importance;
  scr->repetition      = repetition;
  scr->field_dict.size = 0;

  xmlt_rescan_document (layout, &data_tag_dict, &data_attr_dict);
  xmlt_for_each (layout, &scr->field_dict.size, _count_fields);
  scr->field_dict.dict = g_new (dict_pair, scr->field_dict.size);
  
  current_pair = scr->field_dict.dict;
  xmlt_for_each (layout, &current_pair, _fill_fields);
  
  if ((error_code = _check_field_dict (&scr->field_dict)) != ERR_NONE)
    {
      _reply_error (originator, error_code);
      _free_screen (scr, NULL); 
    }
    
  if (is_new_screen)
    {
      screenlist = g_slist_append (screenlist, scr);
      _assign_screen_to_outputs (scr, layout);
    }
}

/*--------------------------------------------------- sch_remove_screen --*/
int
sch_remove_screen (void *originator, const char *name)
{
  GSList     *found;
  screen_id  id;

  debug_2 ("remove screen %s", name);

  id.originator = originator;
  id.name = name;
  
  found = g_slist_find_custom (screenlist, &id, _screen_id_compare);

  if (found == NULL)
    return 0;

  screenlist = g_slist_remove_link (screenlist, found);
  _free_screen (found->data, NULL);
  g_slist_free_1 (found);

  return 1;
}

/*------------------------------------------------- sch_register_output --*/
int
sch_register_output (void *driver_handle, const char *device_name)
{
  sch_output *new_output;
  GSList *found;

  assert (driver_handle != NULL);
  assert (device_name != NULL);
  assert (*device_name != 0);

  debug_2 ("sch_register_output '%s'", device_name);
  
  found = g_slist_find_custom (outputlist, device_name, _driver_name_compare);
  if (found != NULL)
    {
      debug ("  (already registered)");
      return 0;
    }
  
  new_output = g_new (sch_output, 1);
  new_output->driver_handle = driver_handle;
  new_output->device_name = g_strdup (device_name);
  new_output->screens = NULL;
  new_output->on_display = NULL;
  new_output->ticks = 0;
  
  if (drivers_get_type (driver_handle) == DRV_CHARACTER)
    {
      drv_dimensions dim = drivers_get_dimensions (driver_handle);
      new_output->c_canvas = cc_create_canvas (dim.width, dim.height);
    }
  else
    {
      new_output->c_canvas = NULL;
      /** \todo Initialize graphical canvas */
    }
  
  outputlist = g_slist_append (outputlist, new_output);  
  assert (outputlist != NULL);

  return 1;
}

/*----------------------------------------------------- sch_update_data --*/
void
sch_update_data (void *originator, const char *name, xml_node *data)
{
  sch_screen *this_screen;
  screen_id search_id;
  GSList *this_screen_iter;
  GSList *output_iter;
  sch_output *output;
  sch_screen_instance *instance;

  debug_2 ("sch_update_data for %s", name);
  
  search_id.originator = originator;
  search_id.name = name;

  this_screen_iter = 
              g_slist_find_custom (screenlist, &search_id, _screen_id_compare);

  if (this_screen_iter == NULL)
    {
      debug ("  unknown screen ID");
      _reply_error (originator, ERR_UNKNOWN_SCREEN_ID);
      return;
    }
  
  this_screen = (sch_screen *)this_screen_iter->data;

  if (this_screen == NULL)
    {
      debug ("  unknown screen");
      return;
    }
  
  /* Copy the new values from the XML data into the screen's data fields. */
  xmlt_rescan_document (data, &data_tag_dict, &data_attr_dict);
  xmlt_for_each (data, &(this_screen->field_dict), _parse_data);
 
  /* Loop through all outputs to see if any of them is displaying this
   * screen at the moment.  If so, the canvas needs to be updates to reflect
   * the changes. */
  output_iter = outputlist;
  while (output_iter != NULL)
    {
      output = (sch_output *)output_iter->data;
      output_iter = g_slist_next (output_iter);

      debug_2 ("  checking update for %s", output->device_name); 
      
      instance = (sch_screen_instance *)output->on_display->data;
      /*
      if (instance->scr == this_screen)
        _refresh_output (output);
        */
    }
}

/*--------------------------------------------------------- sch_cleanup --*/
void
sch_cleanup ()
{
  g_slist_foreach (outputlist, _free_driver, NULL);
  g_slist_foreach (screenlist, _free_screen, NULL);
  g_slist_free (outputlist);
  g_slist_free (screenlist);
}

GSList *
sch_registered_devices ()
{
  return outputlist;
}

GSList *
sch_registered_screens ()
{
  return screenlist;
}

/*------------------------------------------------------------ sch_tick --*/
gboolean
sch_tick (gpointer userdata)
{
  static GSList              *iter = NULL;
  static sch_output          *output = NULL;
  static sch_screen_instance *instance = NULL;
  (void)userdata;

  iter = outputlist;    
  while (iter != NULL)
    {
      output = (sch_output *)iter->data;
      iter = g_slist_next (iter);
      assert (output != NULL);

      if (output->screens == NULL || output->screens->data == NULL)
        continue;
      
      instance = (sch_screen_instance *)output->on_display->data;
      assert (instance != NULL);

      ++output->ticks;
      if (output->ticks > 200) /* \todo Handle screen rotation properly */
        {
          output->ticks = 0;
          /* Cycle to the next screen and show it. */
          debug ("Cycle to next screen");
          _next_screen (output);
        }
      else 
        {
          /* Handle every element separately. */
          cl_layout      *layout = instance->layout.c_layout;
          dictionary     *dict = &instance->scr->field_dict;
          cl_layout_elem *elem = layout->layout_data;
          size_t i;

          for (i = 0; i < layout->layout_data_size; ++i, ++elem)
            {
              /** \todo Code smell. Move this somewhere else. */
              /*
              if (elem->ref_field == NULL)
                elem->ref_field = dict_lookup (dict, elem->utf8_data);
              */
              
              cr_render_elem (elem, dict, output->c_canvas);
            }

          drivers_process_canvas (output->driver_handle, output->c_canvas);
          cc_clear_damaged_regions (output->c_canvas);
        }
    }

  return TRUE;
}
