/*
 *  scheduler.h - Routines for managing screens and scheduling them for
 *                being displayed.
 *                This file is part of the FreeLCD package.
 *  
 *  $Id: scheduler.h,v 1.4 2004/06/20 14:18:52 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */


#ifndef _SCHEDULER_H_
#define _SCHEDULER_H_ 1

#include <glib.h>

#include "common/xmlt.h"
#include "renderers/charlayout.h"
#include "renderers/charcanvas.h"

typedef enum
{
  IMP_BACKGROUND, IMP_NORMAL, IMP_NOTIFICATION, IMP_ALERT
}
sch_importance;

/** A screen is identified by this combination of a name, and the network
 * address of the client that submitted it. */
typedef struct
{
  void       *originator;
  const char *name;
}
screen_id;

/** A screen that will be displayed on a device. This is an abstract
 * representation of a screen, the actual layout and rendering is tied
 * to a device. */
typedef struct
{
  screen_id      id;          /**< A unique identifier. */
  sch_importance importance;  /**< The importance of this screen. */
  size_t         repetition;  /**< The number of times to display it, zero
                                   is infinite. */
  dictionary     field_dict;  /**< Contents of the data fields. */
}
sch_screen;

/** An instance of an #sch_screen, as shown on a device.  A screen has
 * one or more of these instances. */
typedef struct
{
  union
    {
      cl_layout  *c_layout;
      /* gl_layout  *g_layout; */
    }
  layout;
  
  sch_screen *scr;
  size_t     times_shown;
}
sch_screen_instance;

/** All information associated with a device.  Depending on the type of
 * device, either c_canvas or g_canvas is used, and one of them is
 * NULL. */
typedef struct
{
  cc_canvas  *c_canvas;      /**< Canvas for character devices. */
  /* gr_canvas  *g_canvas; */     /**< Canvas for graphical devices. */ 
  const char *device_name;   /**< Name of the device. */
  void       *driver_handle; /**< Driver handle as returned by
                                  drivers_create_output() */
  GSList     *screens;       /**< List of struct #sch_screen_instance. */
  GSList     *on_display;    /**< Screen that is being displayed. */
  size_t     ticks;          /**< Number of ticks the current screen has 
                                  been displayed. */
}
sch_output;


/** Initialise the scheduler. */
void
sch_init (void);
        
/** Free all resources. */
void
sch_cleanup (void);
        
/** Register a device and a driver to display screens on.
 * \param driver_handle The driver to use for this output.
 * \param device_name The name of the device to use for this output.
 * \return 1 if this operation was successful, 0 if the driver-device pair
 *         already had been registered. */
int
sch_register_output (void *driver_handle, const char *device_name);

/** Register a screen that can be displayed on a device.
 * \param originator The TCP connection that is associated with this screen.
 * \param name The screen's name.
 * \param importance This parameter will determine how much and how quickly
 *                   the screen will appear on the output.
 * \param repetition How many times the screen will be shown before it is
 *                   removed from the queue.
 * \param layout The screen's layout. */
void
sch_add_screen (void *originator, const char *name, sch_importance importance, 
                int repetition, xml_node *layout);

/** Remove a screen that had been added with sch_add_screen() earlier.
 * \param originator The TCP connection that is associated with this screen.
 * \param name The screen's name.
 * \return 1 if the screen was removed, 0 if it could not be found. */
int
sch_remove_screen (void *originator, const char *name);

/** Update all fields in a screen with new data.
 * \param originator The TCP connection that is associated with this update.
 * \param name The screen's name.
 * \param data The new values for the data fields. */
void
sch_update_data (void *originator, const char *name, xml_node *data);

GSList *
sch_registered_devices ();

GSList *
sch_registered_screens ();

/** Take care of all timing issues, e.g. scrolling.
 * This function must be called at a regular interval. In the current
 * implementation it's invoked 20 times per second.
 * \param userdata Dummy parameter
 * \return Dummy boolean, always true. */
gboolean
sch_tick (gpointer userdata);

#endif
