#!/usr/bin/python
#
# runtests.py : Run automatic unit tests on C files.
#
# $Id: runtests.py,v 1.4 2004/06/20 14:45:54 unicorn Exp $
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Copyright (C) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
#


# Short runtests.py HOWTO
#
#
# This script searches the directory '..' recursively for C files,
# and checks if the files found are suitable for being tested.  If
# you want to unit-test a source file, insert an
# #ifdef UNIT_TEST_<filename> somewhere.  <filename> is the uppercase
# filename, with every dot replaced by an underscore.  So, the file
# xmlt.c will have a #ifdef UNIT_TEST_XMLT_C directive.  Between the
# #ifdef and the #endif, there should be a main() routine that
# brutally tests all functions.  If all tests pass, it should exit
# silently with return code 0.  If something fails, it should print a
# descriptive message on stdout and return something non-zero.
#
# If you would like to pass extra flags to the compiler, you can
# define them somewhere in a comment with the keyword UNIT_CFLAGS.
# Anything following this keyword will be passed to gcc.
#
# If the module depends on other C files, these can be specified in
# the same way using the keyword UNIT_EXTRA.
#
# Check out ../common/slist.c if you would like to see a working
# example.


import sys
import os
import string
import glob
import re

GCC_CALL  = 'gcc -g -I. -I..'
TEST_EXEC = 'test-me'

ran_tests = 0
failed_builds = 0
failed_tests = 0
core_dumps = 0
files = []

def CoreDumped(filename):
    global core_dumps
    print "*** CORE DUMPED WHILE TESTING ", filename
    basename = os.path.basename(filename)
    os.rename("core", "core"+basename)
    os.rename(TEST_EXEC, "test-"+basename)
    print "    Core dump is moved to 'core."+basename+"'."
    print "    Test executable is moved to 'test-"+basename+"'."
    core_dumps += 1


def TestFailed(filename):
    global failed_tests
    print "*** TEST FAILED FOR ", filename
    failed_tests += 1


def BuildFailed(filename):
    global failed_builds
    print "*** BUILD FAILED FOR ", filename
    failed_builds += 1


def CompileAndRun(filename, cflags, extra):
    global ran_tests
    
    ran_tests += 1
    print "--- Testing", filename

    base = os.path.basename(filename)
    command = GCC_CALL + " -DUNIT_TEST_" + base.replace(".","_").upper() \
              + " -I" + os.path.dirname(filename) + " " + cflags \
              + " -o test-me " + extra + " " + filename

    if os.system(command) == 0 and os.path.exists(TEST_EXEC):
        result = os.popen("./"+TEST_EXEC).readlines()
        if result:
            for rline in result:
                print "   ", string.rstrip(rline)
            TestFailed(filename)
        
        if os.path.exists("core"):
            CoreDumped(filename)
        else:
            os.remove(TEST_EXEC)
    else:
        BuildFailed(filename)
        

def ProcessSourceFile(filename):
    lines = open(filename, 'r').readlines()
    has_testcase = 0
    cflags = ""
    extra = ""
    
    match_test  = re.compile(r"^#\s*ifdef\s+UNIT_TEST")
    match_flag  = re.compile(r"/\*\s*UNIT_CFLAGS\s+([^\t ].*)\s?\*/")
    match_extra = re.compile(r"/\*\s*UNIT_EXTRA\s+([^\t ].*)\s?\*/")
    
    for line in lines:
        flag_result = match_flag.match(line)
        if flag_result:
            cflags += flag_result.group(1)
            
        extra_result = match_extra.match(line)
        if extra_result:
            extra += extra_result.group(1)
            
        test_result = match_test.match(line)
        if test_result:
            has_testcase = 1
        
    if has_testcase:
        CompileAndRun(filename, cflags, extra)
        

def PrintSummary():
    global ran_tests
    global failed_tests
    global failed_builds
    global core_dumps
    
    if ran_tests:
        print ""
        print "SUMMARY"
        print "-------"
        print ran_tests, "tests ran"
        if failed_builds:
            print failed_builds, "builds failed"
        if failed_tests:
            print failed_tests, "tests failed"
        if core_dumps:
            print core_dumps, "core dumps"
            
        success = ran_tests - (failed_builds + failed_tests + core_dumps)
        print success * 100 / ran_tests, "% succeeded"

        
def AddFilesToList(dummy, dirname, dirfiles):
    global files

    for file in dirfiles:
        if file[-2:] == '.c':
            files.append(dirname + '/' + file)
        
    
def main(args):
    global files
    
    args = args[1:]
    os.path.walk("..", AddFilesToList, None)
    for filename in files:
        ProcessSourceFile(filename)
    PrintSummary()

if __name__ == '__main__':
    sys.exit (main (sys.argv))
